"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationProduct = exports.Product = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const association_manager_1 = require("./private/association-manager");
const validation_1 = require("./private/validation");
const aws_servicecatalog_1 = require("aws-cdk-lib/aws-servicecatalog");
class ProductBase extends aws_cdk_lib_1.Resource {
    associateTagOptions(tagOptions) {
        association_manager_1.AssociationManager.associateTagOptions(this, this.productId, tagOptions);
    }
}
/**
 * Abstract class for Service Catalog Product.
 */
class Product extends ProductBase {
    /**
     * Creates a Product construct that represents an external product.
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param productArn Product Arn
     */
    static fromProductArn(scope, id, productArn) {
        const arn = aws_cdk_lib_1.Stack.of(scope).splitArn(productArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME);
        const productId = arn.resourceName;
        if (!productId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + productArn);
        }
        return new class extends ProductBase {
            constructor() {
                super(...arguments);
                this.productId = productId;
                this.productArn = productArn;
            }
        }(scope, id);
    }
}
exports.Product = Product;
_a = JSII_RTTI_SYMBOL_1;
Product[_a] = { fqn: "@aws-cdk/aws-servicecatalog-alpha.Product", version: "2.22.0-alpha.0" };
/**
 * A Service Catalog Cloudformation Product.
 */
class CloudFormationProduct extends Product {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicecatalog_alpha_CloudFormationProductProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.validateProductProps(props);
        const product = new aws_servicecatalog_1.CfnCloudFormationProduct(this, 'Resource', {
            acceptLanguage: props.messageLanguage,
            description: props.description,
            distributor: props.distributor,
            name: props.productName,
            owner: props.owner,
            provisioningArtifactParameters: this.renderProvisioningArtifacts(props),
            replaceProvisioningArtifacts: props.replaceProductVersionIds,
            supportDescription: props.supportDescription,
            supportEmail: props.supportEmail,
            supportUrl: props.supportUrl,
        });
        this.productId = product.ref;
        this.productArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: 'catalog',
            resource: 'product',
            resourceName: product.ref,
        });
        if (props.tagOptions !== undefined) {
            this.associateTagOptions(props.tagOptions);
        }
    }
    renderProvisioningArtifacts(props) {
        return props.productVersions.map(productVersion => {
            const template = productVersion.cloudFormationTemplate.bind(this);
            validation_1.InputValidator.validateUrl(this.node.path, 'provisioning template url', template.httpUrl);
            return {
                name: productVersion.productVersionName,
                description: productVersion.description,
                disableTemplateValidation: productVersion.validateTemplate === false ? true : false,
                info: {
                    LoadTemplateFromURL: template.httpUrl,
                },
            };
        });
    }
    ;
    validateProductProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'product product name', 1, 100, props.productName);
        validation_1.InputValidator.validateLength(this.node.path, 'product owner', 1, 8191, props.owner);
        validation_1.InputValidator.validateLength(this.node.path, 'product description', 0, 8191, props.description);
        validation_1.InputValidator.validateLength(this.node.path, 'product distributor', 0, 8191, props.distributor);
        validation_1.InputValidator.validateEmail(this.node.path, 'support email', props.supportEmail);
        validation_1.InputValidator.validateUrl(this.node.path, 'support url', props.supportUrl);
        validation_1.InputValidator.validateLength(this.node.path, 'support description', 0, 8191, props.supportDescription);
        if (props.productVersions.length == 0) {
            throw new Error(`Invalid product versions for resource ${this.node.path}, must contain at least 1 product version`);
        }
        props.productVersions.forEach(productVersion => {
            validation_1.InputValidator.validateLength(this.node.path, 'provisioning artifact name', 0, 100, productVersion.productVersionName);
            validation_1.InputValidator.validateLength(this.node.path, 'provisioning artifact description', 0, 8191, productVersion.description);
        });
    }
}
exports.CloudFormationProduct = CloudFormationProduct;
_b = JSII_RTTI_SYMBOL_1;
CloudFormationProduct[_b] = { fqn: "@aws-cdk/aws-servicecatalog-alpha.CloudFormationProduct", version: "2.22.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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