"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateJwtFields = exports.decomposeJwt = void 0;
const assert_1 = require("./assert");
const error_1 = require("./error");
const safe_json_parse_1 = require("./safe-json-parse");
/**
   * Assert that the argument is a valid JWT header object.
   * Throws an error in case it is not.
   *
   * @param header
   * @returns void
   */
function assertJwtHeader(header) {
    if (!safe_json_parse_1.isJsonObject(header)) {
        throw new error_1.JwtParseError('JWT header is not an object');
    }
    if (header.alg !== undefined && typeof header.alg !== 'string') {
        throw new error_1.JwtParseError('JWT header alg claim is not a string');
    }
    if (header.kid !== undefined && typeof header.kid !== 'string') {
        throw new error_1.JwtParseError('JWT header kid claim is not a string');
    }
}
/**
   * Assert that the argument is a valid JWT payload object.
   * Throws an error in case it is not.
   *
   * @param payload
   * @returns void
   */
function assertJwtPayload(payload) {
    if (!safe_json_parse_1.isJsonObject(payload)) {
        throw new error_1.JwtParseError('JWT payload is not an object');
    }
    if (payload.exp !== undefined && !Number.isFinite(payload.exp)) {
        throw new error_1.JwtParseError('JWT payload exp claim is not a number');
    }
    if (payload.iss !== undefined && typeof payload.iss !== 'string') {
        throw new error_1.JwtParseError('JWT payload iss claim is not a string');
    }
    if (payload.aud !== undefined &&
        typeof payload.aud !== 'string' &&
        (!Array.isArray(payload.aud) ||
            payload.aud.some((aud) => typeof aud !== 'string'))) {
        throw new error_1.JwtParseError('JWT payload aud claim is not a string or array of strings');
    }
    if (payload.nbf !== undefined && !Number.isFinite(payload.nbf)) {
        throw new error_1.JwtParseError('JWT payload nbf claim is not a number');
    }
    if (payload.iat !== undefined && !Number.isFinite(payload.iat)) {
        throw new error_1.JwtParseError('JWT payload iat claim is not a number');
    }
    if (payload.scope !== undefined && typeof payload.scope !== 'string') {
        throw new error_1.JwtParseError('JWT payload scope claim is not a string');
    }
    if (payload.jti !== undefined && typeof payload.jti !== 'string') {
        throw new error_1.JwtParseError('JWT payload jti claim is not a string');
    }
}
/**
   * Sanity check, decompose and JSON parse a JWT string into its constituent parts:
   * - header object
   * - payload object
   * - signature string
   *
   * @param jwt The JWT (as string)
   * @returns the decomposed JWT
   */
function decomposeJwt(jwt) {
    // Sanity checks on JWT
    if (!jwt) {
        throw new error_1.JwtParseError('Empty JWT');
    }
    if (typeof jwt !== 'string') {
        throw new error_1.JwtParseError('JWT is not a string');
    }
    if (!jwt.match(/^[A-Za-z0-9_-]+\.[A-Za-z0-9_-]+\.[A-Za-z0-9_-]+$/)) {
        throw new error_1.JwtParseError('JWT string does not consist of exactly 3 parts (header, payload, signature)');
    }
    const [headerB64, payloadB64, signatureB64] = jwt.split('.');
    // B64 decode header and payload
    const [headerString, payloadString] = [headerB64, payloadB64].map((b64) => Buffer.from(b64, 'base64').toString('utf8'));
    // Parse header
    let header;
    try {
        header = safe_json_parse_1.safeJsonParse(headerString);
    }
    catch (err) {
        throw new error_1.JwtParseError('Invalid JWT. Header is not a valid JSON object', err);
    }
    assertJwtHeader(header);
    // parse payload
    let payload;
    try {
        payload = safe_json_parse_1.safeJsonParse(payloadString);
    }
    catch (err) {
        throw new error_1.JwtParseError('Invalid JWT. Payload is not a valid JSON object', err);
    }
    assertJwtPayload(payload);
    return {
        header,
        headerB64,
        payload,
        payloadB64,
        signatureB64,
    };
}
exports.decomposeJwt = decomposeJwt;
/**
   * Validate JWT payload fields. Throws an error in case there's any validation issue.
   *
   * @param payload The (JSON parsed) JWT payload
   * @param options The options to use during validation
   * @returns void
   */
function validateJwtFields(payload, options) {
    var _a, _b, _c;
    // Check expiry
    if (payload.exp !== undefined) {
        if (payload.exp + ((_a = options.graceSeconds) !== null && _a !== void 0 ? _a : 0) < Date.now() / 1000) {
            throw new error_1.JwtExpiredError(`Token expired at ${new Date(payload.exp * 1000).toISOString()}`, payload.exp);
        }
    }
    // Check not before
    if (payload.nbf !== undefined) {
        if (payload.nbf - ((_b = options.graceSeconds) !== null && _b !== void 0 ? _b : 0) > Date.now() / 1000) {
            throw new error_1.JwtNotBeforeError(`Token can't be used before ${new Date(payload.nbf * 1000).toISOString()}`, payload.nbf);
        }
    }
    // Check JWT issuer
    if (options.issuer !== null) {
        if (options.issuer === undefined) {
            throw new error_1.ParameterValidationError('issuer must be provided or set to null explicitly');
        }
        assert_1.assertStringArrayContainsString('Issuer', payload.iss, options.issuer, error_1.JwtInvalidIssuerError);
    }
    // Check audience
    if (options.audience !== null) {
        if (options.audience === undefined) {
            throw new error_1.ParameterValidationError('audience must be provided or set to null explicitly');
        }
        assert_1.assertStringArraysOverlap('Audience', payload.aud, options.audience, error_1.JwtInvalidAudienceError);
    }
    // Check scope
    if (options.scope != null) {
        assert_1.assertStringArraysOverlap('Scope', (_c = payload.scope) === null || _c === void 0 ? void 0 : _c.split(' '), options.scope, error_1.JwtInvalidScopeError);
    }
}
exports.validateJwtFields = validateJwtFields;
//# sourceMappingURL=data:application/json;base64,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