"""
TaskScheduler

:date: Apr 28, 2021
:author: Aldo Diaz, Marcelo Sureda

Provides the functionality of scheduling programmable activities. It's the
main system _scheduler. Allows to execute repeatable tasks daily, weekly or
monthly.
"""
from apscheduler.executors.pool import ThreadPoolExecutor
from apscheduler.jobstores.redis import RedisJobStore
from apscheduler.schedulers.background import BackgroundScheduler
from vcdextension.libraries.logger import Logger
from vcdextension.libraries.vcloudsecurity import VCloudSecurity


class TaskScheduler:
    """
    TaskScheduler class
    Provides scheduling functionality. It's a Pythonic
    implementation of the singleton pattern.
    """
    _instance = None
    _log = Logger()

    def __new__(cls):
        if cls._instance is None:
            # Initializes the scheduler
            try:
                cls._instance = super(TaskScheduler, cls).__new__(cls)
                cls._instance._scheduler = BackgroundScheduler(jobstores={'default': RedisJobStore()},
                                                               executors={'default': ThreadPoolExecutor(10)},
                                                               job_defaults={'coalesce': True, 'max_instances': 1})
                cls._instance._scheduler.start()
                cls._log.info("Task scheduler successfully started")
            except Exception as e:
                cls._log.error(f"<<{type(e).__qualname__}>> {e}")
                del cls._instance
                return None
        return cls._instance

    def schedule_job(self, func, kwargs, **trigger_args):
        """
        Adds the given job to the task scheduler.

        :param Any func: Function to be scheduled
        :param dict kwargs: Keyword arguments
        :param Any trigger_args: Trigger arguments to define execution times
        :return: str: Job reference
        """
        context_id = ""
        # If present, the context should be saved for future reference
        if 'context_id' in kwargs:
            context_id = kwargs['context_id']
            vcd_security = VCloudSecurity()
            vcd_security.keep_context(context_id, True)

        # Generate Job ID: Concatenate function name to context ID and hash it
        job_id = str(abs(hash(func.__name__ + str(kwargs))))

        self._log.info(f"Scheduling job id '{job_id}' to {trigger_args}")
        self._scheduler.add_job(func, trigger='cron', args=[], kwargs=kwargs, id=job_id, jobstore='default',
                                executor='default', replace_existing=True, **trigger_args)
        return job_id

    def remove_job(self, job_id):
        """
        Removes the given job from the task scheduler.

        :param str job_id: Scheduled Job identifier
        """
        self._scheduler.remove_job(job_id=job_id, jobstore='default')

    def stop_scheduler(self):
        """
        Shuts down the scheduler thread
        """
        try:
            self._scheduler.shutdown()
            self._log.info("Scheduler shut down successfully")
        except Exception as e:
            self._log.error(f"<<{type(e).__qualname__}>> {e}")
