"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const path = require("path");
const s3deploy = require("../lib");
/* eslint-disable max-len */
test('deploy from local directory asset', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        ServiceToken: {
            'Fn::GetAtt': [
                'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536',
                'Arn',
            ],
        },
        SourceBucketNames: [{
                Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3Bucket9CD8B20A',
            }],
        SourceObjectKeys: [{
                'Fn::Join': [
                    '',
                    [
                        {
                            'Fn::Select': [
                                0,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            'Fn::Select': [
                                1,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                ],
            }],
        DestinationBucketName: {
            Ref: 'DestC383B82A',
        },
    });
});
test('deploy from local directory assets', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [
            s3deploy.Source.asset(path.join(__dirname, 'my-website')),
            s3deploy.Source.asset(path.join(__dirname, 'my-website-second')),
        ],
        destinationBucket: bucket,
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        ServiceToken: {
            'Fn::GetAtt': [
                'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536',
                'Arn',
            ],
        },
        SourceBucketNames: [
            {
                Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3Bucket9CD8B20A',
            },
            {
                Ref: 'AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3Bucket99793559',
            },
        ],
        SourceObjectKeys: [
            {
                'Fn::Join': [
                    '',
                    [
                        {
                            'Fn::Select': [
                                0,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            'Fn::Select': [
                                1,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                ],
            },
            {
                'Fn::Join': [
                    '',
                    [
                        {
                            'Fn::Select': [
                                0,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3VersionKeyD9ACE665',
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            'Fn::Select': [
                                1,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3VersionKeyD9ACE665',
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                ],
            },
        ],
        DestinationBucketName: {
            Ref: 'DestC383B82A',
        },
    });
});
test('fails if local asset is a non-zip file', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // THEN
    expect(() => new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website', 'index.html'))],
        destinationBucket: bucket,
    })).toThrow(/Asset path must be either a \.zip file or a directory/);
});
test('deploy from a local .zip file', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
    });
});
test('honors passed asset options', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'), {
                exclude: ['*', '!index.html'],
            })],
        destinationBucket: bucket,
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        ServiceToken: {
            'Fn::GetAtt': [
                'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536',
                'Arn',
            ],
        },
        SourceBucketNames: [{
                Ref: 'AssetParameters86f8bca4f28a0bcafef0a98fe4cea25c0071aca27401e35cfaecd06313373bcaS3BucketB41AE64D',
            }],
        SourceObjectKeys: [{
                'Fn::Join': [
                    '',
                    [
                        {
                            'Fn::Select': [
                                0,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParameters86f8bca4f28a0bcafef0a98fe4cea25c0071aca27401e35cfaecd06313373bcaS3VersionKeyF3CBA38F',
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            'Fn::Select': [
                                1,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: 'AssetParameters86f8bca4f28a0bcafef0a98fe4cea25c0071aca27401e35cfaecd06313373bcaS3VersionKeyF3CBA38F',
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                ],
            }],
        DestinationBucketName: {
            Ref: 'DestC383B82A',
        },
    });
});
test('retainOnDelete can be used to retain files when resource is deleted', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
        retainOnDelete: true,
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        RetainOnDelete: true,
    });
});
test('user metadata is correctly transformed', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
        metadata: {
            A: '1',
            B: '2',
        },
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        UserMetadata: { 'x-amzn-meta-a': '1', 'x-amzn-meta-b': '2' },
    });
});
test('system metadata is correctly transformed', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
        contentType: 'text/html',
        contentLanguage: 'en',
        storageClass: s3deploy.StorageClass.INTELLIGENT_TIERING,
        contentDisposition: 'inline',
        serverSideEncryption: s3deploy.ServerSideEncryption.AWS_KMS,
        serverSideEncryptionAwsKmsKeyId: 'mykey',
        serverSideEncryptionCustomerAlgorithm: 'rot13',
        websiteRedirectLocation: 'example',
        cacheControl: [s3deploy.CacheControl.setPublic(), s3deploy.CacheControl.maxAge(cdk.Duration.hours(1))],
        expires: s3deploy.Expires.after(cdk.Duration.hours(12)),
    });
    // THEN
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        SystemMetadata: {
            'content-type': 'text/html',
            'content-language': 'en',
            'content-disposition': 'inline',
            'storage-class': 'INTELLIGENT_TIERING',
            'sse': 'aws:kms',
            'sse-kms-key-id': 'mykey',
            'cache-control': 'public, max-age=3600',
            'expires': s3deploy.Expires.after(cdk.Duration.hours(12)).value,
            'sse-c-copy-source': 'rot13',
            'website-redirect': 'example',
        },
    });
});
test('expires type has correct values', () => {
    expect(s3deploy.Expires.atDate(new Date('Sun, 26 Jan 2020 00:53:20 GMT')).value).toEqual('Sun, 26 Jan 2020 00:53:20 GMT');
    expect(s3deploy.Expires.atTimestamp(1580000000000).value).toEqual('Sun, 26 Jan 2020 00:53:20 GMT');
    expect(Math.abs(new Date(s3deploy.Expires.after(cdk.Duration.minutes(10)).value).getTime() - (Date.now() + 600000)) < 15000).toBeTruthy();
    expect(s3deploy.Expires.fromString('Tue, 04 Feb 2020 08:45:33 GMT').value).toEqual('Tue, 04 Feb 2020 08:45:33 GMT');
});
test('cache control type has correct values', () => {
    expect(s3deploy.CacheControl.mustRevalidate().value).toEqual('must-revalidate');
    expect(s3deploy.CacheControl.noCache().value).toEqual('no-cache');
    expect(s3deploy.CacheControl.noTransform().value).toEqual('no-transform');
    expect(s3deploy.CacheControl.setPublic().value).toEqual('public');
    expect(s3deploy.CacheControl.setPrivate().value).toEqual('private');
    expect(s3deploy.CacheControl.proxyRevalidate().value).toEqual('proxy-revalidate');
    expect(s3deploy.CacheControl.maxAge(cdk.Duration.minutes(1)).value).toEqual('max-age=60');
    expect(s3deploy.CacheControl.sMaxAge(cdk.Duration.minutes(1)).value).toEqual('s-maxage=60');
    expect(s3deploy.CacheControl.fromString('only-if-cached').value).toEqual('only-if-cached');
});
test('storage class type has correct values', () => {
    expect(s3deploy.StorageClass.STANDARD).toEqual('STANDARD');
    expect(s3deploy.StorageClass.REDUCED_REDUNDANCY).toEqual('REDUCED_REDUNDANCY');
    expect(s3deploy.StorageClass.STANDARD_IA).toEqual('STANDARD_IA');
    expect(s3deploy.StorageClass.ONEZONE_IA).toEqual('ONEZONE_IA');
    expect(s3deploy.StorageClass.INTELLIGENT_TIERING).toEqual('INTELLIGENT_TIERING');
    expect(s3deploy.StorageClass.GLACIER).toEqual('GLACIER');
    expect(s3deploy.StorageClass.DEEP_ARCHIVE).toEqual('DEEP_ARCHIVE');
});
test('server side encryption type has correct values', () => {
    expect(s3deploy.ServerSideEncryption.AES_256).toEqual('AES256');
    expect(s3deploy.ServerSideEncryption.AWS_KMS).toEqual('aws:kms');
});
test('distribution can be used to provide a CloudFront distribution for invalidation', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    const distribution = new cloudfront.CloudFrontWebDistribution(stack, 'Distribution', {
        originConfigs: [
            {
                s3OriginSource: {
                    s3BucketSource: bucket,
                },
                behaviors: [{ isDefaultBehavior: true }],
            },
        ],
    });
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
        distribution,
        distributionPaths: ['/images/*'],
    });
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        DistributionId: {
            Ref: 'DistributionCFDistribution882A7313',
        },
        DistributionPaths: ['/images/*'],
    });
});
test('invalidation can happen without distributionPaths provided', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    const distribution = new cloudfront.CloudFrontWebDistribution(stack, 'Distribution', {
        originConfigs: [
            {
                s3OriginSource: {
                    s3BucketSource: bucket,
                },
                behaviors: [{ isDefaultBehavior: true }],
            },
        ],
    });
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
        destinationBucket: bucket,
        distribution,
    });
    expect(stack).toHaveResource('Custom::CDKBucketDeployment', {
        DistributionId: {
            Ref: 'DistributionCFDistribution882A7313',
        },
    });
});
test('fails if distribution paths provided but not distribution ID', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // THEN
    expect(() => new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website', 'index.html'))],
        destinationBucket: bucket,
        distributionPaths: ['/images/*'],
    })).toThrow(/Distribution must be specified if distribution paths are specified/);
});
test('lambda execution role gets permissions to read from the source bucket and read/write in destination', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const source = new s3.Bucket(stack, 'Source');
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.bucket(source, 'file.zip')],
        destinationBucket: bucket,
    });
    // THEN
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'Source71E471F1',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'Source71E471F1',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                },
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                        's3:DeleteObject*',
                        's3:PutObject*',
                        's3:Abort*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'DestC383B82A',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'DestC383B82A',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                },
            ],
            Version: '2012-10-17',
        },
        PolicyName: 'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756CServiceRoleDefaultPolicy88902FDF',
        Roles: [
            {
                Ref: 'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756CServiceRole89A01265',
            },
        ],
    });
});
test('memoryLimit can be used to specify the memory limit for the deployment resource handler', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    // we define 3 deployments with 2 different memory configurations
    new s3deploy.BucketDeployment(stack, 'Deploy256-1', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
        memoryLimit: 256,
    });
    new s3deploy.BucketDeployment(stack, 'Deploy256-2', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
        memoryLimit: 256,
    });
    new s3deploy.BucketDeployment(stack, 'Deploy1024', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
        memoryLimit: 1024,
    });
    // THEN
    // we expect to find only two handlers, one for each configuration
    expect(stack).toCountResources('AWS::Lambda::Function', 2);
    expect(stack).toHaveResource('AWS::Lambda::Function', { MemorySize: 256 });
    expect(stack).toHaveResource('AWS::Lambda::Function', { MemorySize: 1024 });
});
test('deployment allows custom role to be supplied', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    const existingRole = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('lambda.amazon.com'),
    });
    // WHEN
    new s3deploy.BucketDeployment(stack, 'DeployWithRole', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
        role: existingRole,
    });
    // THEN
    expect(stack).toCountResources('AWS::IAM::Role', 1);
    expect(stack).toCountResources('AWS::Lambda::Function', 1);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Role: {
            'Fn::GetAtt': [
                'Role1ABCC5F0',
                'Arn',
            ],
        },
    });
});
test('deploy without deleting missing files from destination', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
        prune: false,
    });
    expect(stack).toHaveResourceLike('Custom::CDKBucketDeployment', {
        Prune: false,
    });
});
test('Deployment role gets KMS permissions when using assets from new style synthesizer', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        synthesizer: new cdk.DefaultStackSynthesizer(),
    });
    const bucket = new s3.Bucket(stack, 'Dest');
    // WHEN
    new s3deploy.BucketDeployment(stack, 'Deploy', {
        sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
        destinationBucket: bucket,
    });
    // THEN
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Version: '2012-10-17',
            Statement: assert_1.arrayWith({
                Action: ['kms:Decrypt', 'kms:DescribeKey'],
                Effect: 'Allow',
                Resource: { 'Fn::ImportValue': 'CdkBootstrap-hnb659fds-FileAssetKeyArn' },
            }),
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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