"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrowApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const lambda = require("aws-cdk-lib/aws-lambda");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const logs = require("aws-cdk-lib/aws-logs");
/**
 * For copying shared code to all paths
 */
const fse = require("fs-extra");
const DEFAULT_LAMBDA_CODE = `
exports.handler = async function (event, context, callback) {
  try {
    const data = {
      statusCode: 201,
    };
    return data;
  } catch (uncaughtError) {
    console.error(uncaughtError);
    throw uncaughtError;
  }
}
`;
/**
 * @experimental
 */
class CrowApi extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Pulling out props
        const { sourceDirectory = 'src', sharedDirectory = 'shared', useAuthorizerLambda = false, authorizerDirectory = 'authorizer', authorizerLambdaConfiguration = {}, tokenAuthorizerConfiguration = {}, createApiKey = false, logRetention = logs.RetentionDays.ONE_WEEK, apiGatewayConfiguration = {}, apiGatewayName = 'crow-api', lambdaConfigurations = {}, methodConfigurations = {}, } = props;
        // Initializing constants
        const LAMBDA_RUNTIME = lambda.Runtime.NODEJS_14_X;
        const SPECIAL_DIRECTORIES = [
            sharedDirectory,
            authorizerDirectory,
        ];
        // Helpers functions for constructor
        // Prepares default Lambda props and overrides them with user input
        function bundleLambdaProps(codePath, userConfiguration, sharedLayer) {
            let layers;
            if (sharedLayer) {
                const { layers: userLayers = [], } = userConfiguration;
                layers = [sharedLayer, ...userLayers];
            }
            const defaultProps = {
                runtime: LAMBDA_RUNTIME,
                code: lambda.Code.fromAsset(codePath),
                handler: 'index.handler',
                logRetention,
            };
            const lambdaProps = {
                ...defaultProps,
                ...userConfiguration,
                layers,
            };
            return lambdaProps;
        }
        // Returns child directories given the path of a parent
        function getDirectoryChildren(parentDirectory) {
            try {
                const directories = fse.readdirSync(parentDirectory, { withFileTypes: true })
                    .filter((dirent) => dirent.isDirectory())
                    .map((dirent) => dirent.name);
                return directories;
            }
            catch {
                /**
                 * The only time I have run into this was when the src/ directory
                 * was empty.
                 * If it is empty, let CDK tree validation tell user that the
                 * REST API does not have any methods.
                 */
            }
            return [];
        }
        // A default Lambda function is needed for the API Gateway
        const defaultLambda = new lambda.Function(this, 'default-crow-lambda', {
            runtime: lambda.Runtime.NODEJS_12_X,
            code: new lambda.InlineCode(DEFAULT_LAMBDA_CODE),
            handler: 'index.handler',
            logRetention,
        });
        // API Gateway log group
        const gatewayLogGroup = new logs.LogGroup(this, 'api-access-logs', {
            retention: logs.RetentionDays.ONE_WEEK,
        });
        // The API Gateway itself
        const gateway = new apigateway.LambdaRestApi(this, apiGatewayName, {
            handler: defaultLambda,
            proxy: false,
            deploy: true,
            deployOptions: {
                loggingLevel: apigateway.MethodLoggingLevel.ERROR,
                accessLogDestination: new apigateway.LogGroupLogDestination(gatewayLogGroup),
            },
            apiKeySourceType: createApiKey ? apigateway.ApiKeySourceType.HEADER : undefined,
            ...apiGatewayConfiguration,
        });
        // Create API key if desired
        if (createApiKey) {
            const apiKey = gateway.addApiKey('api-key');
            const usagePlan = new apigateway.UsagePlan(this, 'usage-plan', {
                throttle: {
                    burstLimit: 5000,
                    rateLimit: 10000,
                },
                apiStages: [
                    {
                        api: gateway,
                        stage: gateway.deploymentStage,
                    },
                ],
            });
            usagePlan.addApiKey(apiKey);
        }
        // Create Lambda layer out of shared directory if it exists
        const sourceSharedDirectory = `${sourceDirectory}/${sharedDirectory}`;
        let sharedLayer;
        if (fse.existsSync(sourceSharedDirectory)) {
            sharedLayer = new lambda.LayerVersion(this, 'shared-layer', {
                code: lambda.Code.fromAsset(sourceSharedDirectory),
                compatibleRuntimes: [LAMBDA_RUNTIME],
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            this.lambdaLayer = sharedLayer;
        }
        // Create Lambda authorizer to be used in subsequent Methods
        let tokenAuthorizer;
        if (useAuthorizerLambda) {
            const fullAuthorizerDirectory = `${sourceDirectory}/${authorizerDirectory}`;
            const authorizerLambdaProps = bundleLambdaProps(fullAuthorizerDirectory, authorizerLambdaConfiguration, sharedLayer);
            const authorizerLambda = new lambda.Function(this, 'authorizer-lambda', authorizerLambdaProps);
            this.authorizerLambda = authorizerLambda;
            const bundledTokenAuthConfig = {
                handler: authorizerLambda,
                resultsCacheTtl: cdk.Duration.seconds(3600),
                ...tokenAuthorizerConfiguration,
            };
            tokenAuthorizer = new apigateway.TokenAuthorizer(this, 'token-authorizer', bundledTokenAuthConfig);
        }
        // Time to start walking the directories
        const root = sourceDirectory;
        const verbs = ['get', 'post', 'put', 'delete'];
        const graph = {};
        const lambdasByPath = {};
        // Initialize with root
        graph['/'] = {
            resource: gateway.root,
            path: root,
            paths: [],
            verbs: [],
        };
        // First element in tuple is directory path, second is API path
        const nodes = [[root, '/']];
        // BFS that creates API Gateway structure using addMethod
        while (nodes.length) {
            // The `|| ['type', 'script']` piece is needed or TS throws a fit
            const [directoryPath, apiPath] = nodes.shift() || ['type', 'script'];
            const children = getDirectoryChildren(directoryPath);
            // For debugging purposes
            // console.log(`${apiPath}'s children are: ${children}`);
            // Don't have to worry about previously visited nodes
            // since this is a file structure
            // ...unless there are symlinks? Haven't run into that
            children.forEach((child) => {
                const newDirectoryPath = `${directoryPath}/${child}`;
                // If we're on the root path, don't separate with a slash (/)
                //   because it ends up looking like //child-path
                const newApiPath = apiPath === '/' ? `/${child}` : `${apiPath}/${child}`;
                if (verbs.includes(child)) {
                    // If directory is a verb, we don't traverse it anymore
                    //   and need to create an API Gateway method and Lambda
                    const userLambdaConfiguration = lambdaConfigurations[newApiPath]
                        || {};
                    const lambdaProps = bundleLambdaProps(newDirectoryPath, userLambdaConfiguration, sharedLayer);
                    const newLambda = new lambda.Function(this, newDirectoryPath, lambdaProps);
                    // Pull out useAuthorizerLambda value
                    const { useAuthorizerLambda: authorizerLambdaConfigured, ...userMethodConfiguration } = methodConfigurations[newApiPath] || {};
                    let methodConfiguration = userMethodConfiguration;
                    // If this method should be behind an authorizer Lambda
                    //   construct the methodConfiguration object as such
                    if (authorizerLambdaConfigured && useAuthorizerLambda) {
                        methodConfiguration = {
                            ...userMethodConfiguration,
                            authorizationType: apigateway.AuthorizationType.CUSTOM,
                            authorizer: tokenAuthorizer,
                        };
                    }
                    graph[apiPath].resource.addMethod(child.toUpperCase(), new apigateway.LambdaIntegration(newLambda), methodConfiguration);
                    graph[apiPath].verbs.push(child);
                    lambdasByPath[newApiPath] = newLambda;
                }
                else if (SPECIAL_DIRECTORIES.includes(child)) {
                    // The special directories should not result in an API path
                    // This means the API also cannot have a resource with the
                    //   same name
                }
                else {
                    // If directory is not a verb, create new API Gateway resource
                    //   for use by verb directory later
                    const newResource = graph[apiPath].resource
                        .resourceForPath(child);
                    nodes.push([newDirectoryPath, newApiPath]);
                    // Add child to parent's paths
                    graph[apiPath].paths.push(child);
                    // Initialize graph node to include child
                    graph[newApiPath] = {
                        resource: newResource,
                        path: newDirectoryPath,
                        paths: [],
                        verbs: [],
                    };
                }
            });
        }
        // For debugging purposes
        // console.log(graph);
        // Expose API Gateway
        this.gateway = gateway;
        this.lambdaFunctions = lambdasByPath;
    }
}
exports.CrowApi = CrowApi;
_a = JSII_RTTI_SYMBOL_1;
CrowApi[_a] = { fqn: "crow-api.CrowApi", version: "2.1.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLG1DQUFtQztBQUNuQywyQ0FBdUM7QUFDdkMsaURBQWlEO0FBQ2pELHlEQUF5RDtBQUN6RCw2Q0FBNkM7QUFFN0M7O0dBRUc7QUFDSCxnQ0FBZ0M7QUFFaEMsTUFBTSxtQkFBbUIsR0FBRzs7Ozs7Ozs7Ozs7O0NBWTNCLENBQUM7Ozs7QUF3REYsTUFBYSxPQUFRLFNBQVEsc0JBQVM7Ozs7SUFPcEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFvQjtRQUM1RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLG9CQUFvQjtRQUNwQixNQUFNLEVBQ0osZUFBZSxHQUFHLEtBQUssRUFDdkIsZUFBZSxHQUFHLFFBQVEsRUFDMUIsbUJBQW1CLEdBQUcsS0FBSyxFQUMzQixtQkFBbUIsR0FBRyxZQUFZLEVBQ2xDLDZCQUE2QixHQUFHLEVBQUUsRUFDbEMsNEJBQTRCLEdBQUcsRUFBRSxFQUNqQyxZQUFZLEdBQUcsS0FBSyxFQUNwQixZQUFZLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxRQUFRLEVBQzFDLHVCQUF1QixHQUFHLEVBQUUsRUFDNUIsY0FBYyxHQUFHLFVBQVUsRUFDM0Isb0JBQW9CLEdBQUcsRUFBRSxFQUN6QixvQkFBb0IsR0FBRyxFQUFFLEdBQzFCLEdBQUcsS0FBSyxDQUFDO1FBRVYseUJBQXlCO1FBQ3pCLE1BQU0sY0FBYyxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDO1FBQ2xELE1BQU0sbUJBQW1CLEdBQUc7WUFDMUIsZUFBZTtZQUNmLG1CQUFtQjtTQUNwQixDQUFDO1FBRUYsb0NBQW9DO1FBRXBDLG1FQUFtRTtRQUNuRSxTQUFTLGlCQUFpQixDQUN4QixRQUFnQixFQUNoQixpQkFBdUMsRUFDdkMsV0FBNEM7WUFFNUMsSUFBSSxNQUFNLENBQUM7WUFDWCxJQUFJLFdBQVcsRUFBRTtnQkFDZixNQUFNLEVBQ0osTUFBTSxFQUFFLFVBQVUsR0FBRyxFQUFFLEdBQ3hCLEdBQUcsaUJBQWlCLENBQUM7Z0JBQ3RCLE1BQU0sR0FBRyxDQUFDLFdBQVcsRUFBRSxHQUFHLFVBQVUsQ0FBQyxDQUFDO2FBQ3ZDO1lBRUQsTUFBTSxZQUFZLEdBQUc7Z0JBQ25CLE9BQU8sRUFBRSxjQUFjO2dCQUN2QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDO2dCQUNyQyxPQUFPLEVBQUUsZUFBZTtnQkFDeEIsWUFBWTthQUNiLENBQUM7WUFFRixNQUFNLFdBQVcsR0FBRztnQkFDbEIsR0FBRyxZQUFZO2dCQUNmLEdBQUcsaUJBQWlCO2dCQUNwQixNQUFNO2FBQ1AsQ0FBQTtZQUVELE9BQU8sV0FBVyxDQUFDO1FBQ3JCLENBQUM7UUFFRCx1REFBdUQ7UUFDdkQsU0FBUyxvQkFBb0IsQ0FBQyxlQUF1QjtZQUNuRCxJQUFJO2dCQUNGLE1BQU0sV0FBVyxHQUFHLEdBQUcsQ0FBQyxXQUFXLENBQUMsZUFBZSxFQUFFLEVBQUUsYUFBYSxFQUFFLElBQUksRUFBRSxDQUFDO3FCQUMxRSxNQUFNLENBQUMsQ0FBQyxNQUFXLEVBQUUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUUsQ0FBQztxQkFDN0MsR0FBRyxDQUFDLENBQUMsTUFBVyxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ3JDLE9BQU8sV0FBVyxDQUFDO2FBQ3BCO1lBQUMsTUFBTTtnQkFDTjs7Ozs7bUJBS0c7YUFDSjtZQUNELE9BQU8sRUFBRSxDQUFDO1FBQ1osQ0FBQztRQUVELDBEQUEwRDtRQUMxRCxNQUFNLGFBQWEsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO1lBQ3JFLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDbkMsSUFBSSxFQUFFLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQztZQUNoRCxPQUFPLEVBQUUsZUFBZTtZQUN4QixZQUFZO1NBQ2IsQ0FBQyxDQUFDO1FBRUgsd0JBQXdCO1FBQ3hCLE1BQU0sZUFBZSxHQUFHLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDakUsU0FBUyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUTtTQUN2QyxDQUFDLENBQUM7UUFFSCx5QkFBeUI7UUFDekIsTUFBTSxPQUFPLEdBQUcsSUFBSSxVQUFVLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDakUsT0FBTyxFQUFFLGFBQWE7WUFDdEIsS0FBSyxFQUFFLEtBQUs7WUFDWixNQUFNLEVBQUUsSUFBSTtZQUNaLGFBQWEsRUFBRTtnQkFDYixZQUFZLEVBQUUsVUFBVSxDQUFDLGtCQUFrQixDQUFDLEtBQUs7Z0JBQ2pELG9CQUFvQixFQUFFLElBQUksVUFBVSxDQUFDLHNCQUFzQixDQUFDLGVBQWUsQ0FBQzthQUM3RTtZQUNELGdCQUFnQixFQUFFLFlBQVksQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMvRSxHQUFHLHVCQUF1QjtTQUMzQixDQUFDLENBQUM7UUFFSCw0QkFBNEI7UUFDNUIsSUFBSSxZQUFZLEVBQUU7WUFDaEIsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUM1QyxNQUFNLFNBQVMsR0FBRyxJQUFJLFVBQVUsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtnQkFDN0QsUUFBUSxFQUFFO29CQUNSLFVBQVUsRUFBRSxJQUFJO29CQUNoQixTQUFTLEVBQUUsS0FBSztpQkFDakI7Z0JBQ0QsU0FBUyxFQUFFO29CQUNUO3dCQUNFLEdBQUcsRUFBRSxPQUFPO3dCQUNaLEtBQUssRUFBRSxPQUFPLENBQUMsZUFBZTtxQkFDL0I7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFDSCxTQUFTLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQzdCO1FBRUQsMkRBQTJEO1FBQzNELE1BQU0scUJBQXFCLEdBQUcsR0FBRyxlQUFlLElBQUksZUFBZSxFQUFFLENBQUM7UUFDdEUsSUFBSSxXQUE0QyxDQUFDO1FBQ2pELElBQUksR0FBRyxDQUFDLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFO1lBQ3pDLFdBQVcsR0FBRyxJQUFJLE1BQU0sQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtnQkFDMUQsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLHFCQUFxQixDQUFDO2dCQUNsRCxrQkFBa0IsRUFBRSxDQUFDLGNBQWMsQ0FBQztnQkFDcEMsYUFBYSxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsT0FBTzthQUN6QyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsV0FBVyxHQUFHLFdBQVcsQ0FBQztTQUNoQztRQUVELDREQUE0RDtRQUM1RCxJQUFJLGVBQXVDLENBQUM7UUFDNUMsSUFBSSxtQkFBbUIsRUFBRTtZQUN2QixNQUFNLHVCQUF1QixHQUFHLEdBQUcsZUFBZSxJQUFJLG1CQUFtQixFQUFFLENBQUM7WUFFNUUsTUFBTSxxQkFBcUIsR0FBRyxpQkFBaUIsQ0FBQyx1QkFBdUIsRUFBRSw2QkFBNkIsRUFBRSxXQUFXLENBQUMsQ0FBQztZQUVySCxNQUFNLGdCQUFnQixHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLEVBQUUscUJBQXFCLENBQUMsQ0FBQztZQUMvRixJQUFJLENBQUMsZ0JBQWdCLEdBQUcsZ0JBQWdCLENBQUM7WUFFekMsTUFBTSxzQkFBc0IsR0FBRztnQkFDN0IsT0FBTyxFQUFFLGdCQUFnQjtnQkFDekIsZUFBZSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQztnQkFDM0MsR0FBRyw0QkFBNEI7YUFDaEMsQ0FBQztZQUNGLGVBQWUsR0FBRyxJQUFJLFVBQVUsQ0FBQyxlQUFlLENBQzlDLElBQUksRUFDSixrQkFBa0IsRUFDbEIsc0JBQXNCLENBQ3ZCLENBQUM7U0FDSDtRQUVELHdDQUF3QztRQUN4QyxNQUFNLElBQUksR0FBRyxlQUFlLENBQUM7UUFDN0IsTUFBTSxLQUFLLEdBQUcsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUMvQyxNQUFNLEtBQUssR0FBWSxFQUFFLENBQUM7UUFDMUIsTUFBTSxhQUFhLEdBQWtCLEVBQUUsQ0FBQztRQUV4Qyx1QkFBdUI7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxHQUFHO1lBQ1gsUUFBUSxFQUFFLE9BQU8sQ0FBQyxJQUFJO1lBQ3RCLElBQUksRUFBRSxJQUFJO1lBQ1YsS0FBSyxFQUFFLEVBQUU7WUFDVCxLQUFLLEVBQUUsRUFBRTtTQUNWLENBQUM7UUFDRiwrREFBK0Q7UUFDL0QsTUFBTSxLQUFLLEdBQXVCLENBQUMsQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUVoRCx5REFBeUQ7UUFDekQsT0FBTyxLQUFLLENBQUMsTUFBTSxFQUFFO1lBQ25CLGlFQUFpRTtZQUNqRSxNQUFNLENBQUMsYUFBYSxFQUFFLE9BQU8sQ0FBQyxHQUFHLEtBQUssQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztZQUNyRSxNQUFNLFFBQVEsR0FBVSxvQkFBb0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUU1RCx5QkFBeUI7WUFDekIseURBQXlEO1lBRXpELHFEQUFxRDtZQUNyRCxpQ0FBaUM7WUFDakMsc0RBQXNEO1lBQ3RELFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRTtnQkFFekIsTUFBTSxnQkFBZ0IsR0FBRyxHQUFHLGFBQWEsSUFBSSxLQUFLLEVBQUUsQ0FBQztnQkFDckQsNkRBQTZEO2dCQUM3RCxpREFBaUQ7Z0JBQ2pELE1BQU0sVUFBVSxHQUFHLE9BQU8sS0FBSyxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDLEdBQUcsT0FBTyxJQUFJLEtBQUssRUFBRSxDQUFDO2dCQUV6RSxJQUFJLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLEVBQUU7b0JBQ3pCLHVEQUF1RDtvQkFDdkQsd0RBQXdEO29CQUN4RCxNQUFNLHVCQUF1QixHQUFHLG9CQUFvQixDQUFDLFVBQVUsQ0FBQzsyQkFDM0QsRUFBRSxDQUFDO29CQUNSLE1BQU0sV0FBVyxHQUFHLGlCQUFpQixDQUNuQyxnQkFBZ0IsRUFDaEIsdUJBQXVCLEVBQ3ZCLFdBQVcsQ0FDWixDQUFDO29CQUNGLE1BQU0sU0FBUyxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FDbkMsSUFBSSxFQUNKLGdCQUFnQixFQUNoQixXQUFXLENBQ1osQ0FBQztvQkFFRixxQ0FBcUM7b0JBQ3JDLE1BQU0sRUFDSixtQkFBbUIsRUFBRSwwQkFBMEIsRUFDL0MsR0FBRyx1QkFBdUIsRUFDM0IsR0FBRyxvQkFBb0IsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLENBQUM7b0JBQzNDLElBQUksbUJBQW1CLEdBQUcsdUJBQXVCLENBQUM7b0JBQ2xELHVEQUF1RDtvQkFDdkQscURBQXFEO29CQUNyRCxJQUFJLDBCQUEwQixJQUFJLG1CQUFtQixFQUFFO3dCQUNyRCxtQkFBbUIsR0FBRzs0QkFDcEIsR0FBRyx1QkFBdUI7NEJBQzFCLGlCQUFpQixFQUFFLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNOzRCQUN0RCxVQUFVLEVBQUUsZUFBZTt5QkFDNUIsQ0FBQTtxQkFDRjtvQkFFRCxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FDL0IsS0FBSyxDQUFDLFdBQVcsRUFBRSxFQUNuQixJQUFJLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLENBQUMsRUFDM0MsbUJBQW1CLENBQ3BCLENBQUM7b0JBQ0YsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7b0JBQ2pDLGFBQWEsQ0FBQyxVQUFVLENBQUMsR0FBRyxTQUFTLENBQUM7aUJBRXZDO3FCQUFNLElBQUksbUJBQW1CLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxFQUFFO29CQUM5QywyREFBMkQ7b0JBQzNELDBEQUEwRDtvQkFDMUQsY0FBYztpQkFDZjtxQkFBTTtvQkFDTCw4REFBOEQ7b0JBQzlELG9DQUFvQztvQkFFcEMsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFFBQVE7eUJBQ3hDLGVBQWUsQ0FBQyxLQUFLLENBQUMsQ0FBQztvQkFFMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUM7b0JBRTNDLDhCQUE4QjtvQkFDOUIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7b0JBRWpDLHlDQUF5QztvQkFDekMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxHQUFHO3dCQUNsQixRQUFRLEVBQUUsV0FBVzt3QkFDckIsSUFBSSxFQUFFLGdCQUFnQjt3QkFDdEIsS0FBSyxFQUFFLEVBQUU7d0JBQ1QsS0FBSyxFQUFFLEVBQUU7cUJBQ1YsQ0FBQztpQkFFSDtZQUNILENBQUMsQ0FBQyxDQUFDO1NBQ0o7UUFFRCx5QkFBeUI7UUFDekIsc0JBQXNCO1FBRXRCLHFCQUFxQjtRQUNyQixJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztRQUN2QixJQUFJLENBQUMsZUFBZSxHQUFHLGFBQWEsQ0FBQztJQUN2QyxDQUFDOztBQS9RSCwwQkFnUkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBhcGlnYXRld2F5IGZyb20gJ2F3cy1jZGstbGliL2F3cy1hcGlnYXRld2F5JztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuXG4vKipcbiAqIEZvciBjb3B5aW5nIHNoYXJlZCBjb2RlIHRvIGFsbCBwYXRoc1xuICovXG5pbXBvcnQgKiBhcyBmc2UgZnJvbSAnZnMtZXh0cmEnO1xuXG5jb25zdCBERUZBVUxUX0xBTUJEQV9DT0RFID0gYFxuZXhwb3J0cy5oYW5kbGVyID0gYXN5bmMgZnVuY3Rpb24gKGV2ZW50LCBjb250ZXh0LCBjYWxsYmFjaykge1xuICB0cnkge1xuICAgIGNvbnN0IGRhdGEgPSB7XG4gICAgICBzdGF0dXNDb2RlOiAyMDEsXG4gICAgfTtcbiAgICByZXR1cm4gZGF0YTtcbiAgfSBjYXRjaCAodW5jYXVnaHRFcnJvcikge1xuICAgIGNvbnNvbGUuZXJyb3IodW5jYXVnaHRFcnJvcik7XG4gICAgdGhyb3cgdW5jYXVnaHRFcnJvcjtcbiAgfVxufVxuYDtcblxuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFzQnlQYXRoIHtcbiAgW3BhdGg6IHN0cmluZ106IGxhbWJkYS5GdW5jdGlvbixcbn1cblxuZXhwb3J0IGludGVyZmFjZSBDcm93TGFtYmRhQ29uZmlndXJhdGlvbnMge1xuICBbbGFtYmRhQnlQYXRoOiBzdHJpbmddOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyxcbn1cblxuZXhwb3J0IGludGVyZmFjZSBDcm93TWV0aG9kQ29uZmlndXJhdGlvbiBleHRlbmRzIGFwaWdhdGV3YXkuTWV0aG9kT3B0aW9ucyB7XG4gIHJlYWRvbmx5IHVzZUF1dGhvcml6ZXJMYW1iZGE6IGJvb2xlYW4sXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQ3Jvd01ldGhvZENvbmZpZ3VyYXRpb25zIHtcbiAgLy8gbWV0aG9kQnlQYXRoIHNob3VsZCBiZSBsYW1iZGEuRnVuY3Rpb25Qcm9wc1xuICAvLyB3aXRob3V0IGFueXRoaW5nIHJlcXVpcmVkXG4gIC8vIGJ1dCBqc2lpIGRvZXMgbm90IGFsbG93IGZvciBPbWl0IHR5cGVcbiAgW21ldGhvZEJ5UGF0aDogc3RyaW5nXTogQ3Jvd01ldGhvZENvbmZpZ3VyYXRpb24sXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSUNyb3dBcGlQcm9wcyB7XG4gIHNvdXJjZURpcmVjdG9yeT86IHN0cmluZyxcbiAgc2hhcmVkRGlyZWN0b3J5Pzogc3RyaW5nLFxuICB1c2VBdXRob3JpemVyTGFtYmRhPzogYm9vbGVhbixcbiAgYXV0aG9yaXplckRpcmVjdG9yeT86IHN0cmluZyxcbiAgLy8gYXV0aG9yaXplckxhbWJkYUNvbmZpZ3VyYXRpb24gc2hvdWxkIGJlIGxhbWJkYS5GdW5jdGlvblByb3BzXG4gIC8vIHdpdGhvdXQgYW55dGhpbmcgcmVxdWlyZWRcbiAgLy8gYnV0IGpzaWkgZG9lcyBub3QgYWxsb3cgZm9yIE9taXQgdHlwZVxuICBhdXRob3JpemVyTGFtYmRhQ29uZmlndXJhdGlvbj86IGxhbWJkYS5GdW5jdGlvblByb3BzIHwgYW55LFxuICAvLyBhdXRob3JpemVyQ29uZmlndXJhdGlvbiBzaG91bGQgYmUgYXBpZ2F0ZXdheS5Ub2tlbkF1dGhvcml6ZXJQcm9wc1xuICAvLyB3aXRob3V0IGFueXRoaW5nIHJlcXVpcmVkXG4gIC8vIGJ1dCBqc2lpIGRvZXMgbm90IGFsbG93IGZvciBPbWl0IHR5cGVcbiAgdG9rZW5BdXRob3JpemVyQ29uZmlndXJhdGlvbj86IGFwaWdhdGV3YXkuVG9rZW5BdXRob3JpemVyUHJvcHMgfCBhbnksXG4gIGNyZWF0ZUFwaUtleT86IGJvb2xlYW4sXG4gIGxvZ1JldGVudGlvbj86IGxvZ3MuUmV0ZW50aW9uRGF5cyxcbiAgLy8gYXBpR2F0d2F5Q29uZmlndXJhdGlvbiBzaG91bGQgYmUgYXBpZ2F0ZXdheS5MYW1iZGFSZXN0QXBpUHJvcHNcbiAgLy8gd2l0aG91dCBhbnl0aGluZyByZXF1aXJlZFxuICAvLyBidXQganNpaSBkb2VzIG5vdCBhbGxvdyBmb3IgT21pdCB0eXBlXG4gIGFwaUdhdGV3YXlDb25maWd1cmF0aW9uPzogYXBpZ2F0ZXdheS5MYW1iZGFSZXN0QXBpUHJvcHMgfCBhbnksXG4gIGFwaUdhdGV3YXlOYW1lPzogc3RyaW5nLFxuICBsYW1iZGFDb25maWd1cmF0aW9ucz86IENyb3dMYW1iZGFDb25maWd1cmF0aW9ucyxcbiAgbWV0aG9kQ29uZmlndXJhdGlvbnM/OiBDcm93TWV0aG9kQ29uZmlndXJhdGlvbnMsXG59XG5cbmludGVyZmFjZSBGU0dyYXBoTm9kZSB7XG4gIHJlc291cmNlOiBhcGlnYXRld2F5LklSZXNvdXJjZSxcbiAgcGF0aDogc3RyaW5nLFxuICBwYXRoczogc3RyaW5nW10sXG4gIHZlcmJzOiBzdHJpbmdbXSxcbn1cblxuaW50ZXJmYWNlIEZTR3JhcGgge1xuICBbcGF0aDogc3RyaW5nXTogRlNHcmFwaE5vZGUsXG59XG5cbmV4cG9ydCBjbGFzcyBDcm93QXBpIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgcHVibGljIGF1dGhvcml6ZXJMYW1iZGEhOiBsYW1iZGEuRnVuY3Rpb247XG4gIHB1YmxpYyBnYXRld2F5ITogYXBpZ2F0ZXdheS5SZXN0QXBpO1xuICBwdWJsaWMgbGFtYmRhTGF5ZXIhOiBsYW1iZGEuTGF5ZXJWZXJzaW9uIHwgdW5kZWZpbmVkO1xuICBwdWJsaWMgbGFtYmRhRnVuY3Rpb25zITogTGFtYmRhc0J5UGF0aDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IElDcm93QXBpUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgLy8gUHVsbGluZyBvdXQgcHJvcHNcbiAgICBjb25zdCB7XG4gICAgICBzb3VyY2VEaXJlY3RvcnkgPSAnc3JjJyxcbiAgICAgIHNoYXJlZERpcmVjdG9yeSA9ICdzaGFyZWQnLFxuICAgICAgdXNlQXV0aG9yaXplckxhbWJkYSA9IGZhbHNlLFxuICAgICAgYXV0aG9yaXplckRpcmVjdG9yeSA9ICdhdXRob3JpemVyJyxcbiAgICAgIGF1dGhvcml6ZXJMYW1iZGFDb25maWd1cmF0aW9uID0ge30sXG4gICAgICB0b2tlbkF1dGhvcml6ZXJDb25maWd1cmF0aW9uID0ge30sXG4gICAgICBjcmVhdGVBcGlLZXkgPSBmYWxzZSxcbiAgICAgIGxvZ1JldGVudGlvbiA9IGxvZ3MuUmV0ZW50aW9uRGF5cy5PTkVfV0VFSyxcbiAgICAgIGFwaUdhdGV3YXlDb25maWd1cmF0aW9uID0ge30sXG4gICAgICBhcGlHYXRld2F5TmFtZSA9ICdjcm93LWFwaScsXG4gICAgICBsYW1iZGFDb25maWd1cmF0aW9ucyA9IHt9LFxuICAgICAgbWV0aG9kQ29uZmlndXJhdGlvbnMgPSB7fSxcbiAgICB9ID0gcHJvcHM7XG5cbiAgICAvLyBJbml0aWFsaXppbmcgY29uc3RhbnRzXG4gICAgY29uc3QgTEFNQkRBX1JVTlRJTUUgPSBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTRfWDtcbiAgICBjb25zdCBTUEVDSUFMX0RJUkVDVE9SSUVTID0gW1xuICAgICAgc2hhcmVkRGlyZWN0b3J5LFxuICAgICAgYXV0aG9yaXplckRpcmVjdG9yeSxcbiAgICBdO1xuXG4gICAgLy8gSGVscGVycyBmdW5jdGlvbnMgZm9yIGNvbnN0cnVjdG9yXG5cbiAgICAvLyBQcmVwYXJlcyBkZWZhdWx0IExhbWJkYSBwcm9wcyBhbmQgb3ZlcnJpZGVzIHRoZW0gd2l0aCB1c2VyIGlucHV0XG4gICAgZnVuY3Rpb24gYnVuZGxlTGFtYmRhUHJvcHMoXG4gICAgICBjb2RlUGF0aDogc3RyaW5nLFxuICAgICAgdXNlckNvbmZpZ3VyYXRpb246IGxhbWJkYS5GdW5jdGlvblByb3BzLFxuICAgICAgc2hhcmVkTGF5ZXI6IGxhbWJkYS5MYXllclZlcnNpb24gfCB1bmRlZmluZWQsXG4gICAgKSB7XG4gICAgICBsZXQgbGF5ZXJzO1xuICAgICAgaWYgKHNoYXJlZExheWVyKSB7XG4gICAgICAgIGNvbnN0IHtcbiAgICAgICAgICBsYXllcnM6IHVzZXJMYXllcnMgPSBbXSxcbiAgICAgICAgfSA9IHVzZXJDb25maWd1cmF0aW9uO1xuICAgICAgICBsYXllcnMgPSBbc2hhcmVkTGF5ZXIsIC4uLnVzZXJMYXllcnNdO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBkZWZhdWx0UHJvcHMgPSB7XG4gICAgICAgIHJ1bnRpbWU6IExBTUJEQV9SVU5USU1FLFxuICAgICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoY29kZVBhdGgpLFxuICAgICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICAgIGxvZ1JldGVudGlvbixcbiAgICAgIH07XG5cbiAgICAgIGNvbnN0IGxhbWJkYVByb3BzID0ge1xuICAgICAgICAuLi5kZWZhdWx0UHJvcHMsXG4gICAgICAgIC4uLnVzZXJDb25maWd1cmF0aW9uLCAvLyBMZXQgdXNlciBjb25maWd1cmF0aW9uIG92ZXJyaWRlIGFueXRoaW5nIGV4Y2VwdCBsYXllcnNcbiAgICAgICAgbGF5ZXJzLFxuICAgICAgfVxuXG4gICAgICByZXR1cm4gbGFtYmRhUHJvcHM7XG4gICAgfVxuXG4gICAgLy8gUmV0dXJucyBjaGlsZCBkaXJlY3RvcmllcyBnaXZlbiB0aGUgcGF0aCBvZiBhIHBhcmVudFxuICAgIGZ1bmN0aW9uIGdldERpcmVjdG9yeUNoaWxkcmVuKHBhcmVudERpcmVjdG9yeTogc3RyaW5nKSB7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCBkaXJlY3RvcmllcyA9IGZzZS5yZWFkZGlyU3luYyhwYXJlbnREaXJlY3RvcnksIHsgd2l0aEZpbGVUeXBlczogdHJ1ZSB9KVxuICAgICAgICAgIC5maWx0ZXIoKGRpcmVudDogYW55KSA9PiBkaXJlbnQuaXNEaXJlY3RvcnkoKSlcbiAgICAgICAgICAubWFwKChkaXJlbnQ6IGFueSkgPT4gZGlyZW50Lm5hbWUpO1xuICAgICAgICByZXR1cm4gZGlyZWN0b3JpZXM7XG4gICAgICB9IGNhdGNoIHtcbiAgICAgICAgLyoqXG4gICAgICAgICAqIFRoZSBvbmx5IHRpbWUgSSBoYXZlIHJ1biBpbnRvIHRoaXMgd2FzIHdoZW4gdGhlIHNyYy8gZGlyZWN0b3J5XG4gICAgICAgICAqIHdhcyBlbXB0eS5cbiAgICAgICAgICogSWYgaXQgaXMgZW1wdHksIGxldCBDREsgdHJlZSB2YWxpZGF0aW9uIHRlbGwgdXNlciB0aGF0IHRoZVxuICAgICAgICAgKiBSRVNUIEFQSSBkb2VzIG5vdCBoYXZlIGFueSBtZXRob2RzLlxuICAgICAgICAgKi9cbiAgICAgIH1cbiAgICAgIHJldHVybiBbXTtcbiAgICB9XG5cbiAgICAvLyBBIGRlZmF1bHQgTGFtYmRhIGZ1bmN0aW9uIGlzIG5lZWRlZCBmb3IgdGhlIEFQSSBHYXRld2F5XG4gICAgY29uc3QgZGVmYXVsdExhbWJkYSA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcywgJ2RlZmF1bHQtY3Jvdy1sYW1iZGEnLCB7XG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTJfWCxcbiAgICAgIGNvZGU6IG5ldyBsYW1iZGEuSW5saW5lQ29kZShERUZBVUxUX0xBTUJEQV9DT0RFKSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIGxvZ1JldGVudGlvbixcbiAgICB9KTtcblxuICAgIC8vIEFQSSBHYXRld2F5IGxvZyBncm91cFxuICAgIGNvbnN0IGdhdGV3YXlMb2dHcm91cCA9IG5ldyBsb2dzLkxvZ0dyb3VwKHRoaXMsICdhcGktYWNjZXNzLWxvZ3MnLCB7XG4gICAgICByZXRlbnRpb246IGxvZ3MuUmV0ZW50aW9uRGF5cy5PTkVfV0VFSyxcbiAgICB9KTtcblxuICAgIC8vIFRoZSBBUEkgR2F0ZXdheSBpdHNlbGZcbiAgICBjb25zdCBnYXRld2F5ID0gbmV3IGFwaWdhdGV3YXkuTGFtYmRhUmVzdEFwaSh0aGlzLCBhcGlHYXRld2F5TmFtZSwge1xuICAgICAgaGFuZGxlcjogZGVmYXVsdExhbWJkYSxcbiAgICAgIHByb3h5OiBmYWxzZSxcbiAgICAgIGRlcGxveTogdHJ1ZSxcbiAgICAgIGRlcGxveU9wdGlvbnM6IHtcbiAgICAgICAgbG9nZ2luZ0xldmVsOiBhcGlnYXRld2F5Lk1ldGhvZExvZ2dpbmdMZXZlbC5FUlJPUixcbiAgICAgICAgYWNjZXNzTG9nRGVzdGluYXRpb246IG5ldyBhcGlnYXRld2F5LkxvZ0dyb3VwTG9nRGVzdGluYXRpb24oZ2F0ZXdheUxvZ0dyb3VwKSxcbiAgICAgIH0sXG4gICAgICBhcGlLZXlTb3VyY2VUeXBlOiBjcmVhdGVBcGlLZXkgPyBhcGlnYXRld2F5LkFwaUtleVNvdXJjZVR5cGUuSEVBREVSIDogdW5kZWZpbmVkLFxuICAgICAgLi4uYXBpR2F0ZXdheUNvbmZpZ3VyYXRpb24sXG4gICAgfSk7XG5cbiAgICAvLyBDcmVhdGUgQVBJIGtleSBpZiBkZXNpcmVkXG4gICAgaWYgKGNyZWF0ZUFwaUtleSkge1xuICAgICAgY29uc3QgYXBpS2V5ID0gZ2F0ZXdheS5hZGRBcGlLZXkoJ2FwaS1rZXknKTtcbiAgICAgIGNvbnN0IHVzYWdlUGxhbiA9IG5ldyBhcGlnYXRld2F5LlVzYWdlUGxhbih0aGlzLCAndXNhZ2UtcGxhbicsIHtcbiAgICAgICAgdGhyb3R0bGU6IHtcbiAgICAgICAgICBidXJzdExpbWl0OiA1MDAwLFxuICAgICAgICAgIHJhdGVMaW1pdDogMTAwMDAsXG4gICAgICAgIH0sXG4gICAgICAgIGFwaVN0YWdlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGFwaTogZ2F0ZXdheSxcbiAgICAgICAgICAgIHN0YWdlOiBnYXRld2F5LmRlcGxveW1lbnRTdGFnZSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSk7XG4gICAgICB1c2FnZVBsYW4uYWRkQXBpS2V5KGFwaUtleSk7XG4gICAgfVxuXG4gICAgLy8gQ3JlYXRlIExhbWJkYSBsYXllciBvdXQgb2Ygc2hhcmVkIGRpcmVjdG9yeSBpZiBpdCBleGlzdHNcbiAgICBjb25zdCBzb3VyY2VTaGFyZWREaXJlY3RvcnkgPSBgJHtzb3VyY2VEaXJlY3Rvcnl9LyR7c2hhcmVkRGlyZWN0b3J5fWA7XG4gICAgbGV0IHNoYXJlZExheWVyOiBsYW1iZGEuTGF5ZXJWZXJzaW9uIHwgdW5kZWZpbmVkO1xuICAgIGlmIChmc2UuZXhpc3RzU3luYyhzb3VyY2VTaGFyZWREaXJlY3RvcnkpKSB7XG4gICAgICBzaGFyZWRMYXllciA9IG5ldyBsYW1iZGEuTGF5ZXJWZXJzaW9uKHRoaXMsICdzaGFyZWQtbGF5ZXInLCB7XG4gICAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChzb3VyY2VTaGFyZWREaXJlY3RvcnkpLFxuICAgICAgICBjb21wYXRpYmxlUnVudGltZXM6IFtMQU1CREFfUlVOVElNRV0sXG4gICAgICAgIHJlbW92YWxQb2xpY3k6IGNkay5SZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICB9KTtcblxuICAgICAgdGhpcy5sYW1iZGFMYXllciA9IHNoYXJlZExheWVyO1xuICAgIH1cblxuICAgIC8vIENyZWF0ZSBMYW1iZGEgYXV0aG9yaXplciB0byBiZSB1c2VkIGluIHN1YnNlcXVlbnQgTWV0aG9kc1xuICAgIGxldCB0b2tlbkF1dGhvcml6ZXI6IGFwaWdhdGV3YXkuSUF1dGhvcml6ZXI7XG4gICAgaWYgKHVzZUF1dGhvcml6ZXJMYW1iZGEpIHtcbiAgICAgIGNvbnN0IGZ1bGxBdXRob3JpemVyRGlyZWN0b3J5ID0gYCR7c291cmNlRGlyZWN0b3J5fS8ke2F1dGhvcml6ZXJEaXJlY3Rvcnl9YDtcblxuICAgICAgY29uc3QgYXV0aG9yaXplckxhbWJkYVByb3BzID0gYnVuZGxlTGFtYmRhUHJvcHMoZnVsbEF1dGhvcml6ZXJEaXJlY3RvcnksIGF1dGhvcml6ZXJMYW1iZGFDb25maWd1cmF0aW9uLCBzaGFyZWRMYXllcik7XG5cbiAgICAgIGNvbnN0IGF1dGhvcml6ZXJMYW1iZGEgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdhdXRob3JpemVyLWxhbWJkYScsIGF1dGhvcml6ZXJMYW1iZGFQcm9wcyk7XG4gICAgICB0aGlzLmF1dGhvcml6ZXJMYW1iZGEgPSBhdXRob3JpemVyTGFtYmRhO1xuXG4gICAgICBjb25zdCBidW5kbGVkVG9rZW5BdXRoQ29uZmlnID0ge1xuICAgICAgICBoYW5kbGVyOiBhdXRob3JpemVyTGFtYmRhLFxuICAgICAgICByZXN1bHRzQ2FjaGVUdGw6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDM2MDApLFxuICAgICAgICAuLi50b2tlbkF1dGhvcml6ZXJDb25maWd1cmF0aW9uLFxuICAgICAgfTtcbiAgICAgIHRva2VuQXV0aG9yaXplciA9IG5ldyBhcGlnYXRld2F5LlRva2VuQXV0aG9yaXplcihcbiAgICAgICAgdGhpcyxcbiAgICAgICAgJ3Rva2VuLWF1dGhvcml6ZXInLFxuICAgICAgICBidW5kbGVkVG9rZW5BdXRoQ29uZmlnXG4gICAgICApO1xuICAgIH1cblxuICAgIC8vIFRpbWUgdG8gc3RhcnQgd2Fsa2luZyB0aGUgZGlyZWN0b3JpZXNcbiAgICBjb25zdCByb290ID0gc291cmNlRGlyZWN0b3J5O1xuICAgIGNvbnN0IHZlcmJzID0gWydnZXQnLCAncG9zdCcsICdwdXQnLCAnZGVsZXRlJ107XG4gICAgY29uc3QgZ3JhcGg6IEZTR3JhcGggPSB7fTtcbiAgICBjb25zdCBsYW1iZGFzQnlQYXRoOiBMYW1iZGFzQnlQYXRoID0ge307XG5cbiAgICAvLyBJbml0aWFsaXplIHdpdGggcm9vdFxuICAgIGdyYXBoWycvJ10gPSB7XG4gICAgICByZXNvdXJjZTogZ2F0ZXdheS5yb290LFxuICAgICAgcGF0aDogcm9vdCxcbiAgICAgIHBhdGhzOiBbXSxcbiAgICAgIHZlcmJzOiBbXSxcbiAgICB9O1xuICAgIC8vIEZpcnN0IGVsZW1lbnQgaW4gdHVwbGUgaXMgZGlyZWN0b3J5IHBhdGgsIHNlY29uZCBpcyBBUEkgcGF0aFxuICAgIGNvbnN0IG5vZGVzOiBbc3RyaW5nLCBzdHJpbmddW10gPSBbW3Jvb3QsICcvJ11dO1xuXG4gICAgLy8gQkZTIHRoYXQgY3JlYXRlcyBBUEkgR2F0ZXdheSBzdHJ1Y3R1cmUgdXNpbmcgYWRkTWV0aG9kXG4gICAgd2hpbGUgKG5vZGVzLmxlbmd0aCkge1xuICAgICAgLy8gVGhlIGB8fCBbJ3R5cGUnLCAnc2NyaXB0J11gIHBpZWNlIGlzIG5lZWRlZCBvciBUUyB0aHJvd3MgYSBmaXRcbiAgICAgIGNvbnN0IFtkaXJlY3RvcnlQYXRoLCBhcGlQYXRoXSA9IG5vZGVzLnNoaWZ0KCkgfHwgWyd0eXBlJywgJ3NjcmlwdCddO1xuICAgICAgY29uc3QgY2hpbGRyZW46IGFueVtdID0gZ2V0RGlyZWN0b3J5Q2hpbGRyZW4oZGlyZWN0b3J5UGF0aCk7XG5cbiAgICAgIC8vIEZvciBkZWJ1Z2dpbmcgcHVycG9zZXNcbiAgICAgIC8vIGNvbnNvbGUubG9nKGAke2FwaVBhdGh9J3MgY2hpbGRyZW4gYXJlOiAke2NoaWxkcmVufWApO1xuXG4gICAgICAvLyBEb24ndCBoYXZlIHRvIHdvcnJ5IGFib3V0IHByZXZpb3VzbHkgdmlzaXRlZCBub2Rlc1xuICAgICAgLy8gc2luY2UgdGhpcyBpcyBhIGZpbGUgc3RydWN0dXJlXG4gICAgICAvLyAuLi51bmxlc3MgdGhlcmUgYXJlIHN5bWxpbmtzPyBIYXZlbid0IHJ1biBpbnRvIHRoYXRcbiAgICAgIGNoaWxkcmVuLmZvckVhY2goKGNoaWxkKSA9PiB7XG5cbiAgICAgICAgY29uc3QgbmV3RGlyZWN0b3J5UGF0aCA9IGAke2RpcmVjdG9yeVBhdGh9LyR7Y2hpbGR9YDtcbiAgICAgICAgLy8gSWYgd2UncmUgb24gdGhlIHJvb3QgcGF0aCwgZG9uJ3Qgc2VwYXJhdGUgd2l0aCBhIHNsYXNoICgvKVxuICAgICAgICAvLyAgIGJlY2F1c2UgaXQgZW5kcyB1cCBsb29raW5nIGxpa2UgLy9jaGlsZC1wYXRoXG4gICAgICAgIGNvbnN0IG5ld0FwaVBhdGggPSBhcGlQYXRoID09PSAnLycgPyBgLyR7Y2hpbGR9YCA6IGAke2FwaVBhdGh9LyR7Y2hpbGR9YDtcblxuICAgICAgICBpZiAodmVyYnMuaW5jbHVkZXMoY2hpbGQpKSB7XG4gICAgICAgICAgLy8gSWYgZGlyZWN0b3J5IGlzIGEgdmVyYiwgd2UgZG9uJ3QgdHJhdmVyc2UgaXQgYW55bW9yZVxuICAgICAgICAgIC8vICAgYW5kIG5lZWQgdG8gY3JlYXRlIGFuIEFQSSBHYXRld2F5IG1ldGhvZCBhbmQgTGFtYmRhXG4gICAgICAgICAgY29uc3QgdXNlckxhbWJkYUNvbmZpZ3VyYXRpb24gPSBsYW1iZGFDb25maWd1cmF0aW9uc1tuZXdBcGlQYXRoXVxuICAgICAgICAgICAgfHwge307XG4gICAgICAgICAgY29uc3QgbGFtYmRhUHJvcHMgPSBidW5kbGVMYW1iZGFQcm9wcyhcbiAgICAgICAgICAgIG5ld0RpcmVjdG9yeVBhdGgsXG4gICAgICAgICAgICB1c2VyTGFtYmRhQ29uZmlndXJhdGlvbixcbiAgICAgICAgICAgIHNoYXJlZExheWVyLFxuICAgICAgICAgICk7XG4gICAgICAgICAgY29uc3QgbmV3TGFtYmRhID0gbmV3IGxhbWJkYS5GdW5jdGlvbihcbiAgICAgICAgICAgIHRoaXMsXG4gICAgICAgICAgICBuZXdEaXJlY3RvcnlQYXRoLFxuICAgICAgICAgICAgbGFtYmRhUHJvcHMsXG4gICAgICAgICAgKTtcblxuICAgICAgICAgIC8vIFB1bGwgb3V0IHVzZUF1dGhvcml6ZXJMYW1iZGEgdmFsdWVcbiAgICAgICAgICBjb25zdCB7XG4gICAgICAgICAgICB1c2VBdXRob3JpemVyTGFtYmRhOiBhdXRob3JpemVyTGFtYmRhQ29uZmlndXJlZCxcbiAgICAgICAgICAgIC4uLnVzZXJNZXRob2RDb25maWd1cmF0aW9uXG4gICAgICAgICAgfSA9IG1ldGhvZENvbmZpZ3VyYXRpb25zW25ld0FwaVBhdGhdIHx8IHt9O1xuICAgICAgICAgIGxldCBtZXRob2RDb25maWd1cmF0aW9uID0gdXNlck1ldGhvZENvbmZpZ3VyYXRpb247XG4gICAgICAgICAgLy8gSWYgdGhpcyBtZXRob2Qgc2hvdWxkIGJlIGJlaGluZCBhbiBhdXRob3JpemVyIExhbWJkYVxuICAgICAgICAgIC8vICAgY29uc3RydWN0IHRoZSBtZXRob2RDb25maWd1cmF0aW9uIG9iamVjdCBhcyBzdWNoXG4gICAgICAgICAgaWYgKGF1dGhvcml6ZXJMYW1iZGFDb25maWd1cmVkICYmIHVzZUF1dGhvcml6ZXJMYW1iZGEpIHtcbiAgICAgICAgICAgIG1ldGhvZENvbmZpZ3VyYXRpb24gPSB7XG4gICAgICAgICAgICAgIC4uLnVzZXJNZXRob2RDb25maWd1cmF0aW9uLFxuICAgICAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBpZ2F0ZXdheS5BdXRob3JpemF0aW9uVHlwZS5DVVNUT00sXG4gICAgICAgICAgICAgIGF1dGhvcml6ZXI6IHRva2VuQXV0aG9yaXplcixcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG5cbiAgICAgICAgICBncmFwaFthcGlQYXRoXS5yZXNvdXJjZS5hZGRNZXRob2QoXG4gICAgICAgICAgICBjaGlsZC50b1VwcGVyQ2FzZSgpLFxuICAgICAgICAgICAgbmV3IGFwaWdhdGV3YXkuTGFtYmRhSW50ZWdyYXRpb24obmV3TGFtYmRhKSxcbiAgICAgICAgICAgIG1ldGhvZENvbmZpZ3VyYXRpb24sXG4gICAgICAgICAgKTtcbiAgICAgICAgICBncmFwaFthcGlQYXRoXS52ZXJicy5wdXNoKGNoaWxkKTtcbiAgICAgICAgICBsYW1iZGFzQnlQYXRoW25ld0FwaVBhdGhdID0gbmV3TGFtYmRhO1xuXG4gICAgICAgIH0gZWxzZSBpZiAoU1BFQ0lBTF9ESVJFQ1RPUklFUy5pbmNsdWRlcyhjaGlsZCkpIHtcbiAgICAgICAgICAvLyBUaGUgc3BlY2lhbCBkaXJlY3RvcmllcyBzaG91bGQgbm90IHJlc3VsdCBpbiBhbiBBUEkgcGF0aFxuICAgICAgICAgIC8vIFRoaXMgbWVhbnMgdGhlIEFQSSBhbHNvIGNhbm5vdCBoYXZlIGEgcmVzb3VyY2Ugd2l0aCB0aGVcbiAgICAgICAgICAvLyAgIHNhbWUgbmFtZVxuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIC8vIElmIGRpcmVjdG9yeSBpcyBub3QgYSB2ZXJiLCBjcmVhdGUgbmV3IEFQSSBHYXRld2F5IHJlc291cmNlXG4gICAgICAgICAgLy8gICBmb3IgdXNlIGJ5IHZlcmIgZGlyZWN0b3J5IGxhdGVyXG5cbiAgICAgICAgICBjb25zdCBuZXdSZXNvdXJjZSA9IGdyYXBoW2FwaVBhdGhdLnJlc291cmNlXG4gICAgICAgICAgICAucmVzb3VyY2VGb3JQYXRoKGNoaWxkKTtcblxuICAgICAgICAgIG5vZGVzLnB1c2goW25ld0RpcmVjdG9yeVBhdGgsIG5ld0FwaVBhdGhdKTtcblxuICAgICAgICAgIC8vIEFkZCBjaGlsZCB0byBwYXJlbnQncyBwYXRoc1xuICAgICAgICAgIGdyYXBoW2FwaVBhdGhdLnBhdGhzLnB1c2goY2hpbGQpO1xuXG4gICAgICAgICAgLy8gSW5pdGlhbGl6ZSBncmFwaCBub2RlIHRvIGluY2x1ZGUgY2hpbGRcbiAgICAgICAgICBncmFwaFtuZXdBcGlQYXRoXSA9IHtcbiAgICAgICAgICAgIHJlc291cmNlOiBuZXdSZXNvdXJjZSxcbiAgICAgICAgICAgIHBhdGg6IG5ld0RpcmVjdG9yeVBhdGgsXG4gICAgICAgICAgICBwYXRoczogW10sXG4gICAgICAgICAgICB2ZXJiczogW10sXG4gICAgICAgICAgfTtcblxuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvLyBGb3IgZGVidWdnaW5nIHB1cnBvc2VzXG4gICAgLy8gY29uc29sZS5sb2coZ3JhcGgpO1xuXG4gICAgLy8gRXhwb3NlIEFQSSBHYXRld2F5XG4gICAgdGhpcy5nYXRld2F5ID0gZ2F0ZXdheTtcbiAgICB0aGlzLmxhbWJkYUZ1bmN0aW9ucyA9IGxhbWJkYXNCeVBhdGg7XG4gIH1cbn1cbiJdfQ==