#
# Copyright (c) 2019-2025
# Pertti Palo, Scott Moisik, Matthew Faytak, and Motoki Saito.
#
# This file is part of the Phonetic Analysis ToolKIT
# (see https://github.com/giuthas/patkit/).
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#
# The example data packaged with this program is licensed under the
# Creative Commons Attribution-NonCommercial-ShareAlike 4.0
# International (CC BY-NC-SA 4.0) License. You should have received a
# copy of the Creative Commons Attribution-NonCommercial-ShareAlike 4.0
# International (CC BY-NC-SA 4.0) License along with the data. If not,
# see <https://creativecommons.org/licenses/by-nc-sa/4.0/> for details.
#
# When using the toolkit for scientific publications, please cite the
# articles listed in README.md. They can also be found in
# citations.bib in BibTeX format.
#


import logging
import math
import multiprocessing as mp
import os.path
import pickle
import warnings
# built in packages
from contextlib import closing

# scientific plotting
import matplotlib.pyplot as plt
# numpy and scipy
import numpy as np
import scipy.io as sio
import scipy.io.wavfile as sio_wavfile
# diffeomorphic demons algorithm implemented in python in the DIPY package
from dipy.align.imwarp import SymmetricDiffeomorphicRegistration
from dipy.align.metrics import CCMetric, EMMetric, SSDMetric
from matplotlib import animation
from matplotlib.backends.backend_pdf import PdfPages
from scipy import interpolate
from scipy.signal import butter, filtfilt, sosfilt
from scipy.signal.windows import kaiser

# local modules
from ..data_import import parse_ultrasound_meta_aaa

# create module logger
ofreg_logger = logging.getLogger('patkit.ofreg')


def parallel_register(ns, index, num_frames, storage):
    ofreg_logger.info("Working on frame pair %d of %d." %
                      (index, num_frames - 1))
    current_im = ns.ultra_interp[index]
    next_im = ns.ultra_interp[index + 1]

    # suppress warnings (from dipy package encouraging us to install "Fury")
    warnings.filterwarnings("ignore")

    # execute the optimization
    storage[index] = {'of': ns.sdr.optimize(next_im, current_im),
                      'current frame': index,
                      'next frame': index + 1}

    # revert back to always displaying warnings
    warnings.filterwarnings("always")


def of(item):
    # inputs: elements in data dictionary generated by get_data_from_dir
    # i.e. all_data[i] is an item

    # TODO don't arrange this externally to these scripts;
    # i.e. transfer the loop in driver.py into here

    ofreg_logger.info("Processing item " +
                      item['base_name'] + " " + item['prompt'] + '.')

    (ult_wav_fs, ult_wav_frames) = sio_wavfile.read(item['ult_wav_file'])

    meta = parse_ultrasound_meta_aaa(item['ult_meta_file'])
    ult_fps = meta['FramesPerSec']
    ult_NumVectors = meta['NumVectors']
    ult_PixPerVector = meta['PixPerVector']
    ult_TimeInSecOfFirstFrame = meta['TimeInSecsOfFirstFrame']
    ult_PixelsPerMm = meta['PixelsPerMm']

    with closing(open(item['ult_file'], 'rb')) as ult_file:
        ult_data = ult_file.read()
        ultra = np.fromstring(ult_data, dtype=np.uint8)
        ultra = ultra.astype("float32")

        ult_no_frames = int(len(ultra) / (ult_NumVectors * ult_PixPerVector))

        # reshape into vectors containing a frame each
        ultra = ultra.reshape(
            (ult_no_frames, ult_NumVectors, ult_PixPerVector))

        # interpolate the data to correct the axis scaling for purposes of image registration
        probe_array_length_mm = 40.0  # TODO 40 mm long linear probe assumed!!!
        probe_depth_mm = ult_PixPerVector/ult_PixelsPerMm
        length_depth_ratio = probe_depth_mm/probe_array_length_mm

        x = np.linspace(1, ult_NumVectors, ult_NumVectors)
        y = np.linspace(1, ult_PixPerVector, ult_PixPerVector)

        xnew = np.linspace(1, ult_NumVectors, ult_NumVectors)
        ynew = np.linspace(1, ult_PixPerVector, math.ceil(
            ult_NumVectors * length_depth_ratio))
        f = interpolate.interp2d(
            x, y, np.transpose(ultra[1, :, :]), kind='linear')

        ultra_interp = []

        # debug plotting
        debug_plot_interpolation = False
        if debug_plot_interpolation:
            fig, ax = plt.subplots(1, 1)
            im = ax.imshow(f(xnew, ynew))

        for fIdx in range(0, ult_no_frames):
            f = interpolate.interp2d(x, y, np.transpose(
                ultra[fIdx, :, :]), kind='linear')
            ultra_interp.append(f(xnew, ynew))

            # debug plotting
            if debug_plot_interpolation:
                im.set_data(ultra_interp[fIdx])
                ax.set_title(str(fIdx))
                fig.canvas.draw_idle()
                plt.pause(0.01)

        # perform registration using diffeomorphic demons algorithm (from DIPY package)
        # https://dipy.org/documentation/1.1.1./examples_built/syn_registration_2d/#example-syn-registration-2d

        # specify the number of iterations in each of the levels in the multiresolution pyramid
        level_iters = [100, 50, 25]

        # create a registration metric
        sigma_diff = 3.0
        radius = 2
        metric = CCMetric(2, sigma_diff, radius)

        # create the registration object
        # TODO it may not be necessary to get the inverse displacement
        # field... setting inv_iter to 0 in an attempt to speed up
        # computation
        sdr = SymmetricDiffeomorphicRegistration(
            metric, level_iters, inv_iter=0)

        # iterate through the frame pairs and perform the registration each
        # ult_no_frames = 25 # just for debugging purposes
        debug_plot_ofreg = False

        # DO REGISTRATION (CHECK FOR PARALLELISM)
        ofoutput = []
        useParallelFlag = False
        if useParallelFlag:
            # setup parallelism for running the registration
            mgr = mp.Manager()
            storage = mgr.dict()  # create the storage for the optical flow
            ns = mgr.Namespace()
            ns.ultra_interp = ultra_interp
            ns.sdr = sdr

            procs = []

            # run the parallel processes
            for fIdx in range(0, ult_no_frames-1):
                proc = mp.Process(target=parallel_register,
                                  args=(ns, fIdx, ult_no_frames, storage))
                procs.append(proc)
                proc.start()

            # finalize the parallel processes
            for proc in procs:
                proc.join()

            # retrieve the output
            ofdisp = storage.values()
        else:
            # do registration without parallel computation support
            ofdisp = []

            for fIdx in range(ult_no_frames - 1):
                ofreg_logger.info("Working on frame pair %d of %d." %
                                  (fIdx, ult_no_frames - 1))
                current_im = ultra_interp[fIdx]
                next_im = ultra_interp[fIdx + 1]

                # execute the optimization
                ofdisp.append({'of': sdr.optimize(next_im, current_im),
                               'current frame': fIdx,
                               'next frame': fIdx + 1})

                # debug plotting
                if debug_plot_ofreg:
                    fig, ax = plt.subplots(1, 2)
                    ax[0].imshow(current_im)
                    ax[1].imshow(next_im)
                    plt.show()
                    plt.pause(0.05)

        ofreg_logger.info(
            "Finished computing optical flow for %s." % (item['base_name']))

        # debug plotting
        debug_plot_quiver = True
        if debug_plot_quiver:
            # visualize registration as quiver plot
            xx, yy = np.meshgrid(range(1, ultra_interp[0].shape[0]),
                                 range(1, ultra_interp[0].shape[1]))

            fig, ax = plt.subplots(1, 1)
            im = ax.imshow(ultra_interp[0])
            quiver = plt.quiver(yy, xx,
                                ofdisp[1]['of'].forward[:, :, 0],
                                ofdisp[1]['of'].forward[:, :, 1],
                                color='r')
            debug_frame = 1

            # create mousewheel callback function for updating the plot to a new frame
            def update_plot(event):
                nonlocal debug_frame
                # detect the mouse wheel action
                if event.button == 'up':
                    debug_frame = min(debug_frame + 1, ult_no_frames - 2)
                elif event.button == 'down':
                    debug_frame = max(debug_frame - 1, 1)
                else:
                    # TODO: this should do something else. some mice also move sideways
                    print("oops")

                # update the plot
                im.set_data(ultra_interp[debug_frame])
                quiver.set_UVC(ofdisp[debug_frame]['of'].forward[:, :, 0],
                               ofdisp[debug_frame]['of'].forward[:, :, 1])
                ax.set_title(str(debug_frame))
                fig.canvas.draw_idle()

            # register the callback function with the figure
            cid = fig.canvas.mpl_connect('scroll_event', update_plot)

        plt.show()

        # compute the ultrasound time vector
        ultra_time = np.linspace(
            0, ult_no_frames, ult_no_frames, endpoint=False) / ult_fps
        ultra_time = ultra_time + ult_TimeInSecOfFirstFrame + .5 / ult_fps

        ult_wav_time = np.linspace(
            0, len(ult_wav_frames), len(ult_wav_frames), endpoint=False)
        ult_wav_time /= ult_wav_fs

        data = {}
        data['ofdisp'] = ofdisp
        data['ultra_interp'] = ultra_interp
        data['ult_time'] = ultra_time
        data['wav_time'] = ult_wav_time
        data['ult_no_frames'] = ult_no_frames
        data['probe_array_length_mm'] = probe_array_length_mm
        data['probe_depth_mm'] = probe_depth_mm

        # Figure out where to save the results.
        data_dir, filename = os.path.split(item['base_name'])
        if 'data' in data_dir:
            save_dir = data_dir.replace('data', 'results')
        else:
            # TODO make sure that the ofbrowse/display code also knows
            # what the directory is called.
            save_dir = data_dir + '_results'

        # Create the results directory if it does not exist.
        if not os.path.isdir(save_dir):
            ofreg_logger.info("Creating directory " + save_dir + ".")
            os.mkdir(save_dir)

        # TODO the registration takes a long time to compute and each
        # should be saved, but this is not the best way.
        save_file = os.path.join(save_dir, filename) + "_OF.pickle"
        pickle.dump(data, open(save_file, "wb"))
        ofreg_logger.info("Saving results to %s" % save_file)
