from typing import List, Union

from django.utils.translation import gettext_lazy as _
from rest_framework.exceptions import ValidationError

from dynamicforms import fields, serializers
from dynamicforms.mixins import F
from examples.enum_field import EnumField
from examples.models import CalendarRecurrence
from examples.recurrence_utils import locale_adjectives, locale_weekdays


class DaysField(fields.CharField):
    def to_internal_value(self, data: str):
        def parse_token(token):
            token = token.strip()
            if len(token.split(" ")) == 2:
                return token.split(" ")
            return token

        if len(data.split(",")):
            return [parse_token(token) for token in data.split(",")]
        return parse_token(data)

    def to_representation(self, value: list, row_data=None):
        return ", ".join([" ".join(token) if not isinstance(token, str) else token for token in value])


class DaysValidator:
    message = _("Wrong input format")

    def __init__(self, message=None):
        self.message = message or self.message

    def __call__(self, value: list):
        def token_validation(token: Union[str, list]):
            if token is None:
                return
            if isinstance(token, str):
                token = token.strip()
                if not token.isnumeric() and 1 <= int(token) <= 31:
                    raise ValidationError(self.message, code="value")
            else:
                if token[0][:3].lower() not in locale_adjectives() or token[1] not in locale_weekdays():
                    raise ValidationError(self.message, code="value")

        for token in value:
            token_validation(token)


class DatesField(fields.CharField):
    def to_internal_value(self, data):
        def parse_token(token):
            token = token.strip()
            if len(val := [d for d in token.split(".") if d]) == 2:
                return val
            return token

        return [parse_token(token) for token in data.split(",")]

    def to_representation(self, value, row_data=None):
        return ", ".join([".".join(token) for token in value])


class DatesValidator:
    message = _("Wrong input format")

    def __init__(self, message=None):
        self.message = message or self.message

    def __call__(self, value: List[List[str]]):
        def token_validation(token: List[str]):
            if len(token) != 2 or not all([d.isnumeric() for d in token]):
                raise ValidationError(self.message, code="value")
            day, month = [int(d) for d in token]
            days_in_month = [31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31]
            if not (0 < month <= 12 and day <= days_in_month[month - 1]):
                raise ValidationError(self.message, code="value")

        for token in value:
            token_validation(token)


class RecurrenceSerializer(serializers.ModelSerializer):
    pattern = EnumField(CalendarRecurrence.Pattern)
    recur = fields.AutoGeneratedField(display=fields.DisplayMode.SUPPRESS)

    every = fields.IntegerField(
        allow_null=True,
        conditional_visibility=F("pattern").is_in(
            [CalendarRecurrence.Pattern.Daily, CalendarRecurrence.Pattern.Weekly]
        ),
        required=False,
    )
    weekdays = fields.MultipleChoiceField(
        allow_null=True,
        choices=dict(zip(("ho", "mo", "tu", "we", "th", "fr", "sa", "su"), range(8))),
        conditional_visibility=F("pattern") == CalendarRecurrence.Pattern.Weekly,
        required=False,
    )
    days = DaysField(
        allow_null=True,
        conditional_visibility=F("pattern") == CalendarRecurrence.Pattern.Monthly,
        required=False,
        validators=[DaysValidator()],
    )
    dates = DatesField(
        allow_null=True,
        conditional_visibility=F("pattern") == CalendarRecurrence.Pattern.Yearly,
        required=False,
        validators=[DatesValidator()],
    )

    def __init__(self, *args, **kwds):
        super().__init__(*args, **kwds)
        self.fields["id"].read_only = False  # https://stackoverflow.com/a/46525126/1760858

    def to_representation(self, instance, row_data=None):
        setattr(instance, "every", instance.recur.get("every", None))
        if instance.pattern == CalendarRecurrence.Pattern.Weekly:
            setattr(instance, "weekdays", instance.recur.get("weekdays", None))
        if instance.pattern == CalendarRecurrence.Pattern.Monthly:
            setattr(instance, "days", instance.recur.get("days", None))
        if instance.pattern == CalendarRecurrence.Pattern.Yearly:
            setattr(instance, "dates", instance.recur.get("dates", None))

        return super().to_representation(instance, row_data)

    def to_internal_value(self, data: dict):
        data = super().to_internal_value(data)
        data["recur"] = dict(
            every=data.pop("every", None),
            days=data.pop("days", None),
            dates=data.pop("dates", None),
            weekdays=list(data.pop("weekdays", set())),
        )
        if data["pattern"] != CalendarRecurrence.Pattern.Weekly.value:
            data.pop("weekdays", None)
        if data["pattern"] != CalendarRecurrence.Pattern.Monthly.value:
            data.pop("days", None)
        if data["pattern"] != CalendarRecurrence.Pattern.Yearly.value:
            data.pop("dates", None)

        return data

    class Meta:
        model = CalendarRecurrence
        exclude = ("recur",)
