"""Módulo para generar archivos excel a partir de QUERYS de la base de datos"""
from os import remove as os_remove, path, getcwd
from datetime import datetime
import pandas as pd
from openpyxl import load_workbook
from openpyxl.styles import Font, Alignment, Border, Side


class ExcelServiceError(Exception):
    """Excepción personalizada para errores en ExcelGenerator"""

    def __init__(self, message: str):
        super().__init__(f"ExcelServiceError: {message}")


class ExcelSheet:
    """Representa una hoja de un archivo excel a partir de un dataframe de pandas"""

    def __init__(self, name: str, data_frame: pd.DataFrame):
        self.name = name
        if not isinstance(data_frame, pd.DataFrame):
            raise ExcelServiceError(
                'Parameter "data_frame" must be a pandas DataFrame')
        self.data_frame = data_frame

    def get_sheet_name(self) -> str:
        """Getter para el nombre de la hoja"""
        return self.name

    def get_data_frame(self) -> pd.DataFrame:
        """Getter para el dataframe"""
        return self.data_frame


class ExcelFile:
    """Clase para generar archivos excel a partir de dataframes de pandas"""
    destination_path: str
    sheets: list[ExcelSheet]

    def __init__(self, destination_path: str = None):
        if destination_path is None:
            file_name = f'AutoGeneratedExcelFile_{
                datetime.now().strftime("%d-%m-%Y_%H-%M-%S")}.xlsx'
            destination_path = path.join(getcwd(), file_name)
        if not destination_path.endswith('.xlsx') and not destination_path.endswith('.xls'):
            destination_path += '.xlsx'
        self.destination_path = destination_path
        self.sheets: list[ExcelSheet] = []

    def append_sheet_from_df(self, data_frame: pd.DataFrame, sheet_name: str = None):
        """Agrega una hoja a la lista de hojas del archivo excel"""
        if sheet_name is None:
            sheet_name = f'Sheet{len(self.sheets) + 1}'
        elif sheet_name in [sheet.get_sheet_name() for sheet in self.sheets]:
            raise ExcelServiceError(
                f'Ya existe una hoja con el nombre "{sheet_name}"')
        self.sheets.append(ExcelSheet(sheet_name, data_frame))

    def _format_workbook(self):
        """Formatea el archivo excel"""
        try:
            thin = Side(border_style="thin", color="000000")
            workbook = load_workbook(self.destination_path)
            for worksheet in workbook.worksheets:
                for col in worksheet.columns:
                    max_length = 0
                    column = col[0].column_letter
                    for cell in col[0:1]:  # Formatear encabezados
                        cell.font = Font(name='Roboto', size=12, bold=True)
                        cell.alignment = Alignment(horizontal='center')
                        cell.border = Border(top=thin, left=thin,
                                             right=thin, bottom=thin)
                        if len(str(cell.value)) > max_length:
                            max_length = len(str(cell.value))
                    for cell in col[1:]:
                        cell.font = Font(name='Roboto', size=10)
                        cell.alignment = Alignment(horizontal='center')
                        cell.border = Border(top=thin, left=thin,
                                             right=thin, bottom=thin)
                        if len(str(cell.value)) > max_length:
                            max_length = len(str(cell.value))

                    adjusted_width = (max_length + 2)
                    worksheet.column_dimensions[column].width = adjusted_width

            workbook.save(self.destination_path)
        except Exception as e:
            raise ExcelServiceError(
                f'Error al formatear el archivo excel: {str(e)}') from e

    def save(self):
        """Guarda el archivo excel en la ruta especificada"""
        try:
            with pd.ExcelWriter(self.destination_path, engine='openpyxl') as writer:  # pylint: disable=abstract-class-instantiated
                for sheet in self.sheets:
                    sheet.get_data_frame().to_excel(
                        writer, sheet_name=sheet.get_sheet_name(), index=False)
            self._format_workbook()
        except Exception as e:
            raise ExcelServiceError(
                f'Cannot save file at "{self.destination_path}": {str(e)}') from e

    def remove(self):
        """Remover el archivo excel"""
        os_remove(self.destination_path)

    def get_destination_path(self) -> str:
        """Getter para la ruta de destino del archivo"""
        return self.destination_path
