import aiofiles
import os
import shutil


async def load_systemd(hub):
    cmd = shutil.which("systemctl")
    if cmd and shutil.which("localectl"):
        hub.log.debug("Adding systemd corn")
        ret = await hub.exec.cmd.run([cmd, "--version"])
        if ret.retcode:
            hub.log.error(ret.stderr)
        else:
            systemd_info = ret.stdout.strip().splitlines()
            hub.corn.CORN.systemd.version = systemd_info[0].split()[1]
            hub.corn.CORN.systemd.features = systemd_info[1]


async def _load_cgroup(hub) -> str:
    path = "/proc/1/cgroup"
    if os.path.exists(path):
        async with aiofiles.open(path) as fh_:
            data = await fh_.read()
            if "docker" in data:
                return "docker"
            elif "name=" in data:
                name = data.split("name=")[1]
                return name.split(":")[0]


async def load_init(hub):
    # Add init grain
    # Default to the cgroup
    hub.log.debug("Adding init grain")
    hub.corn.CORN.init = await _load_cgroup(hub) or "unknown"

    system = "/run/systemd/system"
    cmdline = "/proc/1/cmdline"

    if os.path.exists(system) and os.stat(system):
        hub.corn.CORN.init = "systemd"
    elif os.path.exists(cmdline):
        async with aiofiles.open("/proc/1/cmdline") as fhr:
            init_cmdline = (await fhr.read()).replace("\x00", " ").split()
            if len(init_cmdline) >= 1:
                init_bin = shutil.which(init_cmdline[0])
                if init_bin.endswith("bin/init"):
                    supported_inits = (b"upstart", b"sysvinit", b"systemd")
                    edge_len = max(len(x) for x in supported_inits) - 1
                    buf_size = hub.OPT.get("file_buffer_size", 262144)
                    try:
                        async with aiofiles.open(init_bin, "rb") as fp_:
                            edge = b""
                            buf = (await fp_.read(buf_size)).lower()
                            while buf:
                                if isinstance(buf, str):
                                    # This makes testing easier
                                    buf = buf.encode()
                                buf = edge + buf
                                for item in supported_inits:
                                    if item in buf:
                                        item = item.decode("utf-8")
                                        hub.corn.CORN.init = item
                                        buf = b""
                                        break
                                edge = buf[-edge_len:]
                                buf = (await fp_.read(buf_size)).lower()
                    except (IOError, OSError) as exc:
                        hub.log.error(
                            "Unable to read from init_bin (%s): %s", init_bin, exc
                        )
                elif shutil.which("supervisord") in init_cmdline:
                    hub.corn.CORN.init = "supervisord"
                elif shutil.which("dumb-init") in init_cmdline:
                    # https://github.com/Yelp/dumb-init
                    hub.corn.CORN.init = "dumb-init"
                elif shutil.which("tini") in init_cmdline:
                    # https://github.com/krallin/tini
                    hub.corn.CORN.init = "tini"
                elif "runit" in init_cmdline:
                    hub.corn.CORN.init = "runit"
                elif "/sbin/my_init" in init_cmdline:
                    # Phusion Base docker container use runit for srv mgmt, but
                    # my_init as pid1
                    hub.corn.CORN.init = "runit"
                else:
                    hub.log.debug(
                        "Could not determine init system from command line: (%s)",
                        " ".join(init_cmdline),
                    )
            else:
                # Emtpy init_cmdline
                hub.log.warning("Unable to fetch data from /proc/1/cmdline")
