"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sqs = require("@aws-cdk/aws-sqs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const send_message_1 = require("../../lib/sqs/send-message");
/*
 * Creates a state machine with a task state to send a message to an SQS
 * queue.
 *
 * When the state machine is executed, it will send a message to our
 * queue, which can subsequently be consumed.
 *
 * Stack verification steps:
 * The generated State Machine can be executed from the CLI (or Step Functions console)
 * and runs with an execution status of `Succeeded`.
 *
 * -- aws stepfunctions start-execution --state-machine-arn <state-machine-arn-from-output> provides execution arn
 * -- aws stepfunctions describe-execution --execution-arn <from previous command> returns a status of `Succeeded`
 * -- aws sqs receive-message --queue-url <queue-url-from-output> has a message of 'sending message over'
 */
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-stepfunctions-tasks-sqs-send-message-integ');
const queue = new sqs.Queue(stack, 'show-me-the-messages');
const sendMessageTask = new send_message_1.SqsSendMessage(stack, 'send message to sqs', {
    queue,
    messageBody: sfn.TaskInput.fromText('sending message over'),
});
const finalStatus = new sfn.Pass(stack, 'Final step');
const chain = sfn.Chain.start(sendMessageTask)
    .next(finalStatus);
const sm = new sfn.StateMachine(stack, 'StateMachine', {
    definition: chain,
    timeout: cdk.Duration.seconds(30),
});
new cdk.CfnOutput(stack, 'stateMachineArn', {
    value: sm.stateMachineArn,
});
new cdk.CfnOutput(stack, 'queueUrl', {
    value: queue.queueUrl,
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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