"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
const create_training_job_1 = require("../../lib/sagemaker/create-training-job");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('create basic training job', () => {
    // WHEN
    const task = new create_training_job_1.SageMakerCreateTrainingJob(stack, 'TrainSagemaker', {
        trainingJobName: 'MyTrainJob',
        algorithmSpecification: {
            algorithmName: 'BlazingText',
        },
        inputDataConfig: [
            {
                channelName: 'train',
                dataSource: {
                    s3DataSource: {
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucket', 'mybucket'), 'mytrainpath'),
                    },
                },
            },
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath'),
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            AlgorithmSpecification: {
                AlgorithmName: 'BlazingText',
                TrainingInputMode: 'File',
            },
            InputDataConfig: [
                {
                    ChannelName: 'train',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytrainpath']],
                            },
                        },
                    },
                },
            ],
            OutputDataConfig: {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']],
                },
            },
            ResourceConfig: {
                InstanceCount: 1,
                InstanceType: 'ml.m4.xlarge',
                VolumeSizeInGB: 10,
            },
            RoleArn: { 'Fn::GetAtt': ['TrainSagemakerSagemakerRole89E8C593', 'Arn'] },
            StoppingCondition: {
                MaxRuntimeInSeconds: 3600,
            },
            TrainingJobName: 'MyTrainJob',
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new create_training_job_1.SageMakerCreateTrainingJob(stack, 'TrainSagemaker', {
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            trainingJobName: 'MyTrainJob',
            algorithmSpecification: {
                algorithmName: 'BlazingText',
            },
            inputDataConfig: [
                {
                    channelName: 'train',
                    dataSource: {
                        s3DataSource: {
                            s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucket', 'mybucket'), 'mytrainpath'),
                        },
                    },
                },
            ],
            outputDataConfig: {
                s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath'),
            },
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,RUN_JOB. Received: WAIT_FOR_TASK_TOKEN/i);
});
test('create complex training job', () => {
    // WHEN
    const kmsKey = new kms.Key(stack, 'Key');
    const vpc = new ec2.Vpc(stack, 'VPC');
    const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc, description: 'My SG' });
    securityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'allow ssh access from the world');
    const role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
        ],
    });
    const trainTask = new create_training_job_1.SageMakerCreateTrainingJob(stack, 'TrainSagemaker', {
        trainingJobName: 'MyTrainJob',
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        role,
        algorithmSpecification: {
            algorithmName: 'BlazingText',
            trainingInputMode: tasks.InputMode.FILE,
            metricDefinitions: [
                {
                    name: 'mymetric', regex: 'regex_pattern',
                },
            ],
        },
        hyperparameters: {
            lr: '0.1',
        },
        inputDataConfig: [
            {
                channelName: 'train',
                contentType: 'image/jpeg',
                compressionType: tasks.CompressionType.NONE,
                recordWrapperType: tasks.RecordWrapperType.RECORD_IO,
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucketA', 'mybucket'), 'mytrainpath'),
                    },
                },
            },
            {
                channelName: 'test',
                contentType: 'image/jpeg',
                compressionType: tasks.CompressionType.GZIP,
                recordWrapperType: tasks.RecordWrapperType.RECORD_IO,
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucketB', 'mybucket'), 'mytestpath'),
                    },
                },
            },
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath'),
            encryptionKey: kmsKey,
        },
        resourceConfig: {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
            volumeSize: cdk.Size.gibibytes(50),
            volumeEncryptionKey: kmsKey,
        },
        stoppingCondition: {
            maxRuntime: cdk.Duration.hours(1),
        },
        tags: {
            Project: 'MyProject',
        },
        vpcConfig: {
            vpc,
        },
    });
    trainTask.addSecurityGroup(securityGroup);
    // THEN
    expect(stack.resolve(trainTask.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            TrainingJobName: 'MyTrainJob',
            RoleArn: { 'Fn::GetAtt': ['Role1ABCC5F0', 'Arn'] },
            AlgorithmSpecification: {
                TrainingInputMode: 'File',
                AlgorithmName: 'BlazingText',
                MetricDefinitions: [
                    { Name: 'mymetric', Regex: 'regex_pattern' },
                ],
            },
            HyperParameters: {
                lr: '0.1',
            },
            InputDataConfig: [
                {
                    ChannelName: 'train',
                    CompressionType: 'None',
                    RecordWrapperType: 'RecordIO',
                    ContentType: 'image/jpeg',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytrainpath']],
                            },
                        },
                    },
                },
                {
                    ChannelName: 'test',
                    CompressionType: 'Gzip',
                    RecordWrapperType: 'RecordIO',
                    ContentType: 'image/jpeg',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytestpath']],
                            },
                        },
                    },
                },
            ],
            OutputDataConfig: {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']],
                },
                KmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            ResourceConfig: {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeSizeInGB: 50,
                VolumeKmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            StoppingCondition: {
                MaxRuntimeInSeconds: 3600,
            },
            Tags: [
                { Key: 'Project', Value: 'MyProject' },
            ],
            VpcConfig: {
                SecurityGroupIds: [
                    { 'Fn::GetAtt': ['TrainSagemakerTrainJobSecurityGroup7C858EB9', 'GroupId'] },
                    { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
                ],
                Subnets: [
                    { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                    { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                ],
            },
        },
    });
});
test('pass param to training job', () => {
    // WHEN
    const role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
        ],
    });
    const task = new create_training_job_1.SageMakerCreateTrainingJob(stack, 'TrainSagemaker', {
        trainingJobName: sfn.JsonPath.stringAt('$.JobName'),
        role,
        algorithmSpecification: {
            algorithmName: 'BlazingText',
            trainingInputMode: tasks.InputMode.FILE,
        },
        inputDataConfig: [
            {
                channelName: 'train',
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromJsonExpression('$.S3Bucket'),
                    },
                },
            },
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'Bucket', 'mybucket'), 'myoutputpath'),
        },
        resourceConfig: {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
            volumeSize: cdk.Size.gibibytes(50),
        },
        stoppingCondition: {
            maxRuntime: cdk.Duration.hours(1),
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            'TrainingJobName.$': '$.JobName',
            'RoleArn': { 'Fn::GetAtt': ['Role1ABCC5F0', 'Arn'] },
            'AlgorithmSpecification': {
                TrainingInputMode: 'File',
                AlgorithmName: 'BlazingText',
            },
            'InputDataConfig': [
                {
                    ChannelName: 'train',
                    DataSource: {
                        S3DataSource: {
                            'S3DataType': 'S3Prefix',
                            'S3Uri.$': '$.S3Bucket',
                        },
                    },
                },
            ],
            'OutputDataConfig': {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']],
                },
            },
            'ResourceConfig': {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeSizeInGB: 50,
            },
            'StoppingCondition': {
                MaxRuntimeInSeconds: 3600,
            },
        },
    });
});
test('Cannot create a SageMaker train task with both algorithm name and image name missing', () => {
    expect(() => new create_training_job_1.SageMakerCreateTrainingJob(stack, 'SageMakerTrainingTask', {
        trainingJobName: 'myTrainJob',
        algorithmSpecification: {},
        inputDataConfig: [
            {
                channelName: 'train',
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromJsonExpression('$.S3Bucket'),
                    },
                },
            },
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'Bucket', 'mybucket'), 'myoutputpath/'),
        },
    }))
        .toThrowError(/Must define either an algorithm name or training image URI in the algorithm specification/);
});
//# sourceMappingURL=data:application/json;base64,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