"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lambda = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
/* eslint-disable quote-props */
describe('LambdaInvoke', () => {
    let stack;
    let lambdaFunction;
    beforeEach(() => {
        // GIVEN
        stack = new core_1.Stack();
        lambdaFunction = new lambda.Function(stack, 'Fn', {
            code: lambda.Code.fromInline('foo'),
            handler: 'handler',
            runtime: lambda.Runtime.NODEJS_12_X,
        });
    });
    test('default settings', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            End: true,
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke',
                    ],
                ],
            },
            Parameters: {
                FunctionName: {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                'Payload.$': '$',
            },
        });
    });
    test('optional settings', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            payload: sfn.TaskInput.fromObject({
                foo: 'bar',
            }),
            invocationType: lib_1.LambdaInvocationType.REQUEST_RESPONSE,
            clientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
            qualifier: '1',
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke',
                    ],
                ],
            },
            End: true,
            Parameters: {
                FunctionName: {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                Payload: {
                    foo: 'bar',
                },
                InvocationType: 'RequestResponse',
                ClientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
                Qualifier: '1',
            },
        });
    });
    test('invoke Lambda function and wait for task token', () => {
        // GIVEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            payload: sfn.TaskInput.fromObject({
                token: sfn.JsonPath.taskToken,
            }),
            qualifier: 'my-alias',
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke.waitForTaskToken',
                    ],
                ],
            },
            End: true,
            Parameters: {
                FunctionName: {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                Payload: {
                    'token.$': '$$.Task.Token',
                },
                Qualifier: 'my-alias',
            },
        });
    });
    test('pass part of state input as input to Lambda function ', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            payload: sfn.TaskInput.fromDataAt('$.foo'),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke',
                    ],
                ],
            },
            End: true,
            Parameters: {
                'FunctionName': {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                'Payload.$': '$.foo',
            },
        });
    });
    test('Invoke lambda with payloadResponseOnly', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            payloadResponseOnly: true,
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            End: true,
            Type: 'Task',
            Resource: {
                'Fn::GetAtt': [
                    'Fn9270CBC0',
                    'Arn',
                ],
            },
        });
    });
    test('Invoke lambda with payloadResponseOnly with payload', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            payloadResponseOnly: true,
            payload: sfn.TaskInput.fromObject({
                foo: 'bar',
            }),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            End: true,
            Type: 'Task',
            Resource: {
                'Fn::GetAtt': [
                    'Fn9270CBC0',
                    'Arn',
                ],
            },
            Parameters: {
                foo: 'bar',
            },
        });
    });
    test('fails when integrationPattern used with payloadResponseOnly', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                payloadResponseOnly: true,
                integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
                payload: sfn.TaskInput.fromObject({
                    token: sfn.JsonPath.taskToken,
                }),
            });
        }).toThrow(/The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified./);
    });
    test('fails when invocationType used with payloadResponseOnly', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                payloadResponseOnly: true,
                payload: sfn.TaskInput.fromObject({
                    foo: 'bar',
                }),
                invocationType: lib_1.LambdaInvocationType.REQUEST_RESPONSE,
            });
        }).toThrow(/The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified./);
    });
    test('fails when clientContext used with payloadResponseOnly', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                payloadResponseOnly: true,
                payload: sfn.TaskInput.fromObject({
                    foo: 'bar',
                }),
                clientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
            });
        }).toThrow(/The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified./);
    });
    test('fails when qualifier used with payloadResponseOnly', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                payloadResponseOnly: true,
                payload: sfn.TaskInput.fromObject({
                    foo: 'bar',
                }),
                qualifier: '1',
            });
        }).toThrow(/The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified./);
    });
    test('fails when WAIT_FOR_TASK_TOKEN integration pattern is used without supplying a task token in payload', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            });
        }).toThrow(/Task Token is required in `payload` for callback. Use JsonPath.taskToken to set the token./);
    });
    test('fails when RUN_JOB integration pattern is used', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                integrationPattern: sfn.IntegrationPattern.RUN_JOB,
            });
        }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,WAIT_FOR_TASK_TOKEN. Received: RUN_JOB/);
    });
});
//# sourceMappingURL=data:application/json;base64,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