"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const tasks = require("../../lib");
const start_job_run_1 = require("../../lib/glue/start-job-run");
const glueJobName = 'GlueJob';
let stack;
beforeEach(() => {
    stack = new core_1.Stack();
});
test('Invoke glue job with just job ARN', () => {
    const task = new start_job_run_1.GlueStartJobRun(stack, 'Task', {
        glueJobName,
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::glue:startJobRun',
                ],
            ],
        },
        End: true,
        Parameters: {
            JobName: glueJobName,
        },
    });
});
test('Invoke glue job with full properties', () => {
    const jobArguments = {
        key: 'value',
    };
    const timeoutMinutes = 1440;
    const glueJobTimeout = core_1.Duration.minutes(timeoutMinutes);
    const securityConfiguration = 'securityConfiguration';
    const notifyDelayAfterMinutes = 10;
    const notifyDelayAfter = core_1.Duration.minutes(notifyDelayAfterMinutes);
    const task = new start_job_run_1.GlueStartJobRun(stack, 'Task', {
        glueJobName,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        arguments: sfn.TaskInput.fromObject(jobArguments),
        timeout: glueJobTimeout,
        securityConfiguration,
        notifyDelayAfter,
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::glue:startJobRun.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            JobName: glueJobName,
            Arguments: jobArguments,
            Timeout: timeoutMinutes,
            SecurityConfiguration: securityConfiguration,
            NotificationProperty: {
                NotifyDelayAfter: notifyDelayAfterMinutes,
            },
        },
    });
});
test('job arguments can reference state input', () => {
    const task = new start_job_run_1.GlueStartJobRun(stack, 'Task', {
        glueJobName,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        arguments: sfn.TaskInput.fromDataAt('$.input'),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::glue:startJobRun.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            'JobName': glueJobName,
            'Arguments.$': '$.input',
        },
    });
});
test('permitted role actions limited to start job run if service integration pattern is REQUEST_RESPONSE', () => {
    const task = new start_job_run_1.GlueStartJobRun(stack, 'Task', {
        glueJobName,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [{
                    Action: 'glue:StartJobRun',
                }],
        },
    });
});
test('permitted role actions include start, get, and stop job run if service integration pattern is RUN_JOB', () => {
    const task = new start_job_run_1.GlueStartJobRun(stack, 'Task', {
        glueJobName,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [{
                    Action: [
                        'glue:StartJobRun',
                        'glue:GetJobRun',
                        'glue:GetJobRuns',
                        'glue:BatchStopJobRun',
                    ],
                }],
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunGlueJobTask(glueJobName, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            }),
        });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call Glue./i);
});
//# sourceMappingURL=data:application/json;base64,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