"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../../lib");
let stack;
let clusterRole;
let serviceRole;
let autoScalingRole;
beforeEach(() => {
    var _a;
    // GIVEN
    stack = new cdk.Stack();
    clusterRole = new iam.Role(stack, 'ClusterRole', {
        assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
    });
    serviceRole = new iam.Role(stack, 'ServiceRole', {
        assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
    });
    autoScalingRole = new iam.Role(stack, 'AutoScalingRole', {
        assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
    });
    (_a = autoScalingRole.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        principals: [
            new iam.ServicePrincipal('application-autoscaling.amazonaws.com'),
        ],
        actions: [
            'sts:AssumeRole',
        ],
    }));
});
test('Create Cluster with FIRE_AND_FORGET integrationPattern', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with SYNC integrationPattern', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with clusterConfiguration Name from payload', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: sfn.TaskInput.fromDataAt('$.ClusterName').value,
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            'Name.$': '$.ClusterName',
            'Instances': {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            'VisibleToAllUsers': true,
            'JobFlowRole': {
                Ref: 'ClusterRoleD9CA7471',
            },
            'ServiceRole': {
                Ref: 'ServiceRole4288B192',
            },
            'AutoScalingRole': {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with Tags', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        tags: {
            key: 'value',
        },
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            Tags: [{
                    Key: 'key',
                    Value: 'value',
                }],
        },
    });
});
test('Create Cluster with Applications', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        applications: [
            { name: 'Hive', version: '0.0' },
            { name: 'Spark', version: '0.0' },
        ],
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            Applications: [
                { Name: 'Hive', Version: '0.0' },
                { Name: 'Spark', Version: '0.0' },
            ],
        },
    });
});
test('Create Cluster with Bootstrap Actions', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        bootstrapActions: [{
                name: 'Bootstrap',
                scriptBootstrapAction: {
                    path: 's3://null',
                    args: ['Arg'],
                },
            }],
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            BootstrapActions: [{
                    Name: 'Bootstrap',
                    ScriptBootstrapAction: {
                        Path: 's3://null',
                        Args: ['Arg'],
                    },
                }],
        },
    });
});
test('Create Cluster with Configurations', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        configurations: [{
                classification: 'classification',
                properties: {
                    Key: 'Value',
                },
            }],
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            Configurations: [{
                    Classification: 'classification',
                    Properties: {
                        Key: 'Value',
                    },
                }],
        },
    });
});
test('Create Cluster with KerberosAttributes', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        kerberosAttributes: {
            realm: 'realm',
            adDomainJoinPassword: 'password1',
            adDomainJoinUser: 'user',
            crossRealmTrustPrincipalPassword: 'password2',
            kdcAdminPassword: 'password3',
        },
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
            KerberosAttributes: {
                Realm: 'realm',
                ADDomainJoinPassword: 'password1',
                ADDomainJoinUser: 'user',
                CrossRealmTrustPrincipalPassword: 'password2',
                KdcAdminPassword: 'password3',
            },
        },
    });
});
test('Create Cluster without Roles', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {},
        name: 'Cluster',
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'TaskInstanceRoleB72072BF',
            },
            ServiceRole: {
                Ref: 'TaskServiceRoleBF55F61E',
            },
            AutoScalingRole: {
                Ref: 'TaskAutoScalingRoleD06F8423',
            },
        },
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: 'elasticmapreduce.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
            ],
        },
    });
    // The stack renders the ec2.amazonaws.com Service principal id with a
    // Join to the URLSuffix
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: {
                            'Fn::Join': [
                                '',
                                [
                                    'ec2.',
                                    {
                                        Ref: 'AWS::URLSuffix',
                                    },
                                ],
                            ],
                        },
                    },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
            ],
        },
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: 'elasticmapreduce.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
                {
                    Principal: { Service: 'application-autoscaling.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                },
            ],
        },
    });
});
test('Create Cluster with Instances configuration', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {
            additionalMasterSecurityGroups: ['MasterGroup'],
            additionalSlaveSecurityGroups: ['SlaveGroup'],
            ec2KeyName: 'Ec2KeyName',
            ec2SubnetId: 'Ec2SubnetId',
            ec2SubnetIds: ['Ec2SubnetId'],
            emrManagedMasterSecurityGroup: 'MasterGroup',
            emrManagedSlaveSecurityGroup: 'SlaveGroup',
            hadoopVersion: 'HadoopVersion',
            instanceCount: 1,
            masterInstanceType: 'MasterInstanceType',
            placement: {
                availabilityZone: 'AvailabilityZone',
                availabilityZones: ['AvailabilityZone'],
            },
            serviceAccessSecurityGroup: 'ServiceAccessGroup',
            slaveInstanceType: 'SlaveInstanceType',
            terminationProtected: true,
        },
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                AdditionalMasterSecurityGroups: ['MasterGroup'],
                AdditionalSlaveSecurityGroups: ['SlaveGroup'],
                Ec2KeyName: 'Ec2KeyName',
                Ec2SubnetId: 'Ec2SubnetId',
                Ec2SubnetIds: ['Ec2SubnetId'],
                EmrManagedMasterSecurityGroup: 'MasterGroup',
                EmrManagedSlaveSecurityGroup: 'SlaveGroup',
                HadoopVersion: 'HadoopVersion',
                InstanceCount: 1,
                KeepJobFlowAliveWhenNoSteps: true,
                MasterInstanceType: 'MasterInstanceType',
                Placement: {
                    AvailabilityZone: 'AvailabilityZone',
                    AvailabilityZones: ['AvailabilityZone'],
                },
                ServiceAccessSecurityGroup: 'ServiceAccessGroup',
                SlaveInstanceType: 'SlaveInstanceType',
                TerminationProtected: true,
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Create Cluster with InstanceFleet', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {
            instanceFleets: [{
                    instanceFleetType: lib_1.EmrCreateCluster.InstanceRoleType.MASTER,
                    instanceTypeConfigs: [{
                            bidPrice: '1',
                            bidPriceAsPercentageOfOnDemandPrice: 1,
                            configurations: [{
                                    classification: 'Classification',
                                    properties: {
                                        Key: 'Value',
                                    },
                                }],
                            ebsConfiguration: {
                                ebsBlockDeviceConfigs: [{
                                        volumeSpecification: {
                                            iops: 1,
                                            volumeSize: cdk.Size.gibibytes(1),
                                            volumeType: lib_1.EmrCreateCluster.EbsBlockDeviceVolumeType.STANDARD,
                                        },
                                        volumesPerInstance: 1,
                                    }],
                                ebsOptimized: true,
                            },
                            instanceType: 'm5.xlarge',
                            weightedCapacity: 1,
                        }],
                    launchSpecifications: {
                        spotSpecification: {
                            blockDurationMinutes: 1,
                            timeoutAction: lib_1.EmrCreateCluster.SpotTimeoutAction.TERMINATE_CLUSTER,
                            timeoutDurationMinutes: 1,
                        },
                    },
                    name: 'Master',
                    targetOnDemandCapacity: 1,
                    targetSpotCapacity: 1,
                }],
        },
        clusterRole,
        name: 'Cluster',
        serviceRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
                InstanceFleets: [{
                        InstanceFleetType: 'MASTER',
                        InstanceTypeConfigs: [{
                                BidPrice: '1',
                                BidPriceAsPercentageOfOnDemandPrice: 1,
                                Configurations: [{
                                        Classification: 'Classification',
                                        Properties: {
                                            Key: 'Value',
                                        },
                                    }],
                                EbsConfiguration: {
                                    EbsBlockDeviceConfigs: [{
                                            VolumeSpecification: {
                                                Iops: 1,
                                                SizeInGB: 1,
                                                VolumeType: 'standard',
                                            },
                                            VolumesPerInstance: 1,
                                        }],
                                    EbsOptimized: true,
                                },
                                InstanceType: 'm5.xlarge',
                                WeightedCapacity: 1,
                            }],
                        LaunchSpecifications: {
                            SpotSpecification: {
                                BlockDurationMinutes: 1,
                                TimeoutAction: 'TERMINATE_CLUSTER',
                                TimeoutDurationMinutes: 1,
                            },
                        },
                        Name: 'Master',
                        TargetOnDemandCapacity: 1,
                        TargetSpotCapacity: 1,
                    }],
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
        },
    });
});
test('Create Cluster with InstanceGroup', () => {
    // WHEN
    const task = new lib_1.EmrCreateCluster(stack, 'Task', {
        instances: {
            instanceGroups: [{
                    autoScalingPolicy: {
                        constraints: {
                            maxCapacity: 2,
                            minCapacity: 1,
                        },
                        rules: [{
                                action: {
                                    market: lib_1.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                                    simpleScalingPolicyConfiguration: {
                                        adjustmentType: lib_1.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                        coolDown: 1,
                                        scalingAdjustment: 1,
                                    },
                                },
                                description: 'Description',
                                name: 'Name',
                                trigger: {
                                    cloudWatchAlarmDefinition: {
                                        comparisonOperator: lib_1.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN,
                                        dimensions: [{
                                                key: 'Key',
                                                value: 'Value',
                                            }],
                                        evaluationPeriods: 1,
                                        metricName: 'Name',
                                        namespace: 'Namespace',
                                        period: cdk.Duration.seconds(300),
                                        statistic: lib_1.EmrCreateCluster.CloudWatchAlarmStatistic.AVERAGE,
                                        threshold: 1,
                                        unit: lib_1.EmrCreateCluster.CloudWatchAlarmUnit.NONE,
                                    },
                                },
                            }, {
                                action: {
                                    market: lib_1.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                                    simpleScalingPolicyConfiguration: {
                                        adjustmentType: lib_1.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                        coolDown: 1,
                                        scalingAdjustment: 1,
                                    },
                                },
                                description: 'Description',
                                name: 'Name',
                                trigger: {
                                    cloudWatchAlarmDefinition: {
                                        comparisonOperator: lib_1.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN,
                                        dimensions: [{
                                                key: 'Key',
                                                value: 'Value',
                                            }],
                                        evaluationPeriods: 1,
                                        metricName: 'Name',
                                        namespace: 'Namespace',
                                        period: cdk.Duration.seconds(sfn.JsonPath.numberAt('$.CloudWatchPeriod')),
                                        statistic: lib_1.EmrCreateCluster.CloudWatchAlarmStatistic.AVERAGE,
                                        threshold: 1,
                                        unit: lib_1.EmrCreateCluster.CloudWatchAlarmUnit.NONE,
                                    },
                                },
                            }],
                    },
                    bidPrice: '1',
                    configurations: [{
                            classification: 'Classification',
                            properties: {
                                Key: 'Value',
                            },
                        }],
                    ebsConfiguration: {
                        ebsBlockDeviceConfigs: [{
                                volumeSpecification: {
                                    iops: 1,
                                    volumeSize: cdk.Size.gibibytes(1),
                                    volumeType: lib_1.EmrCreateCluster.EbsBlockDeviceVolumeType.STANDARD,
                                },
                                volumesPerInstance: 1,
                            }],
                        ebsOptimized: true,
                    },
                    instanceCount: 1,
                    instanceRole: lib_1.EmrCreateCluster.InstanceRoleType.MASTER,
                    instanceType: 'm5.xlarge',
                    market: lib_1.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                    name: 'Name',
                }],
        },
        clusterRole,
        name: 'Cluster',
        serviceRole,
        autoScalingRole,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
                InstanceGroups: [{
                        AutoScalingPolicy: {
                            Constraints: {
                                MaxCapacity: 2,
                                MinCapacity: 1,
                            },
                            Rules: [{
                                    Action: {
                                        Market: 'ON_DEMAND',
                                        SimpleScalingPolicyConfiguration: {
                                            AdjustmentType: 'CHANGE_IN_CAPACITY',
                                            CoolDown: 1,
                                            ScalingAdjustment: 1,
                                        },
                                    },
                                    Description: 'Description',
                                    Name: 'Name',
                                    Trigger: {
                                        CloudWatchAlarmDefinition: {
                                            ComparisonOperator: 'GREATER_THAN',
                                            Dimensions: [{
                                                    Key: 'Key',
                                                    Value: 'Value',
                                                }],
                                            EvaluationPeriods: 1,
                                            MetricName: 'Name',
                                            Namespace: 'Namespace',
                                            Period: 300,
                                            Statistic: 'AVERAGE',
                                            Threshold: 1,
                                            Unit: 'NONE',
                                        },
                                    },
                                }, {
                                    Action: {
                                        Market: 'ON_DEMAND',
                                        SimpleScalingPolicyConfiguration: {
                                            AdjustmentType: 'CHANGE_IN_CAPACITY',
                                            CoolDown: 1,
                                            ScalingAdjustment: 1,
                                        },
                                    },
                                    Description: 'Description',
                                    Name: 'Name',
                                    Trigger: {
                                        CloudWatchAlarmDefinition: {
                                            'ComparisonOperator': 'GREATER_THAN',
                                            'Dimensions': [{
                                                    Key: 'Key',
                                                    Value: 'Value',
                                                }],
                                            'EvaluationPeriods': 1,
                                            'MetricName': 'Name',
                                            'Namespace': 'Namespace',
                                            'Period.$': '$.CloudWatchPeriod',
                                            'Statistic': 'AVERAGE',
                                            'Threshold': 1,
                                            'Unit': 'NONE',
                                        },
                                    },
                                }],
                        },
                        BidPrice: '1',
                        Configurations: [{
                                Classification: 'Classification',
                                Properties: {
                                    Key: 'Value',
                                },
                            }],
                        EbsConfiguration: {
                            EbsBlockDeviceConfigs: [{
                                    VolumeSpecification: {
                                        Iops: 1,
                                        SizeInGB: 1,
                                        VolumeType: 'standard',
                                    },
                                    VolumesPerInstance: 1,
                                }],
                            EbsOptimized: true,
                        },
                        InstanceCount: 1,
                        InstanceRole: 'MASTER',
                        InstanceType: 'm5.xlarge',
                        Market: 'ON_DEMAND',
                        Name: 'Name',
                    }],
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192',
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A',
            },
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new lib_1.EmrCreateCluster(stack, 'Task', {
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,RUN_JOB. Received: WAIT_FOR_TASK_TOKEN/);
});
//# sourceMappingURL=data:application/json;base64,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