"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const path = require("path");
const tasks = require("../../lib");
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-ecs-integ2', {
    env: {
        account: process.env.CDK_INTEG_ACCOUNT || process.env.CDK_DEFAULT_ACCOUNT,
        region: process.env.CDK_INTEG_REGION || process.env.CDK_DEFAULT_REGION,
    },
});
const vpc = ec2.Vpc.fromLookup(stack, 'Vpc', {
    isDefault: true,
});
const cluster = new ecs.Cluster(stack, 'FargateCluster', { vpc });
// Build task definition
const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef', {
    memoryLimitMiB: 512,
    cpu: 256,
});
const containerDefinition = taskDefinition.addContainer('TheContainer', {
    image: ecs.ContainerImage.fromAsset(path.resolve(__dirname, 'eventhandler-image')),
    memoryLimitMiB: 256,
    logging: new ecs.AwsLogDriver({ streamPrefix: 'EventDemo' }),
});
// Build state machine
const definition = new sfn.Pass(stack, 'Start', {
    result: sfn.Result.fromObject({ SomeKey: 'SomeValue' }),
}).next(new sfn.Task(stack, 'FargateTask', {
    task: new tasks.RunEcsFargateTask({
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        cluster,
        taskDefinition,
        assignPublicIp: true,
        containerOverrides: [
            {
                containerDefinition,
                environment: [
                    {
                        name: 'SOME_KEY',
                        value: sfn.JsonPath.stringAt('$.SomeKey'),
                    },
                ],
            },
        ],
    }),
}));
new sfn.StateMachine(stack, 'StateMachine', {
    definition,
});
app.synth();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuZmFyZ2F0ZS10YXNrLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiaW50ZWcuZmFyZ2F0ZS10YXNrLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4QyxrREFBa0Q7QUFDbEQscUNBQXFDO0FBQ3JDLDZCQUE2QjtBQUM3QixtQ0FBbUM7QUFFbkMsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUM7QUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTtJQUNqRCxHQUFHLEVBQUU7UUFDSCxPQUFPLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsSUFBSSxPQUFPLENBQUMsR0FBRyxDQUFDLG1CQUFtQjtRQUN6RSxNQUFNLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsSUFBSSxPQUFPLENBQUMsR0FBRyxDQUFDLGtCQUFrQjtLQUN2RTtDQUNGLENBQUMsQ0FBQztBQUVILE1BQU0sR0FBRyxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7SUFDM0MsU0FBUyxFQUFFLElBQUk7Q0FDaEIsQ0FBQyxDQUFDO0FBRUgsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7QUFFbEUsd0JBQXdCO0FBQ3hCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7SUFDckUsY0FBYyxFQUFFLEdBQUc7SUFDbkIsR0FBRyxFQUFFLEdBQUc7Q0FDVCxDQUFDLENBQUM7QUFDSCxNQUFNLG1CQUFtQixHQUFHLGNBQWMsQ0FBQyxZQUFZLENBQUMsY0FBYyxFQUFFO0lBQ3RFLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxvQkFBb0IsQ0FBQyxDQUFDO0lBQ2xGLGNBQWMsRUFBRSxHQUFHO0lBQ25CLE9BQU8sRUFBRSxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsRUFBRSxZQUFZLEVBQUUsV0FBVyxFQUFFLENBQUM7Q0FDN0QsQ0FBQyxDQUFDO0FBRUgsc0JBQXNCO0FBQ3RCLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO0lBQzlDLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxXQUFXLEVBQUUsQ0FBQztDQUN4RCxDQUFDLENBQUMsSUFBSSxDQUNMLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFO0lBQ2pDLElBQUksRUFBRSxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUNoQyxrQkFBa0IsRUFBRSxHQUFHLENBQUMseUJBQXlCLENBQUMsSUFBSTtRQUN0RCxPQUFPO1FBQ1AsY0FBYztRQUNkLGNBQWMsRUFBRSxJQUFJO1FBQ3BCLGtCQUFrQixFQUFFO1lBQ2xCO2dCQUNFLG1CQUFtQjtnQkFDbkIsV0FBVyxFQUFFO29CQUNYO3dCQUNFLElBQUksRUFBRSxVQUFVO3dCQUNoQixLQUFLLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDO3FCQUMxQztpQkFDRjthQUNGO1NBQ0Y7S0FDRixDQUFDO0NBQ0gsQ0FBQyxDQUNILENBQUM7QUFFRixJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtJQUMxQyxVQUFVO0NBQ1gsQ0FBQyxDQUFDO0FBRUgsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgZWNzIGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3MnO1xuaW1wb3J0ICogYXMgc2ZuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyB0YXNrcyBmcm9tICcuLi8uLi9saWInO1xuXG5jb25zdCBhcHAgPSBuZXcgY2RrLkFwcCgpO1xuY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKGFwcCwgJ2F3cy1lY3MtaW50ZWcyJywge1xuICBlbnY6IHtcbiAgICBhY2NvdW50OiBwcm9jZXNzLmVudi5DREtfSU5URUdfQUNDT1VOVCB8fCBwcm9jZXNzLmVudi5DREtfREVGQVVMVF9BQ0NPVU5ULFxuICAgIHJlZ2lvbjogcHJvY2Vzcy5lbnYuQ0RLX0lOVEVHX1JFR0lPTiB8fCBwcm9jZXNzLmVudi5DREtfREVGQVVMVF9SRUdJT04sXG4gIH0sXG59KTtcblxuY29uc3QgdnBjID0gZWMyLlZwYy5mcm9tTG9va3VwKHN0YWNrLCAnVnBjJywge1xuICBpc0RlZmF1bHQ6IHRydWUsXG59KTtcblxuY29uc3QgY2x1c3RlciA9IG5ldyBlY3MuQ2x1c3RlcihzdGFjaywgJ0ZhcmdhdGVDbHVzdGVyJywgeyB2cGMgfSk7XG5cbi8vIEJ1aWxkIHRhc2sgZGVmaW5pdGlvblxuY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkZhcmdhdGVUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnLCB7XG4gIG1lbW9yeUxpbWl0TWlCOiA1MTIsXG4gIGNwdTogMjU2LFxufSk7XG5jb25zdCBjb250YWluZXJEZWZpbml0aW9uID0gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdUaGVDb250YWluZXInLCB7XG4gIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbUFzc2V0KHBhdGgucmVzb2x2ZShfX2Rpcm5hbWUsICdldmVudGhhbmRsZXItaW1hZ2UnKSksXG4gIG1lbW9yeUxpbWl0TWlCOiAyNTYsXG4gIGxvZ2dpbmc6IG5ldyBlY3MuQXdzTG9nRHJpdmVyKHsgc3RyZWFtUHJlZml4OiAnRXZlbnREZW1vJyB9KSxcbn0pO1xuXG4vLyBCdWlsZCBzdGF0ZSBtYWNoaW5lXG5jb25zdCBkZWZpbml0aW9uID0gbmV3IHNmbi5QYXNzKHN0YWNrLCAnU3RhcnQnLCB7XG4gIHJlc3VsdDogc2ZuLlJlc3VsdC5mcm9tT2JqZWN0KHsgU29tZUtleTogJ1NvbWVWYWx1ZScgfSksXG59KS5uZXh0KFxuICBuZXcgc2ZuLlRhc2soc3RhY2ssICdGYXJnYXRlVGFzaycsIHtcbiAgICB0YXNrOiBuZXcgdGFza3MuUnVuRWNzRmFyZ2F0ZVRhc2soe1xuICAgICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5TWU5DLFxuICAgICAgY2x1c3RlcixcbiAgICAgIHRhc2tEZWZpbml0aW9uLFxuICAgICAgYXNzaWduUHVibGljSXA6IHRydWUsXG4gICAgICBjb250YWluZXJPdmVycmlkZXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIGNvbnRhaW5lckRlZmluaXRpb24sXG4gICAgICAgICAgZW52aXJvbm1lbnQ6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgbmFtZTogJ1NPTUVfS0VZJyxcbiAgICAgICAgICAgICAgdmFsdWU6IHNmbi5Kc29uUGF0aC5zdHJpbmdBdCgnJC5Tb21lS2V5JyksXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pLFxuICB9KSxcbik7XG5cbm5ldyBzZm4uU3RhdGVNYWNoaW5lKHN0YWNrLCAnU3RhdGVNYWNoaW5lJywge1xuICBkZWZpbml0aW9uLFxufSk7XG5cbmFwcC5zeW50aCgpO1xuIl19