"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RunBatchJob = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * A Step Functions Task to run AWS Batch
 *
 * @deprecated use `BatchSubmitJob`
 */
class RunBatchJob {
    constructor(props) {
        var _a;
        this.props = props;
        // validate integrationPattern
        this.integrationPattern =
            props.integrationPattern || sfn.ServiceIntegrationPattern.SYNC;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call RunBatchJob.`);
        }
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`timeout must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is reuqired since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_a = props.containerOverrides) === null || _a === void 0 ? void 0 : _a.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('batch', 'submitJob', this.integrationPattern),
            policyStatements: this.configurePolicyStatements(_task),
            parameters: {
                JobDefinition: this.props.jobDefinition.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueue.jobQueueArn,
                Parameters: this.props.payload,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type,
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined,
            },
        };
    }
    configurePolicyStatements(task) {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(task).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*',
                    }),
                    this.props.jobQueue.jobQueueArn,
                ],
                actions: ['batch:SubmitJob'],
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(task).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule',
                    }),
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            }),
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _a;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value,
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`,
                },
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_a = containerOverrides.instanceType) === null || _a === void 0 ? void 0 : _a.toString(),
            Memory: containerOverrides.memory,
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus,
        };
    }
}
exports.RunBatchJob = RunBatchJob;
//# sourceMappingURL=data:application/json;base64,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