"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool Client', () => {
    test('custom domain name', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        const certificate = aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8');
        new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            customDomain: {
                domainName: 'test-domain.example.com',
                certificate,
            },
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
            UserPoolId: stack.resolve(pool.userPoolId),
            Domain: 'test-domain.example.com',
            CustomDomainConfig: {
                CertificateArn: 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8',
            },
        });
    });
    test('cognito domain prefix', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
            UserPoolId: stack.resolve(pool.userPoolId),
            Domain: 'cognito-domain-prefix',
        });
    });
    test('fails when neither cognitoDomain nor customDomain are specified', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const certificate = aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8');
        expect(() => new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
            customDomain: {
                domainName: 'mydomain.com',
                certificate,
            },
        })).toThrow(/cognitoDomain or customDomain must be specified/);
    });
    test('fails when both cognitoDomain and customDomain are specified', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
        })).toThrow(/cognitoDomain or customDomain must be specified/);
    });
    test('fails when domainPrefix has characters outside the allowed charset', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addDomain('Domain1', {
            cognitoDomain: { domainPrefix: 'domain.prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
        expect(() => pool.addDomain('Domain2', {
            cognitoDomain: { domainPrefix: 'Domain-Prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
        expect(() => pool.addDomain('Domain3', {
            cognitoDomain: { domainPrefix: 'dómäin-prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
    });
    test('custom resource is added when cloudFrontDistribution method is called', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const domain = pool.addDomain('Domain', {
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
        });
        // WHEN
        const cfDomainName = domain.cloudFrontDomainName;
        // THEN
        expect(stack.resolve(cfDomainName)).toEqual({
            'Fn::GetAtt': [
                'PoolDomainCloudFrontDomainName340BF87E',
                'DomainDescription.CloudFrontDistribution',
            ],
        });
        expect(stack).toHaveResource('Custom::UserPoolCloudFrontDomainName');
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [{
                        Action: 'cognito-idp:DescribeUserPoolDomain',
                        Effect: 'Allow',
                        Resource: '*',
                    }],
                Version: '2012-10-17',
            },
        });
    });
    describe('signInUrl', () => {
        test('returns the expected URL', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'Pool');
            const domain = pool.addDomain('Domain', {
                cognitoDomain: {
                    domainPrefix: 'cognito-domain-prefix',
                },
            });
            const client = pool.addClient('Client', {
                oAuth: {
                    callbackUrls: ['https://example.com'],
                },
            });
            // WHEN
            const signInUrl = domain.signInUrl(client, {
                redirectUri: 'https://example.com',
            });
            // THEN
            expect(stack.resolve(signInUrl)).toEqual({
                'Fn::Join': [
                    '', [
                        'https://',
                        { Ref: 'PoolDomainCFC71F56' },
                        '.auth.',
                        { Ref: 'AWS::Region' },
                        '.amazoncognito.com/login?client_id=',
                        { Ref: 'PoolClient8A3E5EB7' },
                        '&response_type=code&redirect_uri=https://example.com',
                    ],
                ],
            });
        });
        test('correctly uses the signInPath', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'Pool');
            const domain = pool.addDomain('Domain', {
                cognitoDomain: {
                    domainPrefix: 'cognito-domain-prefix',
                },
            });
            const client = pool.addClient('Client', {
                oAuth: {
                    callbackUrls: ['https://example.com'],
                },
            });
            // WHEN
            const signInUrl = domain.signInUrl(client, {
                redirectUri: 'https://example.com',
                signInPath: '/testsignin',
            });
            // THEN
            expect(signInUrl).toMatch(/amazoncognito\.com\/testsignin\?/);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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