"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool Client', () => {
    test('default setup', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            AllowedOAuthFlows: ['implicit', 'code'],
            AllowedOAuthScopes: ['profile', 'phone', 'email', 'openid', 'aws.cognito.signin.user.admin'],
            CallbackURLs: ['https://example.com'],
            SupportedIdentityProviders: ['COGNITO'],
        });
    });
    test('client name', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        const client1 = new lib_1.UserPoolClient(stack, 'Client1', {
            userPool: pool,
            userPoolClientName: 'myclient',
        });
        const client2 = new lib_1.UserPoolClient(stack, 'Client2', {
            userPool: pool,
        });
        // THEN
        expect(client1.userPoolClientName).toEqual('myclient');
        expect(() => client2.userPoolClientName).toThrow(/available only if specified on the UserPoolClient during initialization/);
    });
    test('import', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const client = lib_1.UserPoolClient.fromUserPoolClientId(stack, 'Client', 'client-id-1');
        // THEN
        expect(client.userPoolClientId).toEqual('client-id-1');
    });
    test('ExplicitAuthFlows is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ExplicitAuthFlows: assert_1.ABSENT,
        });
    });
    test('ExplicitAuthFlows are correctly named', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client', {
            authFlows: {
                adminUserPassword: true,
                custom: true,
                refreshToken: true,
                userPassword: true,
                userSrp: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ExplicitAuthFlows: [
                'ALLOW_USER_PASSWORD_AUTH',
                'ALLOW_ADMIN_USER_PASSWORD_AUTH',
                'ALLOW_CUSTOM_AUTH',
                'ALLOW_USER_SRP_AUTH',
                'ALLOW_REFRESH_TOKEN_AUTH',
            ],
        });
    });
    test('AllowedOAuthFlows are correctly named', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client1', {
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                    implicitCodeGrant: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        pool.addClient('Client2', {
            oAuth: {
                flows: {
                    clientCredentials: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['implicit', 'code'],
            AllowedOAuthFlowsUserPoolClient: true,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['client_credentials'],
            AllowedOAuthFlowsUserPoolClient: true,
        });
    });
    test('callbackUrl defaults are correctly chosen', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        pool.addClient('Client1', {
            oAuth: {
                flows: {
                    clientCredentials: true,
                },
            },
        });
        pool.addClient('Client2', {
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
            },
        });
        pool.addClient('Client3', {
            oAuth: {
                flows: {
                    implicitCodeGrant: true,
                },
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['client_credentials'],
            CallbackURLs: assert_1.ABSENT,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['implicit'],
            CallbackURLs: ['https://example.com'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['code'],
            CallbackURLs: ['https://example.com'],
        });
    });
    test('fails when callbackUrls is empty for codeGrant or implicitGrant', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addClient('Client1', {
            oAuth: {
                flows: { implicitCodeGrant: true },
                callbackUrls: [],
            },
        })).toThrow(/callbackUrl must not be empty/);
        expect(() => pool.addClient('Client3', {
            oAuth: {
                flows: { authorizationCodeGrant: true },
                callbackUrls: [],
            },
        })).toThrow(/callbackUrl must not be empty/);
        expect(() => pool.addClient('Client4', {
            oAuth: {
                flows: { clientCredentials: true },
                callbackUrls: [],
            },
        })).not.toThrow();
    });
    test('fails when clientCredentials OAuth flow is selected along with codeGrant or implicitGrant', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addClient('Client1', {
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                    clientCredentials: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        })).toThrow(/clientCredentials OAuth flow cannot be selected/);
        expect(() => pool.addClient('Client2', {
            oAuth: {
                flows: {
                    implicitCodeGrant: true,
                    clientCredentials: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        })).toThrow(/clientCredentials OAuth flow cannot be selected/);
    });
    test('OAuth scopes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client', {
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [
                    lib_1.OAuthScope.PHONE,
                    lib_1.OAuthScope.EMAIL,
                    lib_1.OAuthScope.OPENID,
                    lib_1.OAuthScope.PROFILE,
                    lib_1.OAuthScope.COGNITO_ADMIN,
                    lib_1.OAuthScope.custom('my-resource-server/my-own-scope'),
                ],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthScopes: [
                'phone',
                'email',
                'openid',
                'profile',
                'aws.cognito.signin.user.admin',
                'my-resource-server/my-own-scope',
            ],
        });
    });
    test('OAuthScope - openid is included when email or phone is specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client1', {
            userPoolClientName: 'Client1',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        pool.addClient('Client2', {
            userPoolClientName: 'Client2',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.EMAIL],
            },
        });
        pool.addClient('Client3', {
            userPoolClientName: 'Client3',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.PROFILE],
            },
        });
        pool.addClient('Client4', {
            userPoolClientName: 'Client4',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.COGNITO_ADMIN],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client1',
            AllowedOAuthScopes: ['phone', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client2',
            AllowedOAuthScopes: ['email', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client3',
            AllowedOAuthScopes: ['profile', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client4',
            AllowedOAuthScopes: ['aws.cognito.signin.user.admin'],
        });
    });
    test('enable user existence errors prevention', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
            preventUserExistenceErrors: true,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: 'ENABLED',
        });
    });
    test('disable user existence errors prevention', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
            preventUserExistenceErrors: false,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: 'LEGACY',
        });
    });
    test('user existence errors prevention is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: assert_1.ABSENT,
        });
    });
    test('default supportedIdentityProviders', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const idp = lib_1.UserPoolIdentityProvider.fromProviderName(stack, 'imported', 'userpool-idp');
        pool.registerIdentityProvider(idp);
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            SupportedIdentityProviders: [
                'userpool-idp',
                'COGNITO',
            ],
        });
    });
    test('supportedIdentityProviders', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('AllEnabled', {
            userPoolClientName: 'AllEnabled',
            supportedIdentityProviders: [
                lib_1.UserPoolClientIdentityProvider.COGNITO,
                lib_1.UserPoolClientIdentityProvider.FACEBOOK,
                lib_1.UserPoolClientIdentityProvider.AMAZON,
            ],
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            ClientName: 'AllEnabled',
            SupportedIdentityProviders: ['COGNITO', 'Facebook', 'LoginWithAmazon'],
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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