"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const cognito_generated_1 = require("./cognito.generated");
/**
 * Define a user pool domain
 */
class UserPoolDomain extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        if (!!props.customDomain === !!props.cognitoDomain) {
            throw new Error('One of, and only one of, cognitoDomain or customDomain must be specified');
        }
        if (((_a = props.cognitoDomain) === null || _a === void 0 ? void 0 : _a.domainPrefix) && !/^[a-z0-9-]+$/.test(props.cognitoDomain.domainPrefix)) {
            throw new Error('domainPrefix for cognitoDomain can contain only lowercase alphabets, numbers and hyphens');
        }
        this.isCognitoDomain = !!props.cognitoDomain;
        const domainName = ((_b = props.cognitoDomain) === null || _b === void 0 ? void 0 : _b.domainPrefix) || ((_c = props.customDomain) === null || _c === void 0 ? void 0 : _c.domainName);
        const resource = new cognito_generated_1.CfnUserPoolDomain(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            domain: domainName,
            customDomainConfig: props.customDomain ? { certificateArn: props.customDomain.certificate.certificateArn } : undefined,
        });
        this.domainName = resource.ref;
    }
    /**
     * The domain name of the CloudFront distribution associated with the user pool domain.
     */
    get cloudFrontDomainName() {
        const sdkCall = {
            service: 'CognitoIdentityServiceProvider',
            action: 'describeUserPoolDomain',
            parameters: {
                Domain: this.domainName,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.domainName),
        };
        const customResource = new custom_resources_1.AwsCustomResource(this, 'CloudFrontDomainName', {
            resourceType: 'Custom::UserPoolCloudFrontDomainName',
            onCreate: sdkCall,
            onUpdate: sdkCall,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                // DescribeUserPoolDomain only supports access level '*'
                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html#amazoncognitouserpools-actions-as-permissions
                resources: ['*'],
            }),
        });
        return customResource.getResponseField('DomainDescription.CloudFrontDistribution');
    }
    /**
     * The URL to the hosted UI associated with this domain
     */
    baseUrl() {
        if (this.isCognitoDomain) {
            return `https://${this.domainName}.auth.${core_1.Stack.of(this).region}.amazoncognito.com`;
        }
        return `https://${this.domainName}`;
    }
    /**
     * The URL to the sign in page in this domain using a specific UserPoolClient
     * @param client [disable-awslint:ref-via-interface] the user pool client that the UI will use to interact with the UserPool
     * @param options options to customize the behaviour of this method.
     */
    signInUrl(client, options) {
        var _a;
        let responseType;
        if (client.oAuthFlows.authorizationCodeGrant) {
            responseType = 'code';
        }
        else if (client.oAuthFlows.implicitCodeGrant) {
            responseType = 'token';
        }
        else {
            throw new Error('signInUrl is not supported for clients without authorizationCodeGrant or implicitCodeGrant flow enabled');
        }
        const path = (_a = options.signInPath) !== null && _a !== void 0 ? _a : '/login';
        return `${this.baseUrl()}${path}?client_id=${client.userPoolClientId}&response_type=${responseType}&redirect_uri=${options.redirectUri}`;
    }
}
exports.UserPoolDomain = UserPoolDomain;
//# sourceMappingURL=data:application/json;base64,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