"""
This module analyzes and manipulates macrostate probability distributions generated by FEASST
flat-histogram simulations.
See more information on FEASST at https://doi.org/10.18434/M3S095

"""

import copy
from pathlib import Path
import numpy as np
import matplotlib.pyplot as plt
import pandas as pd
from scipy.optimize import minimize_scalar
from scipy.optimize import minimize

class MacrostateDistribution:
    """
    Contains macrostates, the natural logarithm of the macrostate probability and other
    per-macrostate quantities generated by a FEASST flat histogram simulation.
    This may include canonical ensemble averages and blocks.
    """
    def __init__(self, file_name, macrostate_header='state', ln_prob_header='ln_prob', comment="#"):
        """
        Constructs all the necessary attributes for a MacrostateDistribution.

        :param str file_name
            The name of the csv file which stores per-macrostate quantities.
        :param str macrostate_header:
            In the grand canonical ensemble, the macrostates are the number of particles.
            This is the header for that column in a pandas data frame.
        :param str ln_prob_header:
            The natural logarithm of the probability of observing each macrostate.

        >>> from pyfeasst import macrostate_distribution
        >>> import pandas as pd
        >>> lnpi = macrostate_distribution.MacrostateDistribution(file_name='../../tests/lnpi.csv')
        """
        self._macrostate_header = macrostate_header
        self._ln_prob_header = ln_prob_header
        self.set_dataframe(pd.read_csv(file_name, comment=comment))
        self.set_minimum_smoothing()

    def set_dataframe(self, dataframe, normalize=True):
        """
        Set the dataframe.

        :param bool normalize: normalize the ln_prob after setting the dataframe.
        """
        self._dataframe = dataframe.copy()
        if normalize:
            self.normalize()

    def concat_dataframe(self, dataframe, add_prefix):
        """Concatenate dataframe (e.g., for average energy per macrostate)."""
        df2 = dataframe.add_prefix(add_prefix)
        self.set_dataframe(pd.concat([self._dataframe, df2], axis=1))

    def dataframe(self):
        """Return the dataframe."""
        return self._dataframe

    def set_macrostates(self, macrostates):
        """Set the macrostates in the dataframe."""
        self._dataframe[self._macrostate_header] = macrostates

    def macrostates(self):
        """Return the macrostates."""
        return self._dataframe[self._macrostate_header]

    def set_ln_prob(self, ln_prob):
        """Set the natural logarithm of the probability."""
        self._dataframe[self._ln_prob_header] = ln_prob

    def ln_prob(self):
        """Return the natural logarithm of the probability."""
        return self._dataframe[self._ln_prob_header]

    def set_minimum_smoothing(self, minimum_smoothing=10):
        """
        During the search for the minimum, the number of macrostates that the minimum must be
        smaller than in both directions.
        Increase this if the data is noisey or if there are undulations in the ln_prob.
        """
        self._minimum_smoothing = minimum_smoothing

    def minimum_smoothing(self):
        '''Return the minimum smoothing.'''
        return self._minimum_smoothing

    def normalize(self, *args):
        """
        Set the sum of the probabilities of a series to one.

        If no arguments provided, use the internal ln_prob.
        If ln_prob is provided as a series, return the normalized series.
        """
        if len(args) == 0:
            self._dataframe[self._ln_prob_header] = self.normalize(self.ln_prob())
            lnpi = self.ln_prob()
        elif len(args) == 1:
            lnpi = args[0]
            lnpi -= np.log(sum(np.exp(args[0])))
        else:
            assert False # unrecognized number of arguments.
        return lnpi

    def average_macrostate(self):
        """
        Return the average macrostate.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.MacrostateDistribution(file_name='../../tests/lnpi.csv')
        >>> lnpi.average_macrostate()
        437.6820787231706
        """
        return (np.exp(self.ln_prob()) * self.macrostates()).sum()

    def ensemble_average(self, header):
        """
        Return the grand canonical ensemble average of the data present in header,
        for example, canonical averages.

        >>> import pandas as pd
        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.MacrostateDistribution(file_name='../../tests/lnpi.csv')
        >>> lnpi.ensemble_average(header='energy')
        -2701.3930043481596
        """
        return (np.exp(self.ln_prob()) * self._dataframe[header]).sum()

    def plot(self, show=False):
        '''Create a matplotlib.pyplot plot of the ln_prob.'''
        plt.plot(self.macrostates(), self.ln_prob())
        plt.xlabel('macrostate')
        plt.ylabel(r'$\ln\Pi$')
        if show:
            plt.show()

    def minimums(self):
        """
        Return the minimums of the ln_prob.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.MacrostateDistribution(file_name='../../tests/lnpi.csv')
        >>> mins = lnpi.minimums()
        >>> len(mins)
        1
        >>> mins.values[0]
        150
        """
        lnp = self.ln_prob()
        mns = lnp[(lnp.shift(1) > lnp) & (lnp.shift(-1) > lnp)]
        for shift in range(2, self._minimum_smoothing+1):
            mns = lnp[mns.notna() &
                      (lnp.shift(shift) > lnp) &
                      (lnp.shift(-shift) > lnp)]
        return self.macrostates()[mns.notna().index]

    def split(self, minimum=-1):
        """
        Return two MacrostateDistribution by splitting at the last minimum in the ln_prob.
        Both MacrostateDistribution will contain the minimum.

        :param int minimum: If -1, find the last minimum. Otherwise, supply it.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.MacrostateDistribution(file_name='../../tests/lnpi.csv')
        >>> vapor, liquid = lnpi.split()
        >>> vapor.average_macrostate()
        1.4212647041954634
        >>> vapor.ensemble_average('energy')
        -0.04911666622772193
        >>> liquid.average_macrostate()
        437.6820787231706
        >>> liquid.ensemble_average('energy')
        -2701.39300434816
        """
        if minimum == -1:
            minimum = self.minimums().values[-1]
        lower = copy.deepcopy(self)
        upper = copy.deepcopy(self)
        lower.set_dataframe(self._dataframe[:minimum].copy())
        lower.normalize()
        upper.set_dataframe(self._dataframe[minimum:].copy())
        upper.normalize()
        return [lower, upper]

    def reweight(self, delta_beta_mu, inplace=False):
        """
        Reweight the ln_prob by delta_beta_mu.
        If inplace, replace current ln_prob with the reweighted ones.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.MacrostateDistribution(file_name='../../tests/lnpi.csv')
        >>> reweight = lnpi.reweight(-1)
        >>> reweight.dataframe()['ln_prob'][0]
        -0.5010868690523108
        >>> reweight = lnpi.reweight(1.5)
        >>> reweight.dataframe()['ln_prob'][0]
        -811.7401044444931
        """
        # avoid negative log by subtracting min
        lnpi_rw = self.ln_prob() + self.macrostates()*delta_beta_mu - self.ln_prob().min()
        if lnpi_rw.max() > 0:
            lnpi_rw -= lnpi_rw.max()
        lnpi_rw = self.normalize(lnpi_rw)
        if inplace:
            self.set_ln_prob(lnpi_rw)
        self_rw = copy.deepcopy(self)
        self_rw.set_ln_prob(lnpi_rw)
        return self_rw

    def _macrostate_objective(self, target_macrostate, delta_beta_mu):
        """
        Return an object function to minimize in order to find delta_beta_mu that,
        upon reweighting, results in an average macrostate equal to target_macrostate.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.MacrostateDistribution(file_name='../../tests/lnpi.csv')
        >>> lnpi._macrostate_objective(target_macrostate=10, delta_beta_mu=-1)
        90.14131550285451
        """
        lnpi_rw = self.reweight(delta_beta_mu)
        return (target_macrostate - lnpi_rw.average_macrostate())**2

    def reweight_to_macrostate(self, target_macrostate, delta_beta_mu_guess=0, tol=1e-6):
        """
        Reweight ln_prob and return the delta_beta_mu for an average target_macrostate.

        :param float delta_beta_mu_guess: Provide a guess for delta_beta_mu which brings the ln_prob
            closer to equilibrium.
        :param float tol: Tolerance of minimization algorithm.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.MacrostateDistribution(file_name='../../tests/lnpi.csv')
        >>> lnpi.reweight_to_macrostate(target_macrostate=10)
        -0.3221837585941306
        >>> round(lnpi.average_macrostate(), 5)
        10.0
        """
        res = minimize(lambda delta_beta_mu: self._macrostate_objective(
            target_macrostate=target_macrostate,
            delta_beta_mu=delta_beta_mu[0]), delta_beta_mu_guess, tol=tol)
        delta_beta_mu_final = res.x[0]
        self.reweight(delta_beta_mu_final, inplace=True)
        return delta_beta_mu_final

    def _equilibrium_objective(self, delta_beta_mu):
        """
        Return an objective function to minimize in order to find equilibrium.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.MacrostateDistribution(file_name='../../tests/lnpi.csv')
        >>> lnpi._equilibrium_objective(-1)
        3.818281828459045
        """
        #print('delta_beta_mu', delta_beta_mu)
        lnpi_rw = self.reweight(delta_beta_mu)
        mins = lnpi_rw.minimums()
        #print('num mins', len(mins))
        if len(mins) == 0:
            return_val = 2.+(lnpi_rw.ln_prob()[0] - lnpi_rw.ln_prob().values[-1])**2
        elif len(mins) == 1:
            mns = mins.values[0]
            p_lo = np.exp(lnpi_rw.ln_prob()[:mns]).sum()
            p_hi = np.exp(lnpi_rw.ln_prob()[mns+1:]).sum()
            # if one side is greatly favored over the other, numerical pecision
            # becomes an issue. Instead, nudge toward conjugate that alters this disaparity
            if abs(p_lo - 1) < 1e-6:
                return_val = 1.1+np.exp(-delta_beta_mu)
            elif abs(p_hi - 1) < 1e-6:
                return_val = 1.1+np.exp(delta_beta_mu)
            else:
                return_val = abs(p_lo-p_hi)
        else:
            #lnpi_rw.plot()
            #plt.show()
            assert False # more than one minima
        return return_val

    def equilibrium(self, delta_beta_mu_guess=-0.1, tol=1e-6):
        """
        Reweight ln_prob to equilibrium and return the delta_beta_mu of the equilibrium condition.

        :param float delta_beta_mu_guess: Provide a guess for delta_beta_mu which brings the ln_prob
            closer to equilibrium.
        :param float tol: Tolerance of minimization algorithm.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.MacrostateDistribution(file_name='../../tests/lnpi.csv')
        >>> lnpi.equilibrium(delta_beta_mu_guess=-1, tol=1e-4)
        -0.31323799472042363
        >>> vapor, liquid = lnpi.split()
        >>> vapor.average_macrostate()/8**3
        0.00200034710256861
        >>> liquid.average_macrostate()/8**3
        0.8430480261823026
        """
        res = minimize_scalar(self._equilibrium_objective,
                              tol=tol,
                              bracket=(0, delta_beta_mu_guess))
        delta_beta_mu_equilibrium = res["x"]
        self.reweight(delta_beta_mu_equilibrium, inplace=True)
        return delta_beta_mu_equilibrium

def splice(windows, extra_overlap=0):
    """
    Return a MacrostateDistribution that is spliced together from multiple MacrostateDistribution.
    Each MacrostateDistribution must overlap by atleast one macrostate with its neighbor.
    If macrostates overlap by more than one (extra_overlap), simply drop the
    macrostates in the upper MacrostateDistribution.

    >>> from pyfeasst import macrostate_distribution
    >>> lnpis=list()
    >>> for i in range(4):
    ...     lnpis.append(macrostate_distribution.MacrostateDistribution(
    ...         file_name='../../tests/lnpin'+str(i)+'.csv'))
    >>> lnpi = macrostate_distribution.splice(lnpis)
    >>> lnpi.set_minimum_smoothing(50)
    >>> lnpi.average_macrostate()
    596.292899018217
    >>> len(lnpi.minimums())
    0
    >>> lnpi.equilibrium(delta_beta_mu_guess=-1.5)
    -1.2768125968551267
    >>> vapor, liquid = lnpi.split()
    >>> vapor.average_macrostate()
    0.0317208114759036
    >>> liquid.average_macrostate()
    523.3553081498679
    """
    macros = list()
    max_minimum_smoothing = 0
    for index, lnpi in enumerate(windows):
        if max_minimum_smoothing < lnpi.minimum_smoothing():
            max_minimum_smoothing = lnpi.minimum_smoothing()
        if index != 0:
            prev_last_index = len(windows[index-1].ln_prob()) - 1
            shift = windows[index-1].ln_prob().values[prev_last_index-extra_overlap] - \
                    lnpi.ln_prob().values[0]
            lnpi.set_ln_prob(lnpi.ln_prob() + shift)
            for _ in range(1 + extra_overlap):
                lnpi.set_dataframe(lnpi.dataframe().iloc[1:, :], normalize=False)
            lnpi.set_macrostates(lnpi.macrostates() + windows[index-1].macrostates().values[-1])
        macros.append(lnpi.dataframe())
    macros_concat = pd.concat(macros)
    macros_concat.reset_index(inplace=True)
    combined = copy.deepcopy(windows[0])
    combined.set_dataframe(macros_concat.copy())
    combined.set_minimum_smoothing(max_minimum_smoothing)
    return combined

def splice_files(prefix, suffix, ln_prob_header=None, extra_overlap=0):
    """
    As above, Return a MacrostateDistribution that is spliced together.
    But this time, provide the prefix and suffix of the filenames and search the files.

    >>> from pyfeasst import macrostate_distribution
    >>> lnpi = macrostate_distribution.splice_files(prefix='../../tests/lj_lnpin', suffix='.txt')
    >>> lnpi.equilibrium()
    -0.3140241084584985
    >>> vapor, liquid = lnpi.split()
    >>> vapor.average_macrostate()/8**3
    0.001995010383395003
    >>> liquid.average_macrostate()/8**3
    0.8431834550386511
    """
    lnpis = list()
    for filename in sorted(Path('.').rglob(prefix+'*'+suffix)):
        if ln_prob_header is None:
            lnpis.append(MacrostateDistribution(file_name=filename))
        else:
            lnpis.append(MacrostateDistribution(file_name=filename, ln_prob_header=ln_prob_header))
    return splice(lnpis, extra_overlap)

if __name__ == "__main__":
    import doctest
    doctest.testmod()
