# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['autocensus']

package_data = \
{'': ['*'], 'autocensus': ['resources/*']}

install_requires = \
['Fiona>=1.8.20,<2.0.0',
 'Shapely>=1.8.0,<2.0.0',
 'appdirs>=1.4.4,<2.0.0',
 'geopandas>=0.10.2,<0.11.0',
 'httpx>=0.20.0,<0.21.0',
 'nest-asyncio>=1.5.1,<2.0.0',
 'pandas>=1.3.4,<2.0.0',
 'socrata-py>=1.0.10,<2.0.0',
 'tenacity>=8.0.1,<9.0.0',
 'typing-extensions>=3.10.0,<4.0.0',
 'us>=2.0.2,<3.0.0',
 'yarl>=1.7.2,<2.0.0']

extras_require = \
{':python_version < "3.8"': ['importlib_metadata>=4.2.0,<5.0.0']}

setup_kwargs = {
    'name': 'autocensus',
    'version': '2.0.5',
    'description': 'A tool for collecting ACS and geospatial data from the Census API',
    'long_description': "# autocensus\n\nA Python package for collecting American Community Survey (ACS) data and associated geometry from the [Census API] in a [pandas] dataframe.\n\n[Census API]: https://www.census.gov/developers\n[pandas]: https://pandas.pydata.org\n\n## Contents\n\n* [Installation](#installation)\n* [Quickstart](#quickstart)\n* [Geometry](#geometry)\n  + [Points](#points)\n  + [Polygons](#polygons)\n    - [Shapefile resolution](#shapefile-resolution)\n    - [Shapefile caching](#shapefile-caching)\n* [Publishing to Socrata](#publishing-to-socrata)\n  + [Credentials](#credentials)\n  + [Example: Create a new dataset](#example-create-a-new-dataset)\n  + [Example: Replace rows in an existing dataset](#example-replace-rows-in-an-existing-dataset)\n  + [Example: Create a new dataset from multiple queries](#example-create-a-new-dataset-from-multiple-queries)\n* [Troubleshooting](#troubleshooting)\n  + [Clearing the cache](#clearing-the-cache)\n\n## Installation\n\nautocensus requires Python 3.7 or higher. Install as follows:\n\n```sh\npip install autocensus\n```\n\nTo run autocensus, you must specify a [Census API key] via either the `census_api_key` keyword argument (as shown in the example below) or by setting the environment variable `CENSUS_API_KEY`.\n\n## Quickstart\n\n```python\nfrom autocensus import Query\n\n# Configure query\nquery = Query(\n    estimate=1,\n    years=[2017, 2018],\n    variables=['DP03_0025E', 'S0103_C01_104E'],\n    for_geo='county:033',\n    in_geo=['state:53'],\n    # Optional arg to add geometry: 'points', 'polygons', or None (default)\n    geometry='points',\n    # Fill in the following with your actual Census API key\n    census_api_key='Your Census API key'\n)\n\n# Run query and collect output in dataframe\ndataframe = query.run()\n```\n\nOutput:\n\n| name                    | geo_id         | geo_type | year | date       | variable_code  | variable_label                                                                             | variable_concept                                  | annotation |  value | geometry  |\n|:------------------------|:---------------|:---------|-----:|:-----------|:---------------|:-------------------------------------------------------------------------------------------|:--------------------------------------------------|-----------:|-------:|:----------|\n| King County, Washington | 0500000US53033 | county   | 2017 | 2017-12-31 | DP03_0025E     | Estimate!!COMMUTING TO WORK!!Mean travel time to work (minutes)                            | SELECTED ECONOMIC CHARACTERISTICS                 |            |   30.0 | POINT (…) |\n| King County, Washington | 0500000US53033 | county   | 2018 | 2018-12-31 | DP03_0025E     | Estimate!!COMMUTING TO WORK!!Workers 16 years and over!!Mean travel time to work (minutes) | SELECTED ECONOMIC CHARACTERISTICS                 |            |   30.2 | POINT (…) |\n| King County, Washington | 0500000US53033 | county   | 2017 | 2017-12-31 | S0103_C01_104E | Total!!Estimate!!GROSS RENT!!Median gross rent (dollars)                                   | POPULATION 65 YEARS AND OVER IN THE UNITED STATES |            | 1555.0 | POINT (…) |\n| King County, Washington | 0500000US53033 | county   | 2018 | 2018-12-31 | S0103_C01_104E | Estimate!!Total!!Renter-occupied housing units!!GROSS RENT!!Median gross rent (dollars)    | POPULATION 65 YEARS AND OVER IN THE UNITED STATES |            | 1674.0 | POINT (…) |\n\n[Census API key]: https://api.census.gov/data/key_signup.html\n\n## Geometry\n\nautocensus supports point- and polygon-based geometry data for many years and geographies by way of the Census Bureau's [Gazetteer Files] and [Cartographic Boundary Files].\n\nHere's how to add geometry to your data:\n\n[Gazetteer Files]: https://www.census.gov/geographies/reference-files/time-series/geo/gazetteer-files.html\n[Cartographic Boundary Files]: https://www.census.gov/geographies/mapping-files/time-series/geo/carto-boundary-file.html\n\n### Points\n\nPoint data from the Census Bureau's Gazetteer Files is generally available for years from 2012 on in the following geographies:\n\n* Nation-level\n  + `urban area`\n  + `zip code tabulation area`\n  + `county`\n  + `congressional district`\n  + `metropolitan statistical area/micropolitan statistical area`\n  + `american indian area/alaska native area/hawaiian home land`\n* State-level\n  + `county subdivision`\n  + `tract`\n  + `place`\n  + `state legislative district (upper chamber)`\n  + `state legislative district (lower chamber)`\n\nExample:\n\n```python\nfrom autocensus import Query\n\nquery = Query(\n    estimate=5,\n    years=[2018],\n    variables=['DP03_0025E'],\n    for_geo=['county:033'],\n    in_geo=['state:53'],\n    geometry='points'\n)\ndataframe = query.run()\n```\n\n### Polygons\n\nPolygon data from the Census Bureau's Cartographic Boundary Shapefiles is generally available for years from 2013 on in the following geographies:\n\n* Nation-level\n  + `nation`\n  + `region`\n  + `division`\n  + `state`\n  + `urban area`\n  + `zip code tabulation area`\n  + `county`\n  + `congressional district`\n  + `metropolitan statistical area/micropolitan statistical area`\n  + `combined statistical area`\n  + `american indian area/alaska native area/hawaiian home land`\n  + `new england city and town area`\n* State-level\n  + `alaska native regional corporation`\n  + `block group`\n  + `county subdivision`\n  + `tract`\n  + `place`\n  + `public use microdata area`\n  + `state legislative district (upper chamber)`\n  + `state legislative district (lower chamber)`\n\nExample:\n\n```python\nfrom autocensus import Query\n\nquery = Query(\n    estimate=5,\n    years=[2018],\n    variables=['DP03_0025E'],\n    for_geo=['county:033'],\n    in_geo=['state:53'],\n    geometry='polygons'\n)\ndataframe = query.run()\n```\n\n#### Shapefile resolution\n\nBy default, autocensus will attempt to fetch almost all shapefiles at a resolution of 1 : 500,000 (`500k`). Some sources among the Cartographic Boundary Shapefiles are also available at the lower resolutions of 1 : 5,000,000 (`5m`) or 1 : 20,000,000 (`20m`). To attempt to download a shapefile at a specific resolution, pass a value to `Query`'s optional `resolution` parameter:\n\n```python\nfrom autocensus import Query\n\nquery = Query(\n    estimate=5,\n    years=[2018],\n    variables=['DP03_0025E'],\n    for_geo=['county:*'],\n    in_geo=['state:53'],\n    geometry='polygons',\n    # Optional arg to set a specific resolution: '500k', '5m', or '20m'\n    resolution='20m'\n)\n```\n\nSetting a specific resolution is only supported for polygon-based geometry.\n\n#### Shapefile caching\n\nTo improve performance across queries that include polygon-based geometry data, autocensus caches shapefiles on disk by default. The cache location varies by platform:\n\n* Linux: `/home/{username}/.cache/autocensus`\n* Mac: `/Users/{username}/Library/Application Support/Caches/autocensus`\n* Windows: `C:\\Users\\{username}\\AppData\\Local\\socrata\\autocensus`\n\nYou can clear the cache by manually deleting the cache directory or by executing the `autocensus.clear_cache` function. See the section [Troubleshooting: Clearing the cache] for more details.\n\n[Troubleshooting: Clearing the cache]: #clearing-the-cache\n\n## Publishing to Socrata\n\nIf you have publishing permissions on a Socrata domain, you can publish your query results directly to Socrata via the method `Query.to_socrata`. This method uses [socrata-py] to upload your dataframe with the appropriate field types, formatting, and metadata.\n\n[socrata-py]: https://github.com/socrata/socrata-py\n\n### Credentials\n\nYou must have a Socrata account with appropriate permissions on the domain to which you are publishing. By default, autocensus will look up your Socrata account credentials under the following pairs of common environment variables:\n\n* `SOCRATA_KEY_ID`, `SOCRATA_KEY_SECRET`\n* `SOCRATA_USERNAME`, `SOCRATA_PASSWORD`\n* `MY_SOCRATA_USERNAME`, `MY_SOCRATA_PASSWORD`\n* `SODA_USERNAME`, `SODA_PASSWORD`\n\nAlternatively, you can supply credentials explicitly by way of the `auth` keyword argument:\n\n```python\nauth = (os.environ['MY_SOCRATA_KEY'], os.environ['MY_SOCRATA_KEY_SECRET'])\nquery.to_socrata(\n    'some-domain.data.socrata.com',\n    auth=auth\n)\n```\n\n### Example: Create a new dataset\n\n```python\n# Run query and publish results as a new dataset on Socrata domain\nquery.to_socrata(\n    'some-domain.data.socrata.com',\n    name='Median Commute Time by Colorado County, 2013–2017',  # Optional\n    description='1-year estimates from the American Community Survey'  # Optional\n)\n```\n\n### Example: Replace rows in an existing dataset\n\n```python\n# Run query and publish results to an existing dataset on Socrata domain\nquery.to_socrata(\n    'some-domain.data.socrata.com',\n    dataset_id='xxxx-xxxx'\n)\n```\n\n### Example: Create a new dataset from multiple queries\n\n```python\nfrom autocensus import Query\nfrom autocensus.socrata import to_socrata\nimport pandas as pd\n\n# County-level query\ncounty_query = Query(\n    estimate=1,\n    years=range(2013, 2018),\n    variables=['DP03_0025E'],\n    for_geo='county:*',\n    in_geo='state:08'\n)\ncounty_dataframe = county_query.run()\n\n# State-level query\nstate_query = Query(\n    estimate=1,\n    years=range(2013, 2018),\n    variables=['DP03_0025E'],\n    for_geo='state:08'\n)\nstate_dataframe = state_query.run()\n\n# Concatenate dataframes and upload to Socrata\ncombined_dataframe = pd.concat([\n    county_dataframe,\n    state_dataframe\n])\nto_socrata(\n    'some-domain.data.socrata.com',\n    dataframe=combined_dataframe,\n    name='Median Commute Time for Colorado State and Counties, 2013–2017',  # Optional\n    description='1-year estimates from the American Community Survey'  # Optional\n)\n```\n\n## Troubleshooting\n\n### Clearing the cache\n\nSometimes it is useful to clear the [cache directory] that autocensus uses to store downloaded shapefiles for future queries, especially if you're running into `BadZipFile: File is not a zip file` errors or other shapefile-related problems. Clear your cache like so:\n\n```python\nimport autocensus\n\nautocensus.clear_cache()\n```\n\n[cache directory]: #shapefile-caching\n",
    'author': 'Christopher Setzer',
    'author_email': 'chris.setzer@socrata.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/socrata/autocensus',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7.1,<4.0',
}


setup(**setup_kwargs)
