# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['application_settings']

package_data = \
{'': ['*']}

install_requires = \
['pathvalidate>=2.5,<3.0', 'pydantic>=1.10,<2.0']

extras_require = \
{':python_version < "3.11"': ['tomli>=2.0.1,<3.0.0']}

setup_kwargs = {
    'name': 'application-settings',
    'version': '0.1.0',
    'description': 'For providing a python application with configuration and/or settings',
    'long_description': '# application_settings\n\n[![Build Status](https://github.com/StockwatchDev/application_settings/actions/workflows/application_settings-tests.yml/badge.svg?branch=develop)](https://github.com/StockwatchDev/application_settings/actions)\n[![codecov](https://codecov.io/gh/StockwatchDev/application_settings/branch/develop/graph/badge.svg)](https://app.codecov.io/gh/StockwatchDev/application_settings)\n[![Checked with mypy](http://www.mypy-lang.org/static/mypy_badge.svg)](http://mypy-lang.org/)\n[![linting: pylint](https://img.shields.io/badge/linting-pylint-yellowgreen)](https://github.com/PyCQA/pylint)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Imports: isort](https://img.shields.io/badge/%20imports-isort-%231674b1?style=flat&labelColor=ef8336)](https://pycqa.github.io/isort/)\n\n## What and why\n\nApplication\\_settings is a module for providing a python application with configuration\nand settings. It uses [toml](https://toml.io/en/) configuration files that are parsed\ninto dataclasses. This brings some benefits:\n\n- Configuration parameters are typed, which allows for improved static code analyses.\n- IDEs will provide helpful hints and completion when using configuration parameters.\n- More control over what happens when a config file contains mistakes\n  (by leveraging the power of [pydantic](https://docs.pydantic.dev/)).\n- Possibility to specify defaults when no config file is found or entries are missing.\n- Configuration described in a human-usable, flexible, standardardized and not overly\n  complex format.\n\nParsing is done once during first access and the resulting configuration is stored\nas a singleton.\n\n## How\n### Install the package\n\nOn Linux: `python -m pip install -U application_settings`.\nOn Windows: `py -m pip install -U application_settings`\n\n### Define config section(s) and the container with application info\n\nExample:\n\n```python\nfrom application_settings import (\n    ConfigBase,\n    ConfigSectionBase,\n)\nfrom pydantic.dataclasses import dataclass\n\n\n@dataclass(frozen=True)\nclass MyExample1ConfigSection(ConfigSectionBase):\n    """Config section for an example"""\n\n    field1: str = "field1"\n    field2: int = 2\n\n\n@dataclass(frozen=True)\nclass MyExampleConfig(ConfigBase):\n    """Config for an example"""\n\n    section1: MyExample1ConfigSection = MyExample1ConfigSection()\n\n```\n\nNote: a `pydantic.dataclasses.dataclass` is a drop-in replacement for the standard\n`dataclasses.dataclass`, with validation, see\n[pydantic dataclasses](https://docs.pydantic.dev/usage/dataclasses/).\n\n### Write a config file\n\nFor the example, the config file `~/.my_example/config.toml` could be something like this:\n\n```toml\n[section1]\nfield1 = "my own version of field1"\nfield2 = 22\n```\n\nThe section names in this file are equal to the fieldnames of your container class and the\nentries in a section consist of the fieldnames of your ConfigSection class(es).\nThe order of sections and/or fields in the toml file does not have to adhere to the order\nin which fields have been specified in the Config(Section) classes.\n\n### Use config parameters in your code\n\n```python\n# the first invocation of get() will create the singleton instance of MyExampleConfig\na_variable: str = MyExampleConfig.get().section1.field1  # a_variable == "my own version of field1"\nanother_variable: int = MyExampleConfig.get().section1.field2  # another_variable == 22\n\n# you can reload a config and / or set a non-default path\nanother_config = MyExampleConfig.get(reload=True, configfile_path="./my_config.tml")\n\n```\n\n## Location of the config file\n\nThe path for the config file can be specified via the optional argument `configfile_path`\nof the `get` method that creates the singleton. The path is not stored; if you `reload`\nthen you again have to pass the `configfile_path`.\n\nYou can specify the path either as a string or as a pathlib `Path`. In case of a string\nspec, it is first validated for the platform that you are using; if the validation fails,\na `ValueError` is raised, otherwise a `Path` is constructed from the string.\n\nIf you do not specify a `configfile_path`, then a default location is fetched via\n`default_config_filepath()`. Class `ConfigBase` provides a default implementation, being\na filename `config.toml` located in a subfolder of your home directory. The default name\nof that subfolder is provided by `default_config_foldername()` and consists\nof a dot, followed by a name derived from your container class: the word `Config` is\nremoved, underscores in front of capitals (except for the first letter) and all lower case.\nSee also the example above. If you do not like the default implementation, you can\noverride `default_config_filepath()` and/or `default_config_foldername()`. If you want to\nenforce that a `configfile_path` is specified in `get()`, then let\n`default_config_filepath()` return `None`.\n\n## Handling deviations in the config file\n\n### When your config file does not adhere to the specified types\n\nWhen loading the config file, the values specified are coerced into the appropriate type\nwhere possible. If type coercion is not possible, then a `pydantic.ValidationError`\nis raised. Consider the case where you would use the following config file for\nthe `MyExampleConfig` defined above:\n\n```toml\n[section1]\nfield1 = true\nfield2 = "22"\n```\n\nThe `bool` specified for `field1` will be coerced into a `str` value of `"true"`.\nThe `str` specified for `field2` will be coerced into an `int` value of `22`.\n\n### When your config file does not contain all specified attributes\n\nIf your Config has one of more sections with attributes that do not have a default\nvalue, then a config file must be loaded and these sections and attributes must be\npresent in the loaded config file. If this is not the case, a `TypeError` is raised.\nAttributes that have default values can be omitted\nfrom the config file without problems.\n\nNote that in the dataclass definitions, attributes without default value have to come\nbefore attributes with default values.\n\n### When your config file contains additional, unspecified attributes\n\nEntries in a config file that are not defined in the Config(Section) classes will simply\nbe ignored silently.\n\n## More advanced typing and validation with pydantic\n\n- Non-standard types useful for configuration, such as network addresses, are offered, see\n  [pydantic types](https://docs.pydantic.dev/usage/types/#pydantic-types)\n- The value of numerous common types can be restricted using\n  [pydantic constrained types](https://docs.pydantic.dev/usage/types/#constrained-types)\n\n## License\n\nThis project is licensed under the terms of the MIT license.\n',
    'author': 'Stockwatchdevs',
    'author_email': 'stockwatchdevs@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/StockwatchDev/application_settings',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
