"""
Flow model
"""

from typing import TYPE_CHECKING, Dict, List, Union

from pydantic import BaseModel, Field

from fluidsdk.filters import Filter
from fluidsdk.intents import *
from fluidsdk.message import *
from fluidsdk.reminders import *
from fluidsdk.status_webhook import *

if TYPE_CHECKING:
    from dataclasses import dataclass as _basemodel_decorator
else:
    _basemodel_decorator = lambda x: x


flow_cache = {}


@_basemodel_decorator
class MessageConfig(BaseModel):
    message_delay: float = Field(
        0, title="Message delay", description="Delay between messages"
    )
    queue_messages: bool = Field(
        True,
        title="Queue Messages",
        description="Whether messages should be queued, and sent all at once after intents are done processing, or should be sent as they are generated by the intents.",
    )


@_basemodel_decorator
class Flow(BaseModel):
    flow_id: str = Field(..., title="Flow ID", description="Name of the flow.")
    flow_description: str = Field(
        ...,
        title="Flow Description",
        description="A short description of what the flow does.",
    )
    data: dict = Field(
        {},
        title="Data",
        description="Global flow data all user conversations have access to.",
    )
    reminders: Dict[str, ReminderConfig] = Field(
        {},
        title="Reminders",
        description="Reminders to use in the flow.",
    )
    filters: Dict[str, Filter] = Field(
        {},
        title="Filters",
        description="Filters to be processed before the steps are processed.",
    )
    steps: Dict[
        str,
        Union[
            AskDefiniteIntent,
            AskOpenIntent,
            SayIntent,
            CollectAttachmentsIntent,
            ConditionIntent,
            GPTGenerateIntent,
            GPTSearchIntent,
            ExpressionIntent,
            HTTPRequestIntent,
            SingleAnswerQuizIntent,
            DynamicImage,
        ],
    ] = Field(
        {},
        title="Steps",
        description="Ordered dictionary of Intents",
    )
    invitation_messages: List[BotMessageUnion] = Field(
        [], title="Invitation Messages", description="Messages sent on invitation."
    )
    end_conversation_messages: List[str] = Field(
        ["Your responses have been recorded."],
        title="End Conversation Message",
        description="Message to be sent at the end of the conversation.",
    )
    referred_conversation_messages: List[str] = Field(
        ["Hello, {name}."],
        title="Referred Conversation Messages",
        description="Message to be sent at the begining of the conversation if the user started the conversation.",
    )
    status_webhook: StatusWebhook = Field(
        StatusWebhook(),
        title="Status Webhook",
        description="URL and data for status webhook.",
    )
    message_config: MessageConfig = Field(
        MessageConfig(),
        title="Message Config",
        description="Config for how messages are sent.",
    )

    class Config:
        smart_union = True
