"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const coreutils_1 = require("@jupyterlab/coreutils");
const coreutils_2 = require("@lumino/coreutils");
const jupyterlab_toastify_1 = require("jupyterlab_toastify");
const semver_1 = __importDefault(require("semver"));
exports.companionID = '@mamba-org/gator-lab:companion';
exports.ICompanionValidator = new coreutils_2.Token('@mamba-org/gator-lab:ICompanionValidator');
// Unicode Combining Diacritical Marks
const COMBINING = /[\u0300-\u036F]/g;
/**
 * Validates that conda packages installed in kernels respect
 * kernel companions version specification.
 *
 * The use case is a JupyterLab version served with pre-installed
 * extensions and not manage possible by the end user. In that case
 * especially for libraries with widgets, the conda package version
 * must be coherent with the labextension pre-installed. Otherwise
 * the model in the kernel will not match the one in the frontend.
 *
 */
class CompanionValidator {
    constructor(kernelManager, envManager, settings) {
        this._isDisposed = false;
        this._companions = {};
        // Resolve promise to disconnect signals at disposal
        this._clean = () => {
            return;
        };
        this._envManager = envManager;
        this._updateCompanions(settings);
        settings.changed.connect(this._updateCompanions, this);
        kernelManager.ready.then(() => {
            this._validateSpecs(kernelManager, kernelManager.specs);
            kernelManager.specsChanged.connect(this._validateSpecs, this);
        });
        const clean = new Promise(((resolve) => {
            this._clean = resolve;
        }).bind(this));
        clean.then(() => {
            settings.changed.disconnect(this._updateCompanions, this);
            kernelManager.specsChanged.disconnect(this._validateSpecs, this);
        });
    }
    /**
     * Test whether the validator is disposed.
     */
    get isDisposed() {
        return this._isDisposed;
    }
    /**
     * Dispose of the resources used by the validator.
     */
    dispose() {
        if (this.isDisposed) {
            return;
        }
        this._clean();
        this._isDisposed = true;
    }
    /**
     * Load the user settings
     *
     * @param settings Plugin user settings
     */
    _updateCompanions(settings) {
        this._companions = settings.get('companions').composite;
    }
    /**
     * Convert a kernel name in conda environment name.
     * This follows nb_conda_kernels naming convention.
     *
     * @param name Conda normalized environment name
     * @returns null if this is not a valid conda environment otherwise the name
     */
    static kernelNameToEnvironment(name) {
        const splitted = name.split('-');
        if (splitted[0] === 'conda') {
            if (splitted.length >= 4) {
                return splitted[2];
            }
            else if (splitted.length === 3 && splitted[2] === 'root') {
                // This is the root
                return 'base';
            }
        }
        return null;
    }
    /**
     * Convert semver specification in conda package specification
     *
     * @param range semver version string
     */
    static _semverToPython(range) {
        if (range) {
            return range
                .split('||')
                .map(r => r.split(' ').join(','))
                .join('|');
        }
        return null;
    }
    /**
     * Check the available kernels
     *
     * @param manager A service manager
     * @param specs Available kernelSpec models
     */
    _validateSpecs(manager, // Needed to connect signal
    specs) {
        return __awaiter(this, void 0, void 0, function* () {
            if (Object.keys(this._companions).length === 0) {
                return;
            }
            const environments = yield this._envManager.environments;
            const normalizedNames = {};
            environments.forEach(env => {
                // Normalization need to match as closely as possible nb_conda_kernels conversion
                const normalized = env.name
                    .normalize('NFKD')
                    .replace(COMBINING, '')
                    .replace(/[^a-zA-Z0-9._-]/g, '_');
                normalizedNames[normalized] = env.name;
            });
            function requestCorrection(updates, manager, name) {
                jupyterlab_toastify_1.INotification.warning(`Environment "${name}" has some inconsistencies.`, {
                    buttons: [
                        {
                            label: 'Correct',
                            caption: 'Correct installed packages',
                            callback: () => {
                                jupyterlab_toastify_1.INotification.inProgress('Correct the environment.').then(toastId => {
                                    manager
                                        .getPackageManager()
                                        .install(updates, name)
                                        .then(() => {
                                        jupyterlab_toastify_1.INotification.update({
                                            toastId,
                                            message: 'Environment corrected',
                                            type: 'success',
                                            autoClose: 5000
                                        });
                                    })
                                        .catch((reason) => {
                                        console.error(reason);
                                        jupyterlab_toastify_1.INotification.update({
                                            toastId,
                                            message: 'Fail to correct the environment.',
                                            type: 'error'
                                        });
                                    });
                                });
                            }
                        }
                    ]
                });
            }
            // Loop on the kernelSpecs
            for (const spec of Object.keys(specs.kernelspecs)) {
                let environment;
                const { conda_env_name, conda_env_path } = specs.kernelspecs[spec]
                    .metadata;
                if (conda_env_path) {
                    environment =
                        conda_env_name === 'root' ? 'base' : coreutils_1.PathExt.basename(conda_env_path);
                }
                else {
                    const name = CompanionValidator.kernelNameToEnvironment(spec);
                    environment = normalizedNames[name];
                }
                if (environment) {
                    try {
                        const packages = yield this._envManager
                            .getPackageManager()
                            .refresh(false, environment);
                        const companions = Object.keys(this._companions);
                        const updates = [];
                        packages.forEach((pkg) => {
                            if (companions.indexOf(pkg.name) >= 0 &&
                                !semver_1.default.satisfies(pkg.version_installed, this._companions[pkg.name])) {
                                let pythonVersion = CompanionValidator._semverToPython(semver_1.default.validRange(this._companions[pkg.name]));
                                if (pythonVersion) {
                                    if ('<>='.indexOf(pythonVersion[0]) < 0) {
                                        pythonVersion = '=' + pythonVersion; // prefix with '=' if nothing
                                    }
                                    updates.push(pkg.name + pythonVersion);
                                }
                            }
                        });
                        if (updates.length > 0) {
                            requestCorrection(updates, this._envManager, environment);
                        }
                    }
                    catch (error) {
                        console.error(`Fail to check environment '${environment}'`, error);
                    }
                }
            }
        });
    }
    /**
     * Validate the kernelSpec models
     *
     * @param kernelSpecs Available kernelSpec models
     */
    validate(kernelSpecs) {
        this._validateSpecs(null, kernelSpecs);
    }
}
exports.CompanionValidator = CompanionValidator;
//# sourceMappingURL=validator.js.map