# Copyright (c) 2018 The Harmonica Developers.
# Distributed under the terms of the BSD 3-Clause License.
# SPDX-License-Identifier: BSD-3-Clause
#
# This code is part of the Fatiando a Terra project (https://www.fatiando.org)
#
"""
Point Masses in Cartesian Coordinates
=====================================

The simplest geometry used to compute gravitational fields are point masses.
Although modelling geologic structures with point masses can be challenging,
they are very useful for other purposes: creating synthetic models, solving
inverse problems, generating equivalent sources for interpolation, etc. The
gravitational fields generated by point masses can be quickly computed either
in Cartesian or in geocentric spherical coordinate systems. We will compute the
gravitational acceleration generated by a set of point masses on a computation
grid given in Cartesian coordinates using the
:func:`harmonica.point_gravity` function.
"""
import pygmt
import verde as vd

import harmonica as hm

# Define the coordinates for two point masses
easting = [5e3, 15e3]
northing = [7e3, 13e3]
# The vertical coordinate is positive upward so negative numbers represent
# depth
upward = [-0.5e3, -1e3]
points = [easting, northing, upward]
# We're using "negative" masses to represent a "mass deficit" since we assume
# measurements are gravity disturbances, not actual gravity values.
masses = [3e11, -10e11]

# Define computation points on a grid at 500m above the ground
coordinates = vd.grid_coordinates(
    region=[0, 20e3, 0, 20e3], shape=(100, 100), extra_coords=500
)

# Compute the downward component of the gravitational acceleration
gravity = hm.point_gravity(
    coordinates, points, masses, field="g_z", coordinate_system="cartesian"
)
print(gravity)
grid = vd.make_xarray_grid(
    coordinates, gravity, data_names="gravity", extra_coords_names="extra"
)

# Plot the results on a map
fig = pygmt.Figure()

title = "Gravitational acceleration (downward)"

maxabs = vd.maxabs(gravity) * 0.80

pygmt.makecpt(cmap="vik", series=(-maxabs, maxabs, 0.3))

with pygmt.config(FONT_TITLE="16p"):
    fig.grdimage(
        region=[0, 20e3, 0, 20e3],
        projection="X10c",
        grid=grid.gravity,
        dpi=1000,
        frame=["a", f"+t{title}", "x+lm", "y+lm"],
        cmap=True,
    )

fig.plot(x=easting, y=northing, style="c0.2c", color="grey")

fig.colorbar(cmap=True, position="JMR", frame=["a.6f.2", "x+lmGal"])

fig.show()
