# AUTOGENERATED! DO NOT EDIT! File to edit: 03_spectra.ipynb (unless otherwise specified).

__all__ = ['calc_psd', 'plot_psds', 'plot_psd_mt', 'calc_bands', 'stat', 'calc_gfp', 'plot_gfp', 'bandpower',
           'calc_bandpowers', 'calc_ch_bands', 'calc_epoch_bands', 'plot_welch', 'calc_tfr', 'calc_csd']

# Cell
from cognify import dataset
import pandas as pd
import mne
import matplotlib.pyplot as plt
import numpy as np

# Cell
from mne.time_frequency import psd_array_multitaper, psd_multitaper
from scipy import signal

# Cell
def calc_psd(ch, sfreq, method='welch', window_sec=None, low=2):
    if method == 'welch':
        if window_sec is not None:
            nperseg = window_sec * sfreq
        else:
            nperseg = (2/low) * sfreq

        # compute modified periodogram
        freqs, psd = signal.welch(ch, sfreq, nperseg=nperseg)

    elif method == 'multitaper':
        psd, freqs = psd_array_multitaper(ch, sfreq, adaptive=True,
                                  normalization='full', verbose=0)

    return freqs, psd

# Cell
def plot_psds(raw, picks=None, figsize=(30,10)):
    plt.figure(figsize=figsize)
    freqs = np.arange(60, fmax, 60)
    ax = plt.axes()
    raw.plot_psd(tmin=tmin, tmax=tmax, fmin=fmin, fmax=fmax, n_fft=n_fft,
                 n_jobs=1, proj=False, ax=ax, color=(0, 0, 1), picks=picks,
                 show=False, average=True)

    raw.plot_psd(tmin=tmin, tmax=tmax, fmin=fmin, fmax=fmax, n_fft=n_fft,
                 n_jobs=1, proj=True, ax=ax, color=(0, 1, 0), picks=picks,
                 show=False, average=True)

    # And now do the same with SSP + notch filtering
    # Pick all channels for notch since the SSP projection mixes channels together
    raw.notch_filter(freqs, n_jobs=1, fir_design='firwin')
    raw.plot_psd(tmin=tmin, tmax=tmax, fmin=fmin, fmax=fmax, n_fft=n_fft,
                 n_jobs=1, proj=True, ax=ax, color=(1, 0, 0), picks=picks,
                 show=False, average=True)

    leg_lines = [line for line in ax.lines if line.get_linestyle() == '-']
    plt.legend(leg_lines, ['Without SSP', 'With SSP', 'SSP + Notch'])

# Cell
def plot_psd_mt(raw, picks=None, figsize=(30,5)):
    psds, freqs = psd_multitaper(raw, low_bias=True, tmin=tmin, tmax=tmax,
                                 fmin=fmin, fmax=fmax, proj=True, picks=picks, n_jobs=1)
    psds = 10 * np.log10(psds) # convert power to decibels
    psds_mean, psds_std = psds.mean(0), psds.std(0)

    plt.figure(figsize=figsize)
    ax = plt.axes()
    ax.plot(freqs, psds_mean, color=(0,0,1))

    ax.fill_between(freqs, psds_mean - psds_std,
                    psds_mean + psds_std, color=(0,1,0))

    ax.set(title='Multitaper PSD', xlabel='Frequency', ylabel='Power Spectral Density (dB)');

# Cell
def calc_bands(bands, epochs, verbose=0):
    frequency_map = list()
    for band, fmin, fmax in bands:
        _epochs = epochs.copy()
        # bandpass filter
        _epochs.filter(fmin, fmax, l_trans_bandwidth=1, h_trans_bandwidth=1, verbose=verbose)
        _epochs.subtract_evoked()

        # get analytical signal (envelope)
        _epochs.apply_hilbert(envelope=True, verbose=verbose)
        frequency_map.append(((band, fmin, fmax), _epochs.average()))
    return frequency_map

# Cell
from mne.stats import bootstrap_confidence_interval
from mne.baseline import rescale

# Cell
def stat(x):
    # sum of squares
    return np.sum(x**2, axis=0)

# Cell
def calc_gfp(band, average, times):
        gfp = np.sum(average.data ** 2, axis=0)
        return mne.baseline.rescale(gfp, times, baseline=(None, 0))

# Cell
def plot_gfp(freq_map, figsize=(30,5), verbose=1):
    n = len(freq_map)
    fig, axes = plt.subplots(n, 1, figsize=figsize, sharex=True, sharey=True)
    colors = plt.get_cmap('winter_r')(np.linspace(0, 1, n))

    for (band, average), color, ax in zip(
            freq_map, colors, axes.ravel()[::-1]):

        times = average.times * 1e3
        gfp = calc_gfp(band, average, times)
        freq_name, fmin, fmax = band

        ax.plot(times, gfp, label=freq_name, color=color, linewidth=2.5)
        ax.axhline(0, linestyle='--', color='grey', linewidth=2)

        # TODO: remove confidence interval calc for plotting
        ci_low, ci_up = bootstrap_confidence_interval(average.data, random_state=0,
                                                      stat_fun=stat)
        ci_low = rescale(ci_low, average.times, baseline=(None, 0), verbose=verbose)
        ci_up = rescale(ci_up, average.times, baseline=(None, 0), verbose=verbose)
        ax.fill_between(times, gfp + ci_up, gfp - ci_low, color=color, alpha=0.3)
        ax.grid(True)
        ax.set_ylabel('GFP')
        ax.annotate('%s (%d-%dHz)' % (freq_name, fmin, fmax),
                    xy=(0.95, 0.8),
                    horizontalalignment='right',
                    xycoords='axes fraction')
        ax.set_xlim(-1000, 3000)

    axes.ravel()[-1].set_xlabel('Time [ms]')

# Cell
from scipy.integrate import simps

# Cell
def bandpower(ch, sfreq, band, method='welch', window_sec=None, relative=False):
    name, low, high = band
    freqs, psd = calc_psd(ch, sfreq, method=method, window_sec=window_sec, low=low)
    freq_res = freqs[1] - freqs[0]
    freq_mask = np.logical_and(freqs >= low, freqs <= high)
    # integrate approximation of the spectrum
    bp = simps(psd[freq_mask], dx=freq_res)

    if relative: bp/= simps(psd, dx=freq_res)
    return bp

# Cell
def calc_bandpowers(ch, sfreq, bands, method='welch', window_sec=None, relative=False):
    bps = dict()
    for band in bands:
        name, _, _ = band
        bps[name] = bandpower(ch, sfreq, band, method, window_sec=window_sec, relative=relative)
    return bps

# Cell
def calc_ch_bands(data, sfreq, bands, method='welch', window_sec=None,relative=False):
    ch_bands = []
    for i,ch in enumerate(data):
        bps = calc_bandpowers(ch, sfreq, bands, method=method, window_sec=window_sec, relative=relative)
        ch_bands.append(bps)
    return ch_bands

# Cell
def calc_epoch_bands(epochs, bands, method='welch', window_sec=None):
    sfreq = epochs.info['sfreq']
    e_bands = [calc_ch_bands(e, sfreq, bands, method=method, window_sec=window_sec) for e in epochs]
    # TODO: drop dataframe casting
    e_powers = pd.DataFrame(e_bands)
    return np.array([[list(pb.values()) for pb in e_powers[ch]] for ch in e_powers.columns])

# Cell
def plot_welch(data, sfreq, band):
    window = 4 * sfreq
    name, low, high = band

    freqs, psd = signal.welch(data, sfreq, nperseg=window)
    freq_mask = np.logical_and(freqs >= low, freqs <= high)

    plt.figure(figsize=(8, 4))
    plt.plot(freqs, psd, color='k', lw=2)
    plt.fill_between(freqs, psd, where=freq_mask, color='skyblue')
    plt.xlabel('Frequency (Hz)')
    plt.ylabel('Power spectral density (V^2 / Hz)')
    plt.ylim([0, psd.max() * 1.1])
    plt.title("Welch's periodogram")
    plt.xlim([0, freqs.max()])

# Cell
from mne.time_frequency import tfr_morlet

# Cell
def calc_tfr(epochs, frange=[1,50], log=False):
    freqs = np.arange(*frange)
    if log: freqs = np.logspace(*np.log10([1, 50]))

    n_cycles = freqs / 2. # calculate cycles for each frequency (nyquist)

    power, itc = tfr_morlet(epochs, freqs=freqs, n_cycles=n_cycles,
                            use_fft=True, return_itc=True, decim=3, n_jobs=1)
    return power, itc

# Cell
from mne.time_frequency import csd_fourier, csd_multitaper, csd_morlet

# Cell
def calc_csd(epochs, freqs, method='fft', n_jobs=1, verbose=0):
    fmin, fmax = freqs[0], freqs[-1]

    if method == 'fft': return csd_fourier(epochs, fmin=fmin, fmax=fmax,
                                           n_jobs=n_jobs, verbose=0)
    if method == 'multi': return csd_multitaper(epochs, fmin=fmin, fmax=fmax,
                                             adaptive=True, n_jobs=n_jobs, verbose=verbose)

    if method == 'morlet': return csd_morlet(epochs, freqs,
                                             n_jobs=n_jobs, verbose=verbose)