"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserTablePrivileges = void 0;
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const database_query_1 = require("./database-query");
const handler_name_1 = require("./database-query-provider/handler-name");
const table_1 = require("../table");
/**
 * Privileges granted to a Redshift user on Redshift tables.
 *
 * This construct is located in the `private` directory to ensure that it is not exported for direct public use. This
 * means that user privileges must be managed through the `Table.grant` method or the `User.addTablePrivileges`
 * method. Thus, each `User` will have at most one `UserTablePrivileges` construct to manage its privileges. For details
 * on why this is a Good Thing, see the README, under "Granting Privileges".
 */
class UserTablePrivileges extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.privileges = props.privileges ?? [];
        new database_query_1.DatabaseQuery(this, 'Resource', {
            ...props,
            handler: handler_name_1.HandlerName.UserTablePrivileges,
            properties: {
                username: props.user.username,
                tablePrivileges: cdk.Lazy.any({
                    produce: () => {
                        const reducedPrivileges = this.privileges.reduce((privileges, { table, actions }) => {
                            const tableName = table.tableName;
                            if (!(tableName in privileges)) {
                                privileges[tableName] = [];
                            }
                            actions = actions.concat(privileges[tableName]);
                            if (actions.includes(table_1.TableAction.ALL)) {
                                actions = [table_1.TableAction.ALL];
                            }
                            if (actions.includes(table_1.TableAction.UPDATE) || actions.includes(table_1.TableAction.DELETE)) {
                                actions.push(table_1.TableAction.SELECT);
                            }
                            privileges[tableName] = Array.from(new Set(actions));
                            return privileges;
                        }, {});
                        const serializedPrivileges = Object.entries(reducedPrivileges).map(([tableName, actions]) => ({
                            tableName: tableName,
                            actions: actions.map(action => table_1.TableAction[action]),
                        }));
                        return serializedPrivileges;
                    },
                }),
            },
        });
    }
    /**
     * Grant this user additional privileges.
     */
    addPrivileges(table, ...actions) {
        this.privileges.push({ table, actions });
    }
}
exports.UserTablePrivileges = UserTablePrivileges;
//# sourceMappingURL=data:application/json;base64,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