"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerPort = exports.LoadBalancer = exports.LoadBalancingProtocol = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener
 *
 * Routes to a fleet of of instances in a VPC.
 */
class LoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.anyValue({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (props.crossZone === undefined || props.crossZone) ? true : false,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer
     *
     * @returns A ListenerPort object that controls connections to the listener port
     */
    addListener(listener) {
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: listener.sslCertificateId,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    addTarget(target) {
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @attribute
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @attribute
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *    const listener = elb.addListener(...);
 *
 *    listener.connections.allowDefaultPortFromAnyIPv4();
 *    // or
 *    instance.connections.allowToDefaultPort(listener);
 */
class ListenerPort {
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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