"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const path = require("path");
const tasks = require("../../lib");
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-stepfunctions-integ');
const handler = new aws_lambda_1.Function(stack, 'Handler', {
    code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
    handler: 'index.main',
    runtime: aws_lambda_1.Runtime.PYTHON_3_6
});
const submitJob = new sfn.Task(stack, 'Invoke Handler', {
    task: new tasks.InvokeFunction(handler),
});
const callbackHandler = new aws_lambda_1.Function(stack, 'CallbackHandler', {
    code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
    handler: 'index.main',
    runtime: aws_lambda_1.Runtime.PYTHON_3_6
});
const taskTokenHandler = new sfn.Task(stack, 'Invoke Handler with task token', {
    task: new tasks.RunLambdaTask(callbackHandler, {
        integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
        payload: {
            token: sfn.Context.taskToken
        }
    }),
    inputPath: '$.guid',
    resultPath: '$.status',
});
const isComplete = new sfn.Choice(stack, 'Job Complete?');
const jobFailed = new sfn.Fail(stack, 'Job Failed', {
    cause: 'AWS Batch Job Failed',
    error: 'DescribeJob returned FAILED',
});
const finalStatus = new sfn.Pass(stack, 'Final step');
const chain = sfn.Chain
    .start(submitJob)
    .next(taskTokenHandler)
    .next(isComplete
    .when(sfn.Condition.stringEquals('$.status', 'FAILED'), jobFailed)
    .when(sfn.Condition.stringEquals('$.status', 'SUCCEEDED'), finalStatus));
new sfn.StateMachine(stack, 'StateMachine', {
    definition: chain,
    timeout: cdk.Duration.seconds(30)
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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