"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * A Step Functions Task to publish messages to SNS topic.
 *
 * A Function can be used directly as a Resource, but this class mirrors
 * integration with other AWS services via a specific class instance.
 */
class PublishToTopic {
    constructor(topic, props) {
        this.topic = topic;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call SNS.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(props.message)) {
                throw new Error('Task Token is missing in message (pass Context.taskToken somewhere in message)');
            }
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('sns', 'publish', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    actions: ['sns:Publish'],
                    resources: [this.topic.topicArn]
                })],
            parameters: {
                TopicArn: this.topic.topicArn,
                Message: this.props.message.value,
                MessageStructure: this.props.messagePerSubscriptionType ? 'json' : undefined,
                Subject: this.props.subject,
            }
        };
    }
}
exports.PublishToTopic = PublishToTopic;
//# sourceMappingURL=data:application/json;base64,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