"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * Invoke a Lambda function as a Task
 *
 * OUTPUT: the output of this task is either the return value of Lambda's
 * Invoke call, or whatever the Lambda Function posted back using
 * `SendTaskSuccess/SendTaskFailure` in `waitForTaskToken` mode.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 */
class RunLambdaTask {
    constructor(lambdaFunction, props = {}) {
        this.lambdaFunction = lambdaFunction;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Lambda.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is missing in payload (pass Context.taskToken somewhere in payload)');
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('lambda', 'invoke', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    resources: [this.lambdaFunction.functionArn],
                    actions: ['lambda:InvokeFunction'],
                })],
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: { LambdaFunctionArn: this.lambdaFunction.functionArn },
            parameters: {
                FunctionName: this.lambdaFunction.functionName,
                Payload: this.props.payload,
                InvocationType: this.props.invocationType,
                ClientContext: this.props.clientContext,
                Qualifier: this.props.qualifier
            }
        };
    }
}
exports.RunLambdaTask = RunLambdaTask;
/**
 * Invocation type of a Lambda
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke synchronously
     *
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke asynchronously
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * TValidate parameter values and verify that the user or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType = exports.InvocationType || (exports.InvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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