"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * Invoke a Glue job as a Task
 *
 * OUTPUT: the output of this task is a JobRun structure, for details consult
 * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-JobRun
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html
 */
class RunGlueJobTask {
    constructor(glueJobName, props = {}) {
        this.glueJobName = glueJobName;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Glue.`);
        }
    }
    bind(task) {
        var _a;
        const notificationProperty = this.props.notifyDelayAfter ? { NotifyDelayAfter: this.props.notifyDelayAfter.toMinutes() } : null;
        let iamActions;
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.FIRE_AND_FORGET) {
            iamActions = ['glue:StartJobRun'];
        }
        else if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            iamActions = [
                'glue:StartJobRun',
                'glue:GetJobRun',
                'glue:GetJobRuns',
                'glue:BatchStopJobRun'
            ];
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('glue', 'startJobRun', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    resources: [
                        core_1.Stack.of(task).formatArn({
                            service: 'glue',
                            resource: 'job',
                            resourceName: this.glueJobName
                        })
                    ],
                    actions: iamActions
                })],
            metricPrefixSingular: 'GlueJob',
            metricPrefixPlural: 'GlueJobs',
            metricDimensions: { GlueJobName: this.glueJobName },
            parameters: {
                JobName: this.glueJobName,
                Arguments: this.props.arguments,
                Timeout: (_a = this.props.timeout) === null || _a === void 0 ? void 0 : _a.toMinutes(),
                SecurityConfiguration: this.props.securityConfiguration,
                NotificationProperty: notificationProperty
            }
        };
    }
}
exports.RunGlueJobTask = RunGlueJobTask;
//# sourceMappingURL=data:application/json;base64,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