"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * A StepFunctions Task to run a Task on ECS or Fargate
 */
class EcsRunTaskBase {
    constructor(props) {
        this.props = props;
        /**
         * Manage allowed network traffic for this service
         */
        this.connections = new ec2.Connections();
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call ECS.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.containerOverrides)) {
            throw new Error('Task Token is missing in containerOverrides (pass Context.taskToken somewhere in containerOverrides)');
        }
        for (const override of this.props.containerOverrides || []) {
            const name = override.containerName;
            if (!cdk.Token.isUnresolved(name)) {
                const cont = this.props.taskDefinition.node.tryFindChild(name);
                if (!cont) {
                    throw new Error(`Overrides mention container with name '${name}', but no such container in task definition`);
                }
            }
        }
    }
    bind(task) {
        if (this.networkConfiguration !== undefined) {
            // Make sure we have a security group if we're using AWSVPC networking
            if (this.securityGroup === undefined) {
                this.securityGroup = new ec2.SecurityGroup(task, 'SecurityGroup', { vpc: this.props.cluster.vpc });
            }
            this.connections.addSecurityGroup(this.securityGroup);
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('ecs', 'runTask', this.integrationPattern),
            parameters: {
                Cluster: this.props.cluster.clusterArn,
                TaskDefinition: this.props.taskDefinition.taskDefinitionArn,
                NetworkConfiguration: this.networkConfiguration,
                Overrides: renderOverrides(this.props.containerOverrides),
                ...this.props.parameters,
            },
            policyStatements: this.makePolicyStatements(task),
        };
    }
    configureAwsVpcNetworking(vpc, assignPublicIp, subnetSelection, securityGroup) {
        if (subnetSelection === undefined) {
            subnetSelection = { subnetType: assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE };
        }
        // If none is given here, one will be created later on during bind()
        this.securityGroup = securityGroup;
        this.networkConfiguration = {
            AwsvpcConfiguration: {
                AssignPublicIp: assignPublicIp !== undefined ? (assignPublicIp ? 'ENABLED' : 'DISABLED') : undefined,
                Subnets: vpc.selectSubnets(subnetSelection).subnetIds,
                SecurityGroups: cdk.Lazy.listValue({ produce: () => [this.securityGroup.securityGroupId] }),
            }
        };
    }
    makePolicyStatements(task) {
        const stack = cdk.Stack.of(task);
        // https://docs.aws.amazon.com/step-functions/latest/dg/ecs-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.props.taskDefinition.taskDefinitionArn],
            }),
            new iam.PolicyStatement({
                actions: ['ecs:StopTask', 'ecs:DescribeTasks'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: cdk.Lazy.listValue({ produce: () => this.taskExecutionRoles().map(r => r.roleArn) })
            }),
        ];
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForECSTaskRule'
                    })]
            }));
        }
        return policyStatements;
    }
    taskExecutionRoles() {
        // Need to be able to pass both Task and Execution role, apparently
        const ret = new Array();
        ret.push(this.props.taskDefinition.taskRole);
        if (this.props.taskDefinition.executionRole) {
            ret.push(this.props.taskDefinition.executionRole);
        }
        return ret;
    }
}
exports.EcsRunTaskBase = EcsRunTaskBase;
function renderOverrides(containerOverrides) {
    if (!containerOverrides) {
        return undefined;
    }
    const ret = new Array();
    for (const override of containerOverrides) {
        ret.push({
            Name: override.containerName,
            Command: override.command,
            Cpu: override.cpu,
            Memory: override.memoryLimit,
            MemoryReservation: override.memoryReservation,
            Environment: override.environment && override.environment.map(e => ({
                Name: e.name,
                Value: e.value,
            }))
        });
    }
    return { ContainerOverrides: ret };
}
//# sourceMappingURL=data:application/json;base64,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