"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretRotation = exports.SecretRotationApplication = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const lambda = require("@aws-cdk/aws-lambda");
const serverless = require("@aws-cdk/aws-sam");
const core_1 = require("@aws-cdk/core");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * A secret rotation serverless application.
 *
 * @stability stable
 */
class SecretRotationApplication {
    /**
     * @stability stable
     */
    constructor(applicationId, semanticVersion, options) {
        this.applicationId = `arn:aws:serverlessrepo:us-east-1:297356227824:applications/${applicationId}`;
        this.semanticVersion = semanticVersion;
        this.isMultiUser = options && options.isMultiUser;
    }
}
exports.SecretRotationApplication = SecretRotationApplication;
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the single user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the multi user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.MARIADB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the single user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the multi user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.MYSQL_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the single user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the multi user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.ORACLE_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the single user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the multi user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the single user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the multi user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the single user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the multi user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the single user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the multi user rotation scheme.
 *
 * @stability stable
 */
SecretRotationApplication.MONGODB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Secret rotation for a service or database.
 *
 * @stability stable
 */
class SecretRotation extends core_2.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.target.connections.defaultPort) {
            throw new Error('The `target` connections must have a default port range.');
        }
        if (props.application.isMultiUser && !props.masterSecret) {
            throw new Error('The `masterSecret` must be specified for application using the multi user scheme.');
        }
        // Max length of 64 chars, get the last 64 chars
        const uniqueId = core_1.Names.uniqueId(this);
        const rotationFunctionName = uniqueId.substring(Math.max(uniqueId.length - 64, 0), uniqueId.length);
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
        });
        props.target.connections.allowDefaultPortFrom(securityGroup);
        const parameters = {
            endpoint: `https://secretsmanager.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}`,
            functionName: rotationFunctionName,
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds.join(','),
            vpcSecurityGroupIds: securityGroup.securityGroupId,
        };
        if (props.excludeCharacters !== undefined) {
            parameters.excludeCharacters = props.excludeCharacters;
        }
        if (props.secret.encryptionKey) {
            parameters.kmsKeyArn = props.secret.encryptionKey.keyArn;
        }
        if (props.masterSecret) {
            parameters.masterSecretArn = props.masterSecret.secretArn;
            if (props.masterSecret.encryptionKey) {
                parameters.masterSecretKmsKeyArn = props.masterSecret.encryptionKey.keyArn;
            }
        }
        const application = new serverless.CfnApplication(this, 'Resource', {
            location: props.application,
            parameters,
        });
        // This creates a CF a dependency between the rotation schedule and the
        // serverless application. This is needed because it's the application
        // that creates the Lambda permission to invoke the function.
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html
        const rotationLambda = lambda.Function.fromFunctionArn(this, 'RotationLambda', core_1.Token.asString(application.getAtt('Outputs.RotationLambdaARN')));
        props.secret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
            automaticallyAfter: props.automaticallyAfter,
        });
        // Prevent master secret deletion when rotation is in place
        if (props.masterSecret) {
            props.masterSecret.denyAccountRootDelete();
        }
    }
}
exports.SecretRotation = SecretRotation;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2VjcmV0LXJvdGF0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic2VjcmV0LXJvdGF0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUF3QztBQUN4Qyw4Q0FBOEM7QUFDOUMsK0NBQStDO0FBQy9DLHdDQUE4RDtBQUk5RCxnSEFBZ0g7QUFDaEgsMkJBQTJCO0FBQzNCLHdDQUEyRDs7Ozs7O0FBZ0IzRCxNQUFhLHlCQUF5Qjs7OztJQW9HcEMsWUFBWSxhQUFxQixFQUFFLGVBQXVCLEVBQUUsT0FBMEM7UUFDcEcsSUFBSSxDQUFDLGFBQWEsR0FBRyw4REFBOEQsYUFBYSxFQUFFLENBQUM7UUFDbkcsSUFBSSxDQUFDLGVBQWUsR0FBRyxlQUFlLENBQUM7UUFDdkMsSUFBSSxDQUFDLFdBQVcsR0FBRyxPQUFPLElBQUksT0FBTyxDQUFDLFdBQVcsQ0FBQztJQUNwRCxDQUFDOztBQXhHSCw4REF5R0M7Ozs7OztBQXJHd0Isc0RBQTRCLEdBQUcsSUFBSSx5QkFBeUIsQ0FBQyw0Q0FBNEMsRUFBRSxRQUFRLENBQUMsQ0FBQzs7Ozs7O0FBS3JILHFEQUEyQixHQUFHLElBQUkseUJBQXlCLENBQUMsMkNBQTJDLEVBQUUsUUFBUSxFQUFFO0lBQ3hJLFdBQVcsRUFBRSxJQUFJO0NBQ2xCLENBQUMsQ0FBQzs7Ozs7O0FBS29CLG9EQUEwQixHQUFHLElBQUkseUJBQXlCLENBQUMsMENBQTBDLEVBQUUsUUFBUSxDQUFDLENBQUM7Ozs7OztBQUtqSCxtREFBeUIsR0FBRyxJQUFJLHlCQUF5QixDQUFDLHlDQUF5QyxFQUFFLFFBQVEsRUFBRTtJQUNwSSxXQUFXLEVBQUUsSUFBSTtDQUNsQixDQUFDLENBQUM7Ozs7OztBQUtvQixxREFBMkIsR0FBRyxJQUFJLHlCQUF5QixDQUFDLDJDQUEyQyxFQUFFLFFBQVEsQ0FBQyxDQUFDOzs7Ozs7QUFLbkgsb0RBQTBCLEdBQUcsSUFBSSx5QkFBeUIsQ0FBQywwQ0FBMEMsRUFBRSxRQUFRLEVBQUU7SUFDdEksV0FBVyxFQUFFLElBQUk7Q0FDbEIsQ0FBQyxDQUFDOzs7Ozs7QUFLb0IsdURBQTZCLEdBQUcsSUFBSSx5QkFBeUIsQ0FBQywrQ0FBK0MsRUFBRSxRQUFRLENBQUMsQ0FBQzs7Ozs7O0FBS3pILHNEQUE0QixHQUFHLElBQUkseUJBQXlCLENBQUMsOENBQThDLEVBQUUsUUFBUSxFQUFFO0lBQzVJLFdBQVcsRUFBRSxJQUFJO0NBQ2xCLENBQUMsQ0FBQzs7Ozs7O0FBS29CLHdEQUE4QixHQUFHLElBQUkseUJBQXlCLENBQUMsOENBQThDLEVBQUUsUUFBUSxDQUFDLENBQUM7Ozs7OztBQUt6SCx1REFBNkIsR0FBRyxJQUFJLHlCQUF5QixDQUFDLDZDQUE2QyxFQUFFLFFBQVEsRUFBRTtJQUM1SSxXQUFXLEVBQUUsSUFBSTtDQUNsQixDQUFDLENBQUM7Ozs7OztBQUtvQix1REFBNkIsR0FBRyxJQUFJLHlCQUF5QixDQUFDLDBDQUEwQyxFQUFFLFFBQVEsQ0FBQyxDQUFDOzs7Ozs7QUFLcEgsc0RBQTRCLEdBQUcsSUFBSSx5QkFBeUIsQ0FBQyx5Q0FBeUMsRUFBRSxRQUFRLEVBQUU7SUFDdkksV0FBVyxFQUFFLElBQUk7Q0FDbEIsQ0FBQyxDQUFDOzs7Ozs7QUFLb0Isc0RBQTRCLEdBQUcsSUFBSSx5QkFBeUIsQ0FBQyx5Q0FBeUMsRUFBRSxRQUFRLENBQUMsQ0FBQzs7Ozs7O0FBS2xILHFEQUEyQixHQUFHLElBQUkseUJBQXlCLENBQUMsd0NBQXdDLEVBQUUsUUFBUSxFQUFFO0lBQ3JJLFdBQVcsRUFBRSxJQUFJO0NBQ2xCLENBQUMsQ0FBQzs7Ozs7O0FBeUdMLE1BQWEsY0FBZSxTQUFRLGdCQUFhOzs7O0lBQy9DLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMEI7UUFDbEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsV0FBVyxFQUFFO1lBQ3pDLE1BQU0sSUFBSSxLQUFLLENBQUMsMERBQTBELENBQUMsQ0FBQztTQUM3RTtRQUVELElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxXQUFXLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3hELE1BQU0sSUFBSSxLQUFLLENBQUMsbUZBQW1GLENBQUMsQ0FBQztTQUN0RztRQUVELGdEQUFnRDtRQUNoRCxNQUFNLFFBQVEsR0FBRyxZQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3RDLE1BQU0sb0JBQW9CLEdBQUcsUUFBUSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEdBQUcsRUFBRSxFQUFFLENBQUMsQ0FBQyxFQUFFLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUVwRyxNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxJQUFJLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO1lBQ3hGLEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRztTQUNmLENBQUMsQ0FBQztRQUNILEtBQUssQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLG9CQUFvQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRTdELE1BQU0sVUFBVSxHQUE4QjtZQUM1QyxRQUFRLEVBQUUsMEJBQTBCLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxJQUFJLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxFQUFFO1lBQ3ZGLFlBQVksRUFBRSxvQkFBb0I7WUFDbEMsWUFBWSxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQztZQUMzRSxtQkFBbUIsRUFBRSxhQUFhLENBQUMsZUFBZTtTQUNuRCxDQUFDO1FBRUYsSUFBSSxLQUFLLENBQUMsaUJBQWlCLEtBQUssU0FBUyxFQUFFO1lBQ3pDLFVBQVUsQ0FBQyxpQkFBaUIsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUM7U0FDeEQ7UUFFRCxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFO1lBQzlCLFVBQVUsQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDO1NBQzFEO1FBRUQsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3RCLFVBQVUsQ0FBQyxlQUFlLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUM7WUFFMUQsSUFBSSxLQUFLLENBQUMsWUFBWSxDQUFDLGFBQWEsRUFBRTtnQkFDcEMsVUFBVSxDQUFDLHFCQUFxQixHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQzthQUM1RTtTQUNGO1FBRUQsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDbEUsUUFBUSxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzNCLFVBQVU7U0FDWCxDQUFDLENBQUM7UUFFSCx1RUFBdUU7UUFDdkUsc0VBQXNFO1FBQ3RFLDZEQUE2RDtRQUM3RCxrR0FBa0c7UUFDbEcsTUFBTSxjQUFjLEdBQUcsTUFBTSxDQUFDLFFBQVEsQ0FBQyxlQUFlLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFLFlBQUssQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQywyQkFBMkIsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVoSixLQUFLLENBQUMsTUFBTSxDQUFDLG1CQUFtQixDQUFDLGtCQUFrQixFQUFFO1lBQ25ELGNBQWM7WUFDZCxrQkFBa0IsRUFBRSxLQUFLLENBQUMsa0JBQWtCO1NBQzdDLENBQUMsQ0FBQztRQUVILDJEQUEyRDtRQUMzRCxJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDdEIsS0FBSyxDQUFDLFlBQVksQ0FBQyxxQkFBcUIsRUFBRSxDQUFDO1NBQzVDO0lBQ0gsQ0FBQztDQUNGO0FBakVELHdDQWlFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIHNlcnZlcmxlc3MgZnJvbSAnQGF3cy1jZGsvYXdzLXNhbSc7XG5pbXBvcnQgeyBEdXJhdGlvbiwgTmFtZXMsIFN0YWNrLCBUb2tlbiB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBJU2VjcmV0IH0gZnJvbSAnLi9zZWNyZXQnO1xuXG4vLyB2MiAtIGtlZXAgdGhpcyBpbXBvcnQgYXMgYSBzZXBhcmF0ZSBzZWN0aW9uIHRvIHJlZHVjZSBtZXJnZSBjb25mbGljdCB3aGVuIGZvcndhcmQgbWVyZ2luZyB3aXRoIHRoZSB2MiBicmFuY2guXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmVcbmltcG9ydCB7IENvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFNlY3JldFJvdGF0aW9uQXBwbGljYXRpb25PcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGlzTXVsdGlVc2VyPzogYm9vbGVhbjtcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBTZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IE1BUklBREJfUk9UQVRJT05fU0lOR0xFX1VTRVIgPSBuZXcgU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbignU2VjcmV0c01hbmFnZXJSRFNNYXJpYURCUm90YXRpb25TaW5nbGVVc2VyJywgJzEuMS42MCcpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IE1BUklBREJfUk9UQVRJT05fTVVMVElfVVNFUiA9IG5ldyBTZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uKCdTZWNyZXRzTWFuYWdlclJEU01hcmlhREJSb3RhdGlvbk11bHRpVXNlcicsICcxLjEuNjAnLCB7XG4gICAgaXNNdWx0aVVzZXI6IHRydWUsXG4gIH0pO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTVlTUUxfUk9UQVRJT05fU0lOR0xFX1VTRVIgPSBuZXcgU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbignU2VjcmV0c01hbmFnZXJSRFNNeVNRTFJvdGF0aW9uU2luZ2xlVXNlcicsICcxLjEuNjAnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTVlTUUxfUk9UQVRJT05fTVVMVElfVVNFUiA9IG5ldyBTZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uKCdTZWNyZXRzTWFuYWdlclJEU015U1FMUm90YXRpb25NdWx0aVVzZXInLCAnMS4xLjYwJywge1xuICAgIGlzTXVsdGlVc2VyOiB0cnVlLFxuICB9KTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBPUkFDTEVfUk9UQVRJT05fU0lOR0xFX1VTRVIgPSBuZXcgU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbignU2VjcmV0c01hbmFnZXJSRFNPcmFjbGVSb3RhdGlvblNpbmdsZVVzZXInLCAnMS4xLjYwJyk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBPUkFDTEVfUk9UQVRJT05fTVVMVElfVVNFUiA9IG5ldyBTZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uKCdTZWNyZXRzTWFuYWdlclJEU09yYWNsZVJvdGF0aW9uTXVsdGlVc2VyJywgJzEuMS42MCcsIHtcbiAgICBpc011bHRpVXNlcjogdHJ1ZSxcbiAgfSk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFBPU1RHUkVTX1JPVEFUSU9OX1NJTkdMRV9VU0VSID0gbmV3IFNlY3JldFJvdGF0aW9uQXBwbGljYXRpb24oJ1NlY3JldHNNYW5hZ2VyUkRTUG9zdGdyZVNRTFJvdGF0aW9uU2luZ2xlVXNlcicsICcxLjEuNjAnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBQT1NUR1JFU19ST1RBVElPTl9NVUxUSV9VU0VSID0gbmV3IFNlY3JldFJvdGF0aW9uQXBwbGljYXRpb24oJ1NlY3JldHNNYW5hZ2VyUkRTUG9zdGdyZVNRTFJvdGF0aW9uTXVsdGlVc2VyJywgJzEuMS42MCcsIHtcbiAgICBpc011bHRpVXNlcjogdHJ1ZSxcbiAgfSk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNRTFNFUlZFUl9ST1RBVElPTl9TSU5HTEVfVVNFUiA9IG5ldyBTZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uKCdTZWNyZXRzTWFuYWdlclJEU1NRTFNlcnZlclJvdGF0aW9uU2luZ2xlVXNlcicsICcxLjEuNjAnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBTUUxTRVJWRVJfUk9UQVRJT05fTVVMVElfVVNFUiA9IG5ldyBTZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uKCdTZWNyZXRzTWFuYWdlclJEU1NRTFNlcnZlclJvdGF0aW9uTXVsdGlVc2VyJywgJzEuMS42MCcsIHtcbiAgICBpc011bHRpVXNlcjogdHJ1ZSxcbiAgfSk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBSRURTSElGVF9ST1RBVElPTl9TSU5HTEVfVVNFUiA9IG5ldyBTZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uKCdTZWNyZXRzTWFuYWdlclJlZHNoaWZ0Um90YXRpb25TaW5nbGVVc2VyJywgJzEuMS42MCcpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBSRURTSElGVF9ST1RBVElPTl9NVUxUSV9VU0VSID0gbmV3IFNlY3JldFJvdGF0aW9uQXBwbGljYXRpb24oJ1NlY3JldHNNYW5hZ2VyUmVkc2hpZnRSb3RhdGlvbk11bHRpVXNlcicsICcxLjEuNjAnLCB7XG4gICAgaXNNdWx0aVVzZXI6IHRydWUsXG4gIH0pO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IE1PTkdPREJfUk9UQVRJT05fU0lOR0xFX1VTRVIgPSBuZXcgU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbignU2VjcmV0c01hbmFnZXJNb25nb0RCUm90YXRpb25TaW5nbGVVc2VyJywgJzEuMS42MCcpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTU9OR09EQl9ST1RBVElPTl9NVUxUSV9VU0VSID0gbmV3IFNlY3JldFJvdGF0aW9uQXBwbGljYXRpb24oJ1NlY3JldHNNYW5hZ2VyTW9uZ29EQlJvdGF0aW9uTXVsdGlVc2VyJywgJzEuMS42MCcsIHtcbiAgICBpc011bHRpVXNlcjogdHJ1ZSxcbiAgfSk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBhcHBsaWNhdGlvbklkOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBzZW1hbnRpY1ZlcnNpb246IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgaXNNdWx0aVVzZXI/OiBib29sZWFuO1xuXG4gIGNvbnN0cnVjdG9yKGFwcGxpY2F0aW9uSWQ6IHN0cmluZywgc2VtYW50aWNWZXJzaW9uOiBzdHJpbmcsIG9wdGlvbnM/OiBTZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uT3B0aW9ucykge1xuICAgIHRoaXMuYXBwbGljYXRpb25JZCA9IGBhcm46YXdzOnNlcnZlcmxlc3NyZXBvOnVzLWVhc3QtMToyOTczNTYyMjc4MjQ6YXBwbGljYXRpb25zLyR7YXBwbGljYXRpb25JZH1gO1xuICAgIHRoaXMuc2VtYW50aWNWZXJzaW9uID0gc2VtYW50aWNWZXJzaW9uO1xuICAgIHRoaXMuaXNNdWx0aVVzZXIgPSBvcHRpb25zICYmIG9wdGlvbnMuaXNNdWx0aVVzZXI7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgU2VjcmV0Um90YXRpb25Qcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzZWNyZXQ6IElTZWNyZXQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1hc3RlclNlY3JldD86IElTZWNyZXQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGF1dG9tYXRpY2FsbHlBZnRlcj86IER1cmF0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFwcGxpY2F0aW9uOiBTZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZwY1N1Ym5ldHM/OiBlYzIuU3VibmV0U2VsZWN0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0YXJnZXQ6IGVjMi5JQ29ubmVjdGFibGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGV4Y2x1ZGVDaGFyYWN0ZXJzPzogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgU2VjcmV0Um90YXRpb24gZXh0ZW5kcyBDb3JlQ29uc3RydWN0IHtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFNlY3JldFJvdGF0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgaWYgKCFwcm9wcy50YXJnZXQuY29ubmVjdGlvbnMuZGVmYXVsdFBvcnQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVGhlIGB0YXJnZXRgIGNvbm5lY3Rpb25zIG11c3QgaGF2ZSBhIGRlZmF1bHQgcG9ydCByYW5nZS4nKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuYXBwbGljYXRpb24uaXNNdWx0aVVzZXIgJiYgIXByb3BzLm1hc3RlclNlY3JldCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgYG1hc3RlclNlY3JldGAgbXVzdCBiZSBzcGVjaWZpZWQgZm9yIGFwcGxpY2F0aW9uIHVzaW5nIHRoZSBtdWx0aSB1c2VyIHNjaGVtZS4nKTtcbiAgICB9XG5cbiAgICAvLyBNYXggbGVuZ3RoIG9mIDY0IGNoYXJzLCBnZXQgdGhlIGxhc3QgNjQgY2hhcnNcbiAgICBjb25zdCB1bmlxdWVJZCA9IE5hbWVzLnVuaXF1ZUlkKHRoaXMpO1xuICAgIGNvbnN0IHJvdGF0aW9uRnVuY3Rpb25OYW1lID0gdW5pcXVlSWQuc3Vic3RyaW5nKE1hdGgubWF4KHVuaXF1ZUlkLmxlbmd0aCAtIDY0LCAwKSwgdW5pcXVlSWQubGVuZ3RoKTtcblxuICAgIGNvbnN0IHNlY3VyaXR5R3JvdXAgPSBwcm9wcy5zZWN1cml0eUdyb3VwIHx8IG5ldyBlYzIuU2VjdXJpdHlHcm91cCh0aGlzLCAnU2VjdXJpdHlHcm91cCcsIHtcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgIH0pO1xuICAgIHByb3BzLnRhcmdldC5jb25uZWN0aW9ucy5hbGxvd0RlZmF1bHRQb3J0RnJvbShzZWN1cml0eUdyb3VwKTtcblxuICAgIGNvbnN0IHBhcmFtZXRlcnM6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH0gPSB7XG4gICAgICBlbmRwb2ludDogYGh0dHBzOi8vc2VjcmV0c21hbmFnZXIuJHtTdGFjay5vZih0aGlzKS5yZWdpb259LiR7U3RhY2sub2YodGhpcykudXJsU3VmZml4fWAsXG4gICAgICBmdW5jdGlvbk5hbWU6IHJvdGF0aW9uRnVuY3Rpb25OYW1lLFxuICAgICAgdnBjU3VibmV0SWRzOiBwcm9wcy52cGMuc2VsZWN0U3VibmV0cyhwcm9wcy52cGNTdWJuZXRzKS5zdWJuZXRJZHMuam9pbignLCcpLFxuICAgICAgdnBjU2VjdXJpdHlHcm91cElkczogc2VjdXJpdHlHcm91cC5zZWN1cml0eUdyb3VwSWQsXG4gICAgfTtcblxuICAgIGlmIChwcm9wcy5leGNsdWRlQ2hhcmFjdGVycyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBwYXJhbWV0ZXJzLmV4Y2x1ZGVDaGFyYWN0ZXJzID0gcHJvcHMuZXhjbHVkZUNoYXJhY3RlcnM7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnNlY3JldC5lbmNyeXB0aW9uS2V5KSB7XG4gICAgICBwYXJhbWV0ZXJzLmttc0tleUFybiA9IHByb3BzLnNlY3JldC5lbmNyeXB0aW9uS2V5LmtleUFybjtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMubWFzdGVyU2VjcmV0KSB7XG4gICAgICBwYXJhbWV0ZXJzLm1hc3RlclNlY3JldEFybiA9IHByb3BzLm1hc3RlclNlY3JldC5zZWNyZXRBcm47XG5cbiAgICAgIGlmIChwcm9wcy5tYXN0ZXJTZWNyZXQuZW5jcnlwdGlvbktleSkge1xuICAgICAgICBwYXJhbWV0ZXJzLm1hc3RlclNlY3JldEttc0tleUFybiA9IHByb3BzLm1hc3RlclNlY3JldC5lbmNyeXB0aW9uS2V5LmtleUFybjtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBzZXJ2ZXJsZXNzLkNmbkFwcGxpY2F0aW9uKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGxvY2F0aW9uOiBwcm9wcy5hcHBsaWNhdGlvbixcbiAgICAgIHBhcmFtZXRlcnMsXG4gICAgfSk7XG5cbiAgICAvLyBUaGlzIGNyZWF0ZXMgYSBDRiBhIGRlcGVuZGVuY3kgYmV0d2VlbiB0aGUgcm90YXRpb24gc2NoZWR1bGUgYW5kIHRoZVxuICAgIC8vIHNlcnZlcmxlc3MgYXBwbGljYXRpb24uIFRoaXMgaXMgbmVlZGVkIGJlY2F1c2UgaXQncyB0aGUgYXBwbGljYXRpb25cbiAgICAvLyB0aGF0IGNyZWF0ZXMgdGhlIExhbWJkYSBwZXJtaXNzaW9uIHRvIGludm9rZSB0aGUgZnVuY3Rpb24uXG4gICAgLy8gU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWNyZXRzbWFuYWdlci9sYXRlc3QvdXNlcmd1aWRlL2ludGVncmF0aW5nX2Nsb3VkZm9ybWF0aW9uLmh0bWxcbiAgICBjb25zdCByb3RhdGlvbkxhbWJkYSA9IGxhbWJkYS5GdW5jdGlvbi5mcm9tRnVuY3Rpb25Bcm4odGhpcywgJ1JvdGF0aW9uTGFtYmRhJywgVG9rZW4uYXNTdHJpbmcoYXBwbGljYXRpb24uZ2V0QXR0KCdPdXRwdXRzLlJvdGF0aW9uTGFtYmRhQVJOJykpKTtcblxuICAgIHByb3BzLnNlY3JldC5hZGRSb3RhdGlvblNjaGVkdWxlKCdSb3RhdGlvblNjaGVkdWxlJywge1xuICAgICAgcm90YXRpb25MYW1iZGEsXG4gICAgICBhdXRvbWF0aWNhbGx5QWZ0ZXI6IHByb3BzLmF1dG9tYXRpY2FsbHlBZnRlcixcbiAgICB9KTtcblxuICAgIC8vIFByZXZlbnQgbWFzdGVyIHNlY3JldCBkZWxldGlvbiB3aGVuIHJvdGF0aW9uIGlzIGluIHBsYWNlXG4gICAgaWYgKHByb3BzLm1hc3RlclNlY3JldCkge1xuICAgICAgcHJvcHMubWFzdGVyU2VjcmV0LmRlbnlBY2NvdW50Um9vdERlbGV0ZSgpO1xuICAgIH1cbiAgfVxufVxuIl19