"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const acm = require("@aws-cdk/aws-certificatemanager");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const test_origin_1 = require("./test-origin");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
test('minimal example renders correctly', () => {
    const origin = test_origin_1.defaultOrigin();
    new lib_1.Distribution(stack, 'MyDist', { defaultBehavior: { origin } });
    expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                ForwardedValues: { QueryString: false },
                TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                ViewerProtocolPolicy: 'allow-all',
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Origins: [{
                    DomainName: 'www.example.com',
                    Id: 'StackMyDistOrigin1D6D5E535',
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only',
                    },
                }],
        },
    });
});
test('exhaustive example of props renders correctly', () => {
    const origin = test_origin_1.defaultOrigin();
    const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
    new lib_1.Distribution(stack, 'MyDist', {
        defaultBehavior: { origin },
        certificate,
        comment: 'a test',
        defaultRootObject: 'index.html',
        domainNames: ['example.com'],
        enabled: false,
        enableIpv6: false,
        enableLogging: true,
        geoRestriction: lib_1.GeoRestriction.blacklist('US', 'GB'),
        httpVersion: lib_1.HttpVersion.HTTP1_1,
        logFilePrefix: 'logs/',
        logIncludesCookies: true,
        priceClass: lib_1.PriceClass.PRICE_CLASS_100,
        webAclId: '473e64fd-f30b-4765-81a0-62ad96dd167a',
    });
    expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: ['example.com'],
            DefaultCacheBehavior: {
                ForwardedValues: { QueryString: false },
                TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                ViewerProtocolPolicy: 'allow-all',
            },
            Comment: 'a test',
            DefaultRootObject: 'index.html',
            Enabled: false,
            HttpVersion: 'http1.1',
            IPV6Enabled: false,
            Logging: {
                Bucket: { 'Fn::GetAtt': ['MyDistLoggingBucket9B8976BC', 'RegionalDomainName'] },
                IncludeCookies: true,
                Prefix: 'logs/',
            },
            Origins: [{
                    DomainName: 'www.example.com',
                    Id: 'StackMyDistOrigin1D6D5E535',
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only',
                    },
                }],
            PriceClass: 'PriceClass_100',
            Restrictions: {
                GeoRestriction: {
                    Locations: ['US', 'GB'],
                    RestrictionType: 'blacklist',
                },
            },
            ViewerCertificate: {
                AcmCertificateArn: 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012',
                SslSupportMethod: 'sni-only',
                MinimumProtocolVersion: 'TLSv1.2_2019',
            },
            WebACLId: '473e64fd-f30b-4765-81a0-62ad96dd167a',
        },
    });
});
describe('multiple behaviors', () => {
    test('a second behavior can\'t be specified with the catch-all path pattern', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: { origin },
                additionalBehaviors: {
                    '*': { origin },
                },
            });
        }).toThrow(/Only the default behavior can have a path pattern of \'*\'/);
    });
    test('a second behavior can be added to the original origin', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                HttpVersion: 'http2',
                IPV6Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
    test('a second behavior can be added to a secondary origin', () => {
        const origin = test_origin_1.defaultOrigin();
        const origin2 = test_origin_1.defaultOrigin('origin2.example.com');
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin: origin2 },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                HttpVersion: 'http2',
                IPV6Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    },
                    {
                        DomainName: 'origin2.example.com',
                        Id: 'StackMyDistOrigin20B96F3AD',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
    test('behavior creation order is preserved', () => {
        const origin = test_origin_1.defaultOrigin();
        const origin2 = test_origin_1.defaultOrigin('origin2.example.com');
        const dist = new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/1*': { origin: origin2 },
            },
        });
        dist.addBehavior('api/2*', origin);
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/1*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    },
                    {
                        PathPattern: 'api/2*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                HttpVersion: 'http2',
                IPV6Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    },
                    {
                        DomainName: 'origin2.example.com',
                        Id: 'StackMyDistOrigin20B96F3AD',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
});
describe('certificates', () => {
    test('should fail if using an imported certificate from outside of us-east-1', () => {
        const origin = test_origin_1.defaultOrigin();
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:eu-west-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                certificate,
            });
        }).toThrow(/Distribution certificates must be in the us-east-1 region and the certificate you provided is in eu-west-1./);
    });
    test('adding a certificate without a domain name throws', () => {
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        expect(() => {
            new lib_1.Distribution(stack, 'Dist1', {
                defaultBehavior: { origin: test_origin_1.defaultOrigin() },
                certificate,
            });
        }).toThrow(/Must specify at least one domain name/);
        expect(() => {
            new lib_1.Distribution(stack, 'Dist2', {
                defaultBehavior: { origin: test_origin_1.defaultOrigin() },
                domainNames: [],
                certificate,
            });
        }).toThrow(/Must specify at least one domain name/);
    });
    test('adding a certificate and domain renders the correct ViewerCertificate and Aliases property', () => {
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin: test_origin_1.defaultOrigin() },
            domainNames: ['example.com', 'www.example.com'],
            certificate,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Aliases: ['example.com', 'www.example.com'],
                ViewerCertificate: {
                    AcmCertificateArn: 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012',
                    SslSupportMethod: 'sni-only',
                    MinimumProtocolVersion: 'TLSv1.2_2019',
                },
            },
        });
    });
});
describe('custom error responses', () => {
    test('should fail if responsePagePath is defined but responseCode is not', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{
                        httpStatus: 404,
                        responsePagePath: '/errors/404.html',
                    }],
            });
        }).toThrow(/\'responseCode\' must be provided if \'responsePagePath\' is defined/);
    });
    test('should fail if only the error code is provided', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{ httpStatus: 404 }],
            });
        }).toThrow(/A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid./);
    });
    test('should render the array of error configs if provided', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin },
            errorResponses: [{
                    httpStatus: 404,
                    responseHttpStatus: 404,
                    responsePagePath: '/errors/404.html',
                },
                {
                    httpStatus: 500,
                    ttl: core_1.Duration.seconds(2),
                }],
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                CustomErrorResponses: [
                    {
                        ErrorCode: 404,
                        ResponseCode: 404,
                        ResponsePagePath: '/errors/404.html',
                    },
                    {
                        ErrorCachingMinTTL: 2,
                        ErrorCode: 500,
                    },
                ],
            },
        });
    });
});
describe('logging', () => {
    test('does not include logging if disabled and no bucket provided', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', { defaultBehavior: { origin } });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: assert_1.ABSENT,
            },
        });
    });
    test('throws error if logging disabled but bucket provided', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: { origin },
                enableLogging: false,
                logBucket: new s3.Bucket(stack, 'Bucket'),
            });
        }).toThrow(/Explicitly disabled logging but provided a logging bucket./);
    });
    test('creates bucket if none is provided', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            enableLogging: true,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: {
                    Bucket: { 'Fn::GetAtt': ['MyDistLoggingBucket9B8976BC', 'RegionalDomainName'] },
                },
            },
        });
    });
    test('uses existing bucket if provided', () => {
        const origin = test_origin_1.defaultOrigin();
        const loggingBucket = new s3.Bucket(stack, 'MyLoggingBucket');
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            logBucket: loggingBucket,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: {
                    Bucket: { 'Fn::GetAtt': ['MyLoggingBucket4382CD04', 'RegionalDomainName'] },
                },
            },
        });
    });
    test('can set prefix and cookies', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            enableLogging: true,
            logFilePrefix: 'logs/',
            logIncludesCookies: true,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: {
                    Bucket: { 'Fn::GetAtt': ['MyDistLoggingBucket9B8976BC', 'RegionalDomainName'] },
                    IncludeCookies: true,
                    Prefix: 'logs/',
                },
            },
        });
    });
});
describe('with Lambda@Edge functions', () => {
    let lambdaFunction;
    let origin;
    beforeEach(() => {
        lambdaFunction = new lambda.Function(stack, 'Function', {
            runtime: lambda.Runtime.NODEJS,
            code: lambda.Code.fromInline('whatever'),
            handler: 'index.handler',
        });
        origin = test_origin_1.defaultOrigin();
    });
    test('can add an edge lambdas to the default behavior', () => {
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: lambdaFunction.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                    },
                ],
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    LambdaFunctionAssociations: [
                        {
                            EventType: 'origin-request',
                            LambdaFunctionARN: {
                                Ref: 'FunctionCurrentVersion4E2B2261477a5ae8059bbaa7813f752292c0f65e',
                            },
                        },
                    ],
                },
            },
        });
    });
    test('can add an edge lambdas to additional behaviors', () => {
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'images/*': {
                    origin,
                    edgeLambdas: [
                        {
                            functionVersion: lambdaFunction.currentVersion,
                            eventType: lib_1.LambdaEdgeEventType.VIEWER_REQUEST,
                        },
                    ],
                },
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                CacheBehaviors: [
                    {
                        PathPattern: 'images/*',
                        LambdaFunctionAssociations: [
                            {
                                EventType: 'viewer-request',
                                LambdaFunctionARN: {
                                    Ref: 'FunctionCurrentVersion4E2B2261477a5ae8059bbaa7813f752292c0f65e',
                                },
                            },
                        ],
                    },
                ],
            },
        });
    });
    test('fails creation when attempting to add the $LATEST function version as an edge Lambda to the default behavior', () => {
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: {
                    origin,
                    edgeLambdas: [
                        {
                            functionVersion: lambdaFunction.latestVersion,
                            eventType: lib_1.LambdaEdgeEventType.ORIGIN_RESPONSE,
                        },
                    ],
                },
            });
        }).toThrow(/\$LATEST function version cannot be used for Lambda@Edge/);
    });
    test('with removable env vars', () => {
        const envLambdaFunction = new lambda.Function(stack, 'EnvFunction', {
            runtime: lambda.Runtime.NODEJS,
            code: lambda.Code.fromInline('whateverwithenv'),
            handler: 'index.handler',
        });
        envLambdaFunction.addEnvironment('KEY', 'value', { removeInEdge: true });
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: envLambdaFunction.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                    },
                ],
            },
        });
        expect(stack).toHaveResource('AWS::Lambda::Function', {
            Environment: assert_1.ABSENT,
            Code: {
                ZipFile: 'whateverwithenv',
            },
        });
    });
    test('with incompatible env vars', () => {
        const envLambdaFunction = new lambda.Function(stack, 'EnvFunction', {
            runtime: lambda.Runtime.NODEJS,
            code: lambda.Code.fromInline('whateverwithenv'),
            handler: 'index.handler',
            environment: {
                KEY: 'value',
            },
        });
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: envLambdaFunction.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                    },
                ],
            },
        });
        expect(() => app.synth()).toThrow(/KEY/);
    });
    test('with singleton function', () => {
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: 'singleton-for-cloudfront',
            runtime: lambda.Runtime.NODEJS_12_X,
            code: lambda.Code.fromInline('code'),
            handler: 'index.handler',
        });
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: singleton.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                    },
                ],
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    LambdaFunctionAssociations: [
                        {
                            EventType: 'origin-request',
                            LambdaFunctionARN: {
                                Ref: 'SingletonLambdasingletonforcloudfrontCurrentVersion0078406348a0962a52448a200cd0dbc0e22edb2a',
                            },
                        },
                    ],
                },
            },
        });
    });
});
test('price class is included if provided', () => {
    const origin = test_origin_1.defaultOrigin();
    new lib_1.Distribution(stack, 'Dist', {
        defaultBehavior: { origin },
        priceClass: lib_1.PriceClass.PRICE_CLASS_200,
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            PriceClass: 'PriceClass_200',
        },
    });
});
test('escape hatches are supported', () => {
    const dist = new lib_1.Distribution(stack, 'Dist', {
        defaultBehavior: { origin: test_origin_1.defaultOrigin },
    });
    const cfnDist = dist.node.defaultChild;
    cfnDist.addPropertyOverride('DistributionConfig.DefaultCacheBehavior.ForwardedValues.Headers', ['*']);
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                ForwardedValues: {
                    Headers: ['*'],
                },
            },
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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