"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontWebDistribution = exports.ViewerCertificate = exports.CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedMethods = exports.OriginSslPolicy = exports.FailoverStatusCode = void 0;
const certificatemanager = require("@aws-cdk/aws-certificatemanager");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
const distribution_1 = require("./distribution");
/**
 * HTTP status code to failover to second origin
 */
var FailoverStatusCode;
(function (FailoverStatusCode) {
    /**
     * Forbidden (403)
     */
    FailoverStatusCode[FailoverStatusCode["FORBIDDEN"] = 403] = "FORBIDDEN";
    /**
     * Not found (404)
     */
    FailoverStatusCode[FailoverStatusCode["NOT_FOUND"] = 404] = "NOT_FOUND";
    /**
     * Internal Server Error (500)
     */
    FailoverStatusCode[FailoverStatusCode["INTERNAL_SERVER_ERROR"] = 500] = "INTERNAL_SERVER_ERROR";
    /**
     * Bad Gateway (502)
     */
    FailoverStatusCode[FailoverStatusCode["BAD_GATEWAY"] = 502] = "BAD_GATEWAY";
    /**
     * Service Unavailable (503)
     */
    FailoverStatusCode[FailoverStatusCode["SERVICE_UNAVAILABLE"] = 503] = "SERVICE_UNAVAILABLE";
    /**
     * Gateway Timeout (504)
     */
    FailoverStatusCode[FailoverStatusCode["GATEWAY_TIMEOUT"] = 504] = "GATEWAY_TIMEOUT";
})(FailoverStatusCode = exports.FailoverStatusCode || (exports.FailoverStatusCode = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFrontWebDistribution } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontWebDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior: true}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 * @resource AWS::CloudFront::Distribution
 */
class CloudFrontWebDistribution extends cdk.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [distribution_1.SSLMethod.SNI]: [
                distribution_1.SecurityPolicyProtocol.TLS_V1, distribution_1.SecurityPolicyProtocol.TLS_V1_1_2016,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2016, distribution_1.SecurityPolicyProtocol.TLS_V1_2_2018,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2_2019,
            ],
            [distribution_1.SSLMethod.VIP]: [distribution_1.SecurityPolicyProtocol.SSL_V3, distribution_1.SecurityPolicyProtocol.TLS_V1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : 'index.html',
            httpVersion: props.httpVersion || distribution_1.HttpVersion.HTTP2,
            priceClass: props.priceClass || distribution_1.PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // eslint-disable-next-line max-len
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        const originGroups = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            let originId = `origin${originIndex}`;
            const originProperty = this.toOriginProperty(originConfig, originId);
            if (originConfig.failoverCustomOriginSource || originConfig.failoverS3OriginSource) {
                const originSecondaryId = `originSecondary${originIndex}`;
                const originSecondaryProperty = this.toOriginProperty({
                    s3OriginSource: originConfig.failoverS3OriginSource,
                    customOriginSource: originConfig.failoverCustomOriginSource,
                    originPath: originConfig.originPath,
                    originHeaders: originConfig.originHeaders,
                }, originSecondaryId);
                const originGroupsId = `OriginGroup${originIndex}`;
                const failoverCodes = (_a = originConfig.failoverCriteriaStatusCodes) !== null && _a !== void 0 ? _a : [500, 502, 503, 504];
                originGroups.push({
                    id: originGroupsId,
                    members: {
                        items: [{ originId }, { originId: originSecondaryId }],
                        quantity: 2,
                    },
                    failoverCriteria: {
                        statusCodes: {
                            items: failoverCodes,
                            quantity: failoverCodes.length,
                        },
                    },
                });
                originId = originGroupsId;
                origins.push(originSecondaryProperty);
            }
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        const originGroupsDistConfig = originGroups.length > 0
            ? {
                items: originGroups,
                quantity: originGroups.length,
            }
            : undefined;
        distributionConfig = {
            ...distributionConfig,
            origins,
            originGroups: originGroupsDistConfig,
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // eslint-disable-next-line max-len
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = {
                ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket');
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketRegionalDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        if (props.geoRestriction) {
            distributionConfig = {
                ...distributionConfig,
                restrictions: {
                    geoRestriction: {
                        restrictionType: props.geoRestriction.restrictionType,
                        locations: props.geoRestriction.locations,
                    },
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || distribution_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.edgeArn,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
    toOriginProperty(originConfig, originId) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        if (!originConfig.s3OriginSource &&
            !originConfig.customOriginSource) {
            throw new Error('There must be at least one origin source - either an s3OriginSource, a customOriginSource');
        }
        if (originConfig.customOriginSource && originConfig.s3OriginSource) {
            throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
        }
        if ([
            originConfig.originHeaders,
            (_a = originConfig.s3OriginSource) === null || _a === void 0 ? void 0 : _a.originHeaders,
            (_b = originConfig.customOriginSource) === null || _b === void 0 ? void 0 : _b.originHeaders,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originHeaders field allowed across origin and failover origins');
        }
        if ([
            originConfig.originPath,
            (_c = originConfig.s3OriginSource) === null || _c === void 0 ? void 0 : _c.originPath,
            (_d = originConfig.customOriginSource) === null || _d === void 0 ? void 0 : _d.originPath,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originPath field allowed across origin and failover origins');
        }
        const headers = (_g = (_e = originConfig.originHeaders) !== null && _e !== void 0 ? _e : (_f = originConfig.s3OriginSource) === null || _f === void 0 ? void 0 : _f.originHeaders) !== null && _g !== void 0 ? _g : (_h = originConfig.customOriginSource) === null || _h === void 0 ? void 0 : _h.originHeaders;
        const originHeaders = [];
        if (headers) {
            Object.keys(headers).forEach((key) => {
                const oHeader = {
                    headerName: key,
                    headerValue: headers[key],
                };
                originHeaders.push(oHeader);
            });
        }
        let s3OriginConfig;
        if (originConfig.s3OriginSource) {
            // first case for backwards compatibility
            if (originConfig.s3OriginSource.originAccessIdentity) {
                // grant CloudFront OriginAccessIdentity read access to S3 bucket
                originConfig.s3OriginSource.s3BucketSource.grantRead(originConfig.s3OriginSource.originAccessIdentity);
                s3OriginConfig = {
                    originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`,
                };
            }
            else {
                s3OriginConfig = {};
            }
        }
        const connectionAttempts = (_j = originConfig.connectionAttempts) !== null && _j !== void 0 ? _j : 3;
        if (connectionAttempts < 1 || 3 < connectionAttempts || !Number.isInteger(connectionAttempts)) {
            throw new Error('connectionAttempts: You can specify 1, 2, or 3 as the number of attempts.');
        }
        const connectionTimeout = (originConfig.connectionTimeout || cdk.Duration.seconds(10)).toSeconds();
        if (connectionTimeout < 1 || 10 < connectionTimeout || !Number.isInteger(connectionTimeout)) {
            throw new Error('connectionTimeout: You can specify a number of seconds between 1 and 10 (inclusive).');
        }
        const originProperty = {
            id: originId,
            domainName: originConfig.s3OriginSource
                ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                : originConfig.customOriginSource.domainName,
            originPath: (_m = (_k = originConfig.originPath) !== null && _k !== void 0 ? _k : (_l = originConfig.customOriginSource) === null || _l === void 0 ? void 0 : _l.originPath) !== null && _m !== void 0 ? _m : (_o = originConfig.s3OriginSource) === null || _o === void 0 ? void 0 : _o.originPath,
            originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
            s3OriginConfig,
            customOriginConfig: originConfig.customOriginSource
                ? {
                    httpPort: originConfig.customOriginSource.httpPort || 80,
                    httpsPort: originConfig.customOriginSource.httpsPort || 443,
                    originKeepaliveTimeout: (originConfig.customOriginSource.originKeepaliveTimeout &&
                        originConfig.customOriginSource.originKeepaliveTimeout.toSeconds()) ||
                        5,
                    originReadTimeout: (originConfig.customOriginSource.originReadTimeout &&
                        originConfig.customOriginSource.originReadTimeout.toSeconds()) ||
                        30,
                    originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy ||
                        distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
                    originSslProtocols: originConfig.customOriginSource
                        .allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                }
                : undefined,
            connectionAttempts,
            connectionTimeout,
        };
        return originProperty;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,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