"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginAccessIdentity = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
class OriginAccessIdentityBase extends cdk.Resource {
    /**
     * The ARN to include in S3 bucket policy to allow CloudFront access
     */
    arn() {
        return cdk.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            account: 'cloudfront',
            resource: 'user',
            resourceName: `CloudFront Origin Access Identity ${this.originAccessIdentityName}`,
        });
    }
}
/**
 * An origin access identity is a special CloudFront user that you can
 * associate with Amazon S3 origins, so that you can secure all or just some of
 * your Amazon S3 content.
 *
 * @resource AWS::CloudFront::CloudFrontOriginAccessIdentity
 */
class OriginAccessIdentity extends OriginAccessIdentityBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new cloudfront_generated_1.CfnCloudFrontOriginAccessIdentity(this, 'Resource', {
            cloudFrontOriginAccessIdentityConfig: {
                comment: (props && props.comment) || 'Allows CloudFront to reach the bucket',
            },
        });
        // physical id - OAI name
        this.originAccessIdentityName = this.getResourceNameAttribute(this.resource.ref);
        // Canonical user to whitelist in S3 Bucket Policy
        this.cloudFrontOriginAccessIdentityS3CanonicalUserId = this.resource.attrS3CanonicalUserId;
        // The principal for must be either the canonical user or a special ARN
        // with the CloudFront Origin Access Id (see `arn()` method). For
        // import/export the OAI is anyway required so the principal is constructed
        // with it. But for the normal case the S3 Canonical User as a nicer
        // interface and does not require constructing the ARN.
        this.grantPrincipal = new iam.CanonicalUserPrincipal(this.cloudFrontOriginAccessIdentityS3CanonicalUserId);
    }
    /**
     * Creates a OriginAccessIdentity by providing the OriginAccessIdentityName
     */
    static fromOriginAccessIdentityName(scope, id, originAccessIdentityName) {
        class Import extends OriginAccessIdentityBase {
            constructor(s, i) {
                super(s, i, { physicalName: originAccessIdentityName });
                this.originAccessIdentityName = originAccessIdentityName;
                this.grantPrincipal = new iam.ArnPrincipal(this.arn());
            }
        }
        return new Import(scope, id);
    }
}
exports.OriginAccessIdentity = OriginAccessIdentity;
//# sourceMappingURL=data:application/json;base64,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