"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginBase = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
class OriginBase {
    constructor(domainName, props = {}) {
        var _a;
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_a = props.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(_scope, options) {
        var _a;
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            originProperty: {
                domainName: this.domainName,
                id: options.originId,
                originPath: this.originPath,
                connectionAttempts: this.connectionAttempts,
                connectionTimeout: (_a = this.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                originCustomHeaders: this.renderCustomHeaders(),
                s3OriginConfig,
                customOriginConfig,
            },
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
}
exports.OriginBase = OriginBase;
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3JpZ2luLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsib3JpZ2luLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUEyRDtBQWtHM0Q7Ozs7O0dBS0c7QUFDSCxNQUFzQixVQUFVO0lBTzlCLFlBQXNCLFVBQWtCLEVBQUUsUUFBcUIsRUFBRTs7UUFDL0QsNkJBQTZCLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxFQUFFLEVBQUUsUUFBRSxLQUFLLENBQUMsaUJBQWlCLDBDQUFFLFNBQVMsR0FBRyxDQUFDO1FBQ2hHLDZCQUE2QixDQUFDLG9CQUFvQixFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRTNGLElBQUksQ0FBQyxVQUFVLEdBQUcsVUFBVSxDQUFDO1FBQzdCLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUM1RCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1FBQ2pELElBQUksQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7UUFDbkQsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO0lBQzNDLENBQUM7SUFFRDs7T0FFRztJQUNJLElBQUksQ0FBQyxNQUFpQixFQUFFLE9BQTBCOztRQUN2RCxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsb0JBQW9CLEVBQUUsQ0FBQztRQUNuRCxNQUFNLGtCQUFrQixHQUFHLElBQUksQ0FBQyx3QkFBd0IsRUFBRSxDQUFDO1FBRTNELElBQUksQ0FBQyxjQUFjLElBQUksQ0FBQyxrQkFBa0IsRUFBRTtZQUMxQyxNQUFNLElBQUksS0FBSyxDQUFDLGdGQUFnRixDQUFDLENBQUM7U0FDbkc7UUFFRCxPQUFPO1lBQ0wsY0FBYyxFQUFFO2dCQUNkLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtnQkFDM0IsRUFBRSxFQUFFLE9BQU8sQ0FBQyxRQUFRO2dCQUNwQixVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7Z0JBQzNCLGtCQUFrQixFQUFFLElBQUksQ0FBQyxrQkFBa0I7Z0JBQzNDLGlCQUFpQixRQUFFLElBQUksQ0FBQyxpQkFBaUIsMENBQUUsU0FBUyxFQUFFO2dCQUN0RCxtQkFBbUIsRUFBRSxJQUFJLENBQUMsbUJBQW1CLEVBQUU7Z0JBQy9DLGNBQWM7Z0JBQ2Qsa0JBQWtCO2FBQ25CO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRCx5REFBeUQ7SUFDL0Msb0JBQW9CO1FBQzVCLE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7SUFFRCw2REFBNkQ7SUFDbkQsd0JBQXdCO1FBQ2hDLE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7SUFFTyxtQkFBbUI7UUFDekIsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUFFLE9BQU8sU0FBUyxDQUFDO1NBQUU7UUFDakcsT0FBTyxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLFVBQVUsRUFBRSxXQUFXLENBQUMsRUFBRSxFQUFFO1lBQzFFLE9BQU8sRUFBRSxVQUFVLEVBQUUsV0FBVyxFQUFFLENBQUM7UUFDckMsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ssa0JBQWtCLENBQUMsVUFBbUI7UUFDNUMsSUFBSSxZQUFLLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQUUsT0FBTyxVQUFVLENBQUM7U0FBRTtRQUMxRCxJQUFJLFVBQVUsS0FBSyxTQUFTLEVBQUU7WUFBRSxPQUFPLFNBQVMsQ0FBQztTQUFFO1FBQ25ELElBQUksSUFBSSxHQUFHLFVBQVUsQ0FBQztRQUN0QixJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUFFLElBQUksR0FBRyxHQUFHLEdBQUcsSUFBSSxDQUFDO1NBQUU7UUFDakQsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQUUsSUFBSSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FBRTtRQUNuRSxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7Q0FDRjtBQXhFRCxnQ0F3RUM7QUFFRDs7R0FFRztBQUNILFNBQVMsNkJBQTZCLENBQUMsSUFBWSxFQUFFLEdBQVcsRUFBRSxHQUFXLEVBQUUsS0FBYyxFQUFFLGFBQXNCLElBQUk7SUFDdkgsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1FBQUUsT0FBTztLQUFFO0lBQ3BDLElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLEtBQUssR0FBRyxHQUFHLElBQUksS0FBSyxHQUFHLEdBQUcsRUFBRTtRQUMxRCxNQUFNLE9BQU8sR0FBRyxVQUFVLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBQzdDLE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxJQUFJLDRCQUE0QixHQUFHLFFBQVEsR0FBRyxHQUFHLE9BQU8sMEJBQTBCLEtBQUssR0FBRyxDQUFDLENBQUM7S0FDaEg7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29uc3RydWN0LCBEdXJhdGlvbiwgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENmbkRpc3RyaWJ1dGlvbiB9IGZyb20gJy4vY2xvdWRmcm9udC5nZW5lcmF0ZWQnO1xuXG4vKipcbiAqIFRoZSBmYWlsb3ZlciBjb25maWd1cmF0aW9uIHVzZWQgZm9yIE9yaWdpbiBHcm91cHMsXG4gKiByZXR1cm5lZCBpbiB7QGxpbmsgT3JpZ2luQmluZENvbmZpZy5mYWlsb3ZlckNvbmZpZ30uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgT3JpZ2luRmFpbG92ZXJDb25maWcge1xuICAvKiogVGhlIG9yaWdpbiB0byB1c2UgYXMgdGhlIGZhbGxiYWNrIG9yaWdpbi4gKi9cbiAgcmVhZG9ubHkgZmFpbG92ZXJPcmlnaW46IElPcmlnaW47XG5cbiAgLyoqXG4gICAqIFRoZSBIVFRQIHN0YXR1cyBjb2RlcyBvZiB0aGUgcmVzcG9uc2UgdGhhdCB0cmlnZ2VyIHF1ZXJ5aW5nIHRoZSBmYWlsb3ZlciBPcmlnaW4uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gNTAwLCA1MDIsIDUwMyBhbmQgNTA0XG4gICAqL1xuICByZWFkb25seSBzdGF0dXNDb2Rlcz86IG51bWJlcltdO1xufVxuXG4vKiogVGhlIHN0cnVjdCByZXR1cm5lZCBmcm9tIHtAbGluayBJT3JpZ2luLmJpbmR9LiAqL1xuZXhwb3J0IGludGVyZmFjZSBPcmlnaW5CaW5kQ29uZmlnIHtcbiAgLyoqXG4gICAqIFRoZSBDbG91ZEZvcm1hdGlvbiBPcmlnaW5Qcm9wZXJ0eSBjb25maWd1cmF0aW9uIGZvciB0aGlzIE9yaWdpbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBub3RoaW5nIGlzIHJldHVybmVkXG4gICAqL1xuICByZWFkb25seSBvcmlnaW5Qcm9wZXJ0eT86IENmbkRpc3RyaWJ1dGlvbi5PcmlnaW5Qcm9wZXJ0eTtcblxuICAvKipcbiAgICogVGhlIGZhaWxvdmVyIGNvbmZpZ3VyYXRpb24gZm9yIHRoaXMgT3JpZ2luLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vdGhpbmcgaXMgcmV0dXJuZWRcbiAgICovXG4gIHJlYWRvbmx5IGZhaWxvdmVyQ29uZmlnPzogT3JpZ2luRmFpbG92ZXJDb25maWc7XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyB0aGUgY29uY2VwdCBvZiBhIENsb3VkRnJvbnQgT3JpZ2luLlxuICogWW91IHByb3ZpZGUgb25lIG9yIG1vcmUgb3JpZ2lucyB3aGVuIGNyZWF0aW5nIGEgRGlzdHJpYnV0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElPcmlnaW4ge1xuICAvKipcbiAgICogVGhlIG1ldGhvZCBjYWxsZWQgd2hlbiBhIGdpdmVuIE9yaWdpbiBpcyBhZGRlZFxuICAgKiAoZm9yIHRoZSBmaXJzdCB0aW1lKSB0byBhIERpc3RyaWJ1dGlvbi5cbiAgICovXG4gIGJpbmQoc2NvcGU6IENvbnN0cnVjdCwgb3B0aW9uczogT3JpZ2luQmluZE9wdGlvbnMpOiBPcmlnaW5CaW5kQ29uZmlnO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgdG8gZGVmaW5lIGFuIE9yaWdpbi5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgT3JpZ2luUHJvcHMge1xuICAvKipcbiAgICogQW4gb3B0aW9uYWwgcGF0aCB0aGF0IENsb3VkRnJvbnQgYXBwZW5kcyB0byB0aGUgb3JpZ2luIGRvbWFpbiBuYW1lIHdoZW4gQ2xvdWRGcm9udCByZXF1ZXN0cyBjb250ZW50IGZyb20gdGhlIG9yaWdpbi5cbiAgICogTXVzdCBiZWdpbiwgYnV0IG5vdCBlbmQsIHdpdGggJy8nIChlLmcuLCAnL3Byb2R1Y3Rpb24vaW1hZ2VzJykuXG4gICAqXG4gICAqIEBkZWZhdWx0ICcvJ1xuICAgKi9cbiAgcmVhZG9ubHkgb3JpZ2luUGF0aD86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBzZWNvbmRzIHRoYXQgQ2xvdWRGcm9udCB3YWl0cyB3aGVuIHRyeWluZyB0byBlc3RhYmxpc2ggYSBjb25uZWN0aW9uIHRvIHRoZSBvcmlnaW4uXG4gICAqIFZhbGlkIHZhbHVlcyBhcmUgMS0xMCBzZWNvbmRzLCBpbmNsdXNpdmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IER1cmF0aW9uLnNlY29uZHMoMTApXG4gICAqL1xuICByZWFkb25seSBjb25uZWN0aW9uVGltZW91dD86IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIHRpbWVzIHRoYXQgQ2xvdWRGcm9udCBhdHRlbXB0cyB0byBjb25uZWN0IHRvIHRoZSBvcmlnaW47IHZhbGlkIHZhbHVlcyBhcmUgMSwgMiwgb3IgMyBhdHRlbXB0cy5cbiAgICpcbiAgICogQGRlZmF1bHQgM1xuICAgKi9cbiAgcmVhZG9ubHkgY29ubmVjdGlvbkF0dGVtcHRzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgSFRUUCBoZWFkZXIgbmFtZXMgYW5kIHZhbHVlcyB0aGF0IENsb3VkRnJvbnQgYWRkcyB0byByZXF1ZXN0cyBpdCBzZW5kcyB0byB0aGUgb3JpZ2luLlxuICAgKlxuICAgKiBAZGVmYXVsdCB7fVxuICAgKi9cbiAgcmVhZG9ubHkgY3VzdG9tSGVhZGVycz86IFJlY29yZDxzdHJpbmcsIHN0cmluZz47XG59XG5cbi8qKlxuICogT3B0aW9ucyBwYXNzZWQgdG8gT3JpZ2luLmJpbmQoKS5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgT3JpZ2luQmluZE9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIGlkZW50aWZpZXIgb2YgdGhpcyBPcmlnaW4sXG4gICAqIGFzIGFzc2lnbmVkIGJ5IHRoZSBEaXN0cmlidXRpb24gdGhpcyBPcmlnaW4gaGFzIGJlZW4gdXNlZCBhZGRlZCB0by5cbiAgICovXG4gIHJlYWRvbmx5IG9yaWdpbklkOiBzdHJpbmc7XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIGRpc3RyaWJ1dGlvbiBvcmlnaW4sIHRoYXQgZGVzY3JpYmVzIHRoZSBBbWF6b24gUzMgYnVja2V0LCBIVFRQIHNlcnZlciAoZm9yIGV4YW1wbGUsIGEgd2ViIHNlcnZlciksXG4gKiBBbWF6b24gTWVkaWFTdG9yZSwgb3Igb3RoZXIgc2VydmVyIGZyb20gd2hpY2ggQ2xvdWRGcm9udCBnZXRzIHlvdXIgZmlsZXMuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgT3JpZ2luQmFzZSBpbXBsZW1lbnRzIElPcmlnaW4ge1xuICBwcml2YXRlIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSBvcmlnaW5QYXRoPzogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGNvbm5lY3Rpb25UaW1lb3V0PzogRHVyYXRpb247XG4gIHByaXZhdGUgcmVhZG9ubHkgY29ubmVjdGlvbkF0dGVtcHRzPzogbnVtYmVyO1xuICBwcml2YXRlIHJlYWRvbmx5IGN1c3RvbUhlYWRlcnM/OiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+O1xuXG4gIHByb3RlY3RlZCBjb25zdHJ1Y3Rvcihkb21haW5OYW1lOiBzdHJpbmcsIHByb3BzOiBPcmlnaW5Qcm9wcyA9IHt9KSB7XG4gICAgdmFsaWRhdGVJbnRJblJhbmdlT3JVbmRlZmluZWQoJ2Nvbm5lY3Rpb25UaW1lb3V0JywgMSwgMTAsIHByb3BzLmNvbm5lY3Rpb25UaW1lb3V0Py50b1NlY29uZHMoKSk7XG4gICAgdmFsaWRhdGVJbnRJblJhbmdlT3JVbmRlZmluZWQoJ2Nvbm5lY3Rpb25BdHRlbXB0cycsIDEsIDMsIHByb3BzLmNvbm5lY3Rpb25BdHRlbXB0cywgZmFsc2UpO1xuXG4gICAgdGhpcy5kb21haW5OYW1lID0gZG9tYWluTmFtZTtcbiAgICB0aGlzLm9yaWdpblBhdGggPSB0aGlzLnZhbGlkYXRlT3JpZ2luUGF0aChwcm9wcy5vcmlnaW5QYXRoKTtcbiAgICB0aGlzLmNvbm5lY3Rpb25UaW1lb3V0ID0gcHJvcHMuY29ubmVjdGlvblRpbWVvdXQ7XG4gICAgdGhpcy5jb25uZWN0aW9uQXR0ZW1wdHMgPSBwcm9wcy5jb25uZWN0aW9uQXR0ZW1wdHM7XG4gICAgdGhpcy5jdXN0b21IZWFkZXJzID0gcHJvcHMuY3VzdG9tSGVhZGVycztcbiAgfVxuXG4gIC8qKlxuICAgKiBCaW5kcyB0aGUgb3JpZ2luIHRvIHRoZSBhc3NvY2lhdGVkIERpc3RyaWJ1dGlvbi4gQ2FuIGJlIHVzZWQgdG8gZ3JhbnQgcGVybWlzc2lvbnMsIGNyZWF0ZSBkZXBlbmRlbnQgcmVzb3VyY2VzLCBldGMuXG4gICAqL1xuICBwdWJsaWMgYmluZChfc2NvcGU6IENvbnN0cnVjdCwgb3B0aW9uczogT3JpZ2luQmluZE9wdGlvbnMpOiBPcmlnaW5CaW5kQ29uZmlnIHtcbiAgICBjb25zdCBzM09yaWdpbkNvbmZpZyA9IHRoaXMucmVuZGVyUzNPcmlnaW5Db25maWcoKTtcbiAgICBjb25zdCBjdXN0b21PcmlnaW5Db25maWcgPSB0aGlzLnJlbmRlckN1c3RvbU9yaWdpbkNvbmZpZygpO1xuXG4gICAgaWYgKCFzM09yaWdpbkNvbmZpZyAmJiAhY3VzdG9tT3JpZ2luQ29uZmlnKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1N1YmNsYXNzIG11c3Qgb3ZlcnJpZGUgYW5kIHByb3ZpZGUgZWl0aGVyIHMzT3JpZ2luQ29uZmlnIG9yIGN1c3RvbU9yaWdpbkNvbmZpZycpO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBvcmlnaW5Qcm9wZXJ0eToge1xuICAgICAgICBkb21haW5OYW1lOiB0aGlzLmRvbWFpbk5hbWUsXG4gICAgICAgIGlkOiBvcHRpb25zLm9yaWdpbklkLFxuICAgICAgICBvcmlnaW5QYXRoOiB0aGlzLm9yaWdpblBhdGgsXG4gICAgICAgIGNvbm5lY3Rpb25BdHRlbXB0czogdGhpcy5jb25uZWN0aW9uQXR0ZW1wdHMsXG4gICAgICAgIGNvbm5lY3Rpb25UaW1lb3V0OiB0aGlzLmNvbm5lY3Rpb25UaW1lb3V0Py50b1NlY29uZHMoKSxcbiAgICAgICAgb3JpZ2luQ3VzdG9tSGVhZGVyczogdGhpcy5yZW5kZXJDdXN0b21IZWFkZXJzKCksXG4gICAgICAgIHMzT3JpZ2luQ29uZmlnLFxuICAgICAgICBjdXN0b21PcmlnaW5Db25maWcsXG4gICAgICB9LFxuICAgIH07XG4gIH1cblxuICAvLyBPdmVycmlkZGVuIGJ5IHN1Yi1jbGFzc2VzIHRvIHByb3ZpZGUgUzMgb3JpZ2luIGNvbmZpZy5cbiAgcHJvdGVjdGVkIHJlbmRlclMzT3JpZ2luQ29uZmlnKCk6IENmbkRpc3RyaWJ1dGlvbi5TM09yaWdpbkNvbmZpZ1Byb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG5cbiAgLy8gT3ZlcnJpZGRlbiBieSBzdWItY2xhc3NlcyB0byBwcm92aWRlIGN1c3RvbSBvcmlnaW4gY29uZmlnLlxuICBwcm90ZWN0ZWQgcmVuZGVyQ3VzdG9tT3JpZ2luQ29uZmlnKCk6IENmbkRpc3RyaWJ1dGlvbi5DdXN0b21PcmlnaW5Db25maWdQcm9wZXJ0eSB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyQ3VzdG9tSGVhZGVycygpOiBDZm5EaXN0cmlidXRpb24uT3JpZ2luQ3VzdG9tSGVhZGVyUHJvcGVydHlbXSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKCF0aGlzLmN1c3RvbUhlYWRlcnMgfHwgT2JqZWN0LmVudHJpZXModGhpcy5jdXN0b21IZWFkZXJzKS5sZW5ndGggPT09IDApIHsgcmV0dXJuIHVuZGVmaW5lZDsgfVxuICAgIHJldHVybiBPYmplY3QuZW50cmllcyh0aGlzLmN1c3RvbUhlYWRlcnMpLm1hcCgoW2hlYWRlck5hbWUsIGhlYWRlclZhbHVlXSkgPT4ge1xuICAgICAgcmV0dXJuIHsgaGVhZGVyTmFtZSwgaGVhZGVyVmFsdWUgfTtcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJZiB0aGUgcGF0aCBpcyBkZWZpbmVkLCBpdCBtdXN0IHN0YXJ0IHdpdGggYSAnLycgYW5kIG5vdCBlbmQgd2l0aCBhICcvJy5cbiAgICogVGhpcyBtZXRob2QgdGFrZXMgaW4gdGhlIG9yaWdpblBhdGgsIGFuZCByZXR1cm5zIGl0IGJhY2sgKGlmIHVuZGVmaW5lZCkgb3IgYWRkcy9yZW1vdmVzIHRoZSAnLycgYXMgYXBwcm9wcmlhdGUuXG4gICAqL1xuICBwcml2YXRlIHZhbGlkYXRlT3JpZ2luUGF0aChvcmlnaW5QYXRoPzogc3RyaW5nKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKG9yaWdpblBhdGgpKSB7IHJldHVybiBvcmlnaW5QYXRoOyB9XG4gICAgaWYgKG9yaWdpblBhdGggPT09IHVuZGVmaW5lZCkgeyByZXR1cm4gdW5kZWZpbmVkOyB9XG4gICAgbGV0IHBhdGggPSBvcmlnaW5QYXRoO1xuICAgIGlmICghcGF0aC5zdGFydHNXaXRoKCcvJykpIHsgcGF0aCA9ICcvJyArIHBhdGg7IH1cbiAgICBpZiAocGF0aC5lbmRzV2l0aCgnLycpKSB7IHBhdGggPSBwYXRoLnN1YnN0cigwLCBwYXRoLmxlbmd0aCAtIDEpOyB9XG4gICAgcmV0dXJuIHBhdGg7XG4gIH1cbn1cblxuLyoqXG4gKiBUaHJvd3MgYW4gZXJyb3IgaWYgYSB2YWx1ZSBpcyBkZWZpbmVkIGFuZCBub3QgYW4gaW50ZWdlciBvciBub3QgaW4gYSByYW5nZS5cbiAqL1xuZnVuY3Rpb24gdmFsaWRhdGVJbnRJblJhbmdlT3JVbmRlZmluZWQobmFtZTogc3RyaW5nLCBtaW46IG51bWJlciwgbWF4OiBudW1iZXIsIHZhbHVlPzogbnVtYmVyLCBpc0R1cmF0aW9uOiBib29sZWFuID0gdHJ1ZSkge1xuICBpZiAodmFsdWUgPT09IHVuZGVmaW5lZCkgeyByZXR1cm47IH1cbiAgaWYgKCFOdW1iZXIuaXNJbnRlZ2VyKHZhbHVlKSB8fCB2YWx1ZSA8IG1pbiB8fCB2YWx1ZSA+IG1heCkge1xuICAgIGNvbnN0IHNlY29uZHMgPSBpc0R1cmF0aW9uID8gJyBzZWNvbmRzJyA6ICcnO1xuICAgIHRocm93IG5ldyBFcnJvcihgJHtuYW1lfTogTXVzdCBiZSBhbiBpbnQgYmV0d2VlbiAke21pbn0gYW5kICR7bWF4fSR7c2Vjb25kc30gKGluY2x1c2l2ZSk7IHJlY2VpdmVkICR7dmFsdWV9LmApO1xuICB9XG59XG4iXX0=