"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cognito = require("@aws-cdk/aws-cognito");
const ec2 = require("@aws-cdk/aws-ec2");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const core_1 = require("@aws-cdk/core");
const actions = require("../lib");
class CognitoStack extends core_1.Stack {
    /// !hide
    constructor(scope, id) {
        var _a;
        super(scope, id);
        const vpc = new ec2.Vpc(this, 'Stack', {
            maxAzs: 2,
        });
        const certificate = {
            certificateArn: (_a = process.env.SELF_SIGNED_CERT_ARN) !== null && _a !== void 0 ? _a : '',
        };
        /// !show
        const lb = new elbv2.ApplicationLoadBalancer(this, 'LB', {
            vpc,
            internetFacing: true,
        });
        const userPool = new cognito.UserPool(this, 'UserPool');
        const userPoolClient = new cognito.UserPoolClient(this, 'Client', {
            userPool,
            // Required minimal configuration for use with an ELB
            generateSecret: true,
            authFlows: {
                userPassword: true,
            },
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: [cognito.OAuthScope.EMAIL],
                callbackUrls: [
                    `https://${lb.loadBalancerDnsName}/oauth2/idpresponse`,
                ],
            },
        });
        const cfnClient = userPoolClient.node.defaultChild;
        cfnClient.addPropertyOverride('RefreshTokenValidity', 1);
        cfnClient.addPropertyOverride('SupportedIdentityProviders', ['COGNITO']);
        const userPoolDomain = new cognito.UserPoolDomain(this, 'Domain', {
            userPool,
            cognitoDomain: {
                domainPrefix: 'test-cdk-prefix',
            },
        });
        lb.addListener('Listener', {
            port: 443,
            certificates: [certificate],
            defaultAction: new actions.AuthenticateCognitoAction({
                userPool,
                userPoolClient,
                userPoolDomain,
                next: elbv2.ListenerAction.fixedResponse(200, {
                    contentType: 'text/plain',
                    messageBody: 'Authenticated',
                }),
            }),
        });
        new core_1.CfnOutput(this, 'DNS', {
            value: lb.loadBalancerDnsName,
        });
    }
}
const app = new core_1.App();
new CognitoStack(app, 'integ-cognito');
app.synth();
//# sourceMappingURL=data:application/json;base64,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