# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['quinn', 'quinn.extensions']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'quinn',
    'version': '0.10.0',
    'description': 'Pyspark helper methods to maximize developer efficiency',
    'long_description': '# Quinn\n\n![CI](https://github.com/MrPowers/quinn/workflows/CI/badge.svg?branch=master)\n\nPyspark helper methods to maximize developer productivity.\n\nQuinn validates DataFrames, extends core classes, defines DataFrame transformations, and provides SQL functions.\n\n![quinn](https://github.com/MrPowers/quinn/blob/master/quinn.png)\n\n## Setup\n\nQuinn is [uploaded to PyPi](https://pypi.org/project/quinn/) and can be installed with this command:\n\n```\npip install quinn\n```\n\n## Pyspark Core Class Extensions\n\n```\nfrom quinn.extensions import *\n```\n\n### Column Extensions\n\n**isFalsy()**\n\n```python\nsource_df.withColumn("is_stuff_falsy", F.col("has_stuff").isFalsy())\n```\n\nReturns `True` if `has_stuff` is `None` or `False`.\n\n**isTruthy()**\n\n```python\nsource_df.withColumn("is_stuff_truthy", F.col("has_stuff").isTruthy())\n```\n\nReturns `True` unless `has_stuff` is `None` or `False`.\n\n**isNullOrBlank()**\n\n```python\nsource_df.withColumn("is_blah_null_or_blank", F.col("blah").isNullOrBlank())\n```\n\nReturns `True` if `blah` is `null` or blank (the empty string or a string that only contains whitespace).\n\n**isNotIn()**\n\n```python\nsource_df.withColumn("is_not_bobs_hobby", F.col("fun_thing").isNotIn(bobs_hobbies))\n```\n\nReturns `True` if `fun_thing` is not included in the `bobs_hobbies` list.\n\n**nullBetween()**\n\n```python\nsource_df.withColumn("is_between", F.col("age").nullBetween(F.col("lower_age"), F.col("upper_age")))\n```\n\nReturns `True` if `age` is between `lower_age` and `upper_age`.  If `lower_age` is populated and `upper_age` is `null`, it will return `True` if `age` is greater than or equal to `lower_age`.  If `lower_age` is `null` and `upper_age` is populate, it will return `True` if `age` is lower than or equal to `upper_age`.\n\n### SparkSession Extensions\n\n**create_df()**\n\n```python\nspark.create_df(\n    [("jose", "a"), ("li", "b"), ("sam", "c")],\n    [("name", StringType(), True), ("blah", StringType(), True)]\n)\n```\n\nCreates DataFrame with a syntax that\'s less verbose than the built-in `createDataFrame` method.\n\n### DataFrame Extensions\n\n**transform()**\n\n```python\nsource_df\\\n    .transform(lambda df: with_greeting(df))\\\n    .transform(lambda df: with_something(df, "crazy"))\n```\n\nAllows for multiple DataFrame transformations to be run and executed.\n\n## Quinn Helper Functions\n\n```python\nimport quinn\n```\n\n### DataFrame Validations\n\n**validate_presence_of_columns()**\n\n```python\nquinn.validate_presence_of_columns(source_df, ["name", "age", "fun"])\n```\n\nRaises an exception unless `source_df` contains the `name`, `age`, and `fun` column.\n\n**validate_schema()**\n\n```python\nquinn.validate_schema(source_df, required_schema)\n```\n\nRaises an exception unless `source_df` contains all the `StructFields` defined in the `required_schema`.\n\n**validate_absence_of_columns()**\n\n```python\nquinn.validate_absence_of_columns(source_df, ["age", "cool"])\n```\n\nRaises an exception if `source_df` contains `age` or `cool` columns.\n\n### Functions\n\n**single_space()**\n\n```python\nactual_df = source_df.withColumn(\n    "words_single_spaced",\n    quinn.single_space(col("words"))\n)\n```\n\n\nReplaces all multispaces with single spaces (e.g. changes `"this has   some"` to `"this has some"`.\n\n**remove_all_whitespace()**\n\n```python\nactual_df = source_df.withColumn(\n    "words_without_whitespace",\n    quinn.remove_all_whitespace(col("words"))\n)\n```\n\nRemoves all whitespace in a string (e.g. changes `"this has some"` to `"thishassome"`.\n\n**anti_trim()**\n\n```python\nactual_df = source_df.withColumn(\n    "words_anti_trimmed",\n    quinn.anti_trim(col("words"))\n)\n```\n\nRemoves all inner whitespace, but doesn\'t delete leading or trailing whitespace (e.g. changes `" this has some "` to `" thishassome "`.\n\n**remove_non_word_characters()**\n\n```python\nactual_df = source_df.withColumn(\n    "words_without_nonword_chars",\n    quinn.remove_non_word_characters(col("words"))\n)\n```\n\nRemoves all non-word characters from a string (e.g. changes `"si%$#@!#$!@#mpsons"` to `"simpsons"`.\n\n**exists()**\n\n```python\nsource_df.withColumn(\n    "any_num_greater_than_5",\n    quinn.exists(lambda n: n > 5)(col("nums"))\n)\n```\n\n`nums` contains lists of numbers and `exists()` returns `True` if any of the numbers in the list are greater than 5.  It\'s similar to the Python `any` function.\n\n**forall()**\n\n```python\nsource_df.withColumn(\n    "all_nums_greater_than_3",\n    quinn.forall(lambda n: n > 3)(col("nums"))\n)\n```\n\n`nums` contains lists of numbers and `forall()` returns `True` if all of the numbers in the list are greater than 3.  It\'s similar to the Python `all` function.\n\n**multi_equals()**\n\n```python\nsource_df.withColumn(\n    "are_s1_and_s2_cat",\n    quinn.multi_equals("cat")(col("s1"), col("s2"))\n)\n```\n\n`multi_equals` returns true if `s1` and `s2` are both equal to `"cat"`.\n\n### Transformations\n\n**snake_case_col_names()**\n\n```python\nquinn.snake_case_col_names(source_df)\n```\n\nConverts all the column names in a DataFrame to snake_case.  It\'s annoying to write SQL queries when columns aren\'t snake cased.\n\n**sort_columns()**\n\n```python\nquinn.sort_columns(source_df, "asc")\n```\n\nSorts the DataFrame columns in alphabetical order.  Wide DataFrames are easier to navigate when they\'re sorted alphabetically.\n\n### DataFrame Helpers\n\n**column_to_list()**\n\n```python\nquinn.column_to_list(source_df, "name")\n```\n\nConverts a column in a DataFrame to a list of values.\n\n**two_columns_to_dictionary()**\n\n```python\nquinn.two_columns_to_dictionary(source_df, "name", "age")\n```\n\nConverts two columns of a DataFrame into a dictionary.  In this example, `name` is the key and `age` is the value.\n\n**to_list_of_dictionaries()**\n\n```python\nquinn.to_list_of_dictionaries(source_df)\n```\n\nConverts an entire DataFrame into a list of dictionaries.\n\n## Contributing\n\nWe are actively looking for feature requests, pull requests, and bug fixes.\n\nAny developer that demonstrates excellence will be invited to be a maintainer of the project.\n',
    'author': 'MrPowers',
    'author_email': 'matthewkevinpowers@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/MrPowers/quinn/',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>3.5',
}


setup(**setup_kwargs)
