# Copyright 2022 CVS Health and/or one of its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

__docformat__ = "google"

from dataclasses import dataclass, field
import numpy as np
import seaborn as sns


@dataclass
class DistP:
    """
    `DistP` stands for `Distribution Parameter`. This class creates a dataclass which contains some useful convenience functions that make instantiating, faking, manipulating, and doing basic mathematical operations with distributions facile.

    ## How can I fit a parameter with DistP?
    Let's assume you have this array from 50 member treatment effects from an experiment
    ```python

    data_array = np.array(
    [0.1243272 , 0.02841775, 0.02462303, 0.05596238, 0.02816799,
    0.06737527, 0.08744445, 0.05503281, 0.06467202, 0.06707346,
    0.12590284, 0.04279422, 0.06179317, 0.05264741, 0.00944362,
    0.02500023, 0.15752293, 0.02699168, 0.04178847, 0.01391412,
    0.04602457, 0.0596269 , 0.06603306, 0.08619834, 0.02477823,
    0.2167056 , 0.04017278, 0.03583264, 0.09153694, 0.03640167,
    0.02588049, 0.09159757, 0.07308884, 0.04771209, 0.09371187,
    0.05632816, 0.00991528, 0.04369652, 0.03447077, 0.03877374,
    0.10061553, 0.05931281, 0.07819679, 0.05195733, 0.11171881,
    0.10992641, 0.0635759 , 0.11434518, 0.05556813, 0.05661915])
    ```
    Let's then try to establish a distribution sampling object that will draw from this distribution faithfullly.

    That's easy! Simply instantiate the DistP object with a `name`, a `lever`, pass the `distfunc` special str of `fitted`, set some `bounds`, then select a `size` or the length of the resampled function (equivalent to fair sample draws), and then pass the array into the `samples`.
    ``` python
    fit_te = DistP( name='treatment effect',
                    lever='lever 1',
                    distfunc='fitted',
                    bounds=[0.0, 1],
                    size = 20,
                    samples = data_array)
    ```
    I recommend using the fitted distributions as much as possible. However, where you can't, you can create a distribution easily!

    Let's say we're trying to simulate the prevalence of a condition and we know
    that it's log-normally distributed and approxmately centered about np.log of (0.02).

    ```python
    prevalence = DistP(name='prevalence',
                       lever='lever 1',
                       distfunc=np.random.lognormal,
                       bounds=[0.002, 1],
                       kwargs={
                           'mean': np.log(0.02),
                           'sigma': np.log(2),
                           'size': size
                       })
    ```

    Args:
        name: str, the name you give the distribution parameter.
        lever: str, this is the name of the value
            lever the distribution is associated with.
        distfunc: object, this is typically a numpy `np.random.<>`
            object, but it can be anything that has the same convention
        bounds: list, upper and lower bounds e.g., [0,1]
        bound_method: str, the bound method 'flavor', choose 'drop_recursive'
        size_kwd: str = 'size'. Some function have other keywords for the length
            of the array generated by the random object. If so, use this.
        kwargs: dict, these are the args that 'go with' the `distfunc` object.
            make sure to pass them with a `**`.

        size: int, this is the default size/length for resumpling or subsampling.
            honestly, the more the merrier.
        samples: list, some defaults are provided, but you can either pass your own
            samples (e.g., you want to fit a distribution), or these samples are a
            response to passing a distfunc and accopmanying args.

        samples_mean: the sample mean
        samples_median: the sample median
        samples_std: the sample std
        samples_percentiles: the samples percentiles ([0,25,50,70,100]).

    return:
    """
    name: str = 'default name'
    lever: str = 'default lever'
    segment: str = 'default segment'
    distfunc: object = 'default'
    bounds: list = field(default_factory=lambda: [0, 1e6])
    bound_method: str = 'drop_recursive'
    size_kwd: str = 'size'
    kwargs: dict = field(default_factory=dict)
    size: int = 5000
    samples: np.ndarray = field(default_factory=lambda: np.array([0, 1, 2]))
    samples_mean: np.float64 = field(default_factory=lambda: 0)
    samples_median: np.float64 = field(default_factory=lambda: 0)
    samples_std: np.float64 = field(default_factory=lambda: 0)
    samples_percentiles: list = field(default_factory=lambda: [0])

    def __post_init__(self):
        """
        post init dataclass function.

        Instantiating class with default values can be a pain, hence the
        default pass. This can be fixed, but if it works, and is dumb,
        it's not dumb.

        Also, there are a few "convenience attribute"s that are carried
        around with each distribution (e.g., mean, median, std, and percentiles)

        Returns:
            self

        """

        if self.distfunc == 'default':
            pass

        elif self.distfunc == 'fitted':
            self.samples = self._fitted()

        else:
            self.samples = self.distfunc(**self.kwargs)
            self.samples_len = len(self.samples)

            if self.bounds:
                self._bounds()

        self.update_diststats()

        return self

    def update_diststats(self):
        self.samples_mean = round(np.mean(self.samples), 2)
        self.samples_median = round(np.median(self.samples), 2)
        self.samples_std = round(np.std(self.samples), 2)

        self.samples_percentiles = []
        for p in np.arange(0, 125, 25):
            self.samples_percentiles.append(round(np.percentile(self.samples, p), 3))
        return self

    def _fitted(self):
        """
        internal function to fit a distribution passed from samples
        and return a uniform sample (recovering the distribution),
        and enabling up/down-sampling of the distribution.

        Returns:
            np.array: array of randomly sampled values from the dist.

        """
        if not self.size:
            self.size = self.samples_len

        samples = self.samples[self.samples >= self.bounds[0]]
        samples = samples[samples <= self.bounds[1]]
        return np.random.choice(samples, self.size)

    def mult_const(self, k):
        """
        scale a distributions, array wise, by k

        Args:
            k: A scalar float

        Returns:
            self

        """
        self.samples = self.samples * k
        return self

    def chain_mult(self, distribution):
        """
        multiply two distributions, array wise

        Args:
            distribution: A distribution object

        Returns:
            self

        """
        self.samples = np.multiply(self.samples, distribution.samples)
        return self

    def chain_divide(self, distribution):
        """
        divide two distributions, array wise

        Args:
            DistP: A distribution object

        Returns:
            self

        """
        self.samples = np.divide(self.samples, distribution.samples)
        return self

    def chain_add(self, distribution):
        """
        add two distributions, array wise

        Args:
            DistP: A distribution object

        Returns:
            self

        """
        self.samples = np.add(self.samples, distribution.samples)
        return self

    def chain_sub(self, distribution):
        """
        subtract two distributions, array wise

        Args:
            DistP: A distribution object

        Returns:
            self

        """
        self.samples = np.subtract(self.samples, distribution.samples)
        return self

    def conf_int(self, ci):
        """
        returns the 95CI

        Args:
            ci: list for np.percentile method e.g.,
                [2.5,97.5] for 95ci

        Returns:
            self

        """
        self.conf = np.percentile(self.samples, [2.5, 97.5])
        if ci:
            self.conf = np.percentile(self.samples, ci)

        return self

    def _bounds(self):
        """
        Internal bounds sample. Can choose 3 'flavors':
        1. `Stack`: Stack simply assigns all values lower (or higher)
            than the bounds at the bounds.
        2.  `drop_recursie`: This is the preferred method. It drops
            values lower (or higher) than the bounds, then upsamples
            the distribution and re clips.

        Returns:
            self

        """
        if self.bound_method == 'stack':
            self.samples[self.samples < self.bounds[0]] = self.bounds[0]
            self.samples[self.samples > self.bounds[1]] = self.bounds[1]

        elif self.bound_method == 'drop_recursive':
            self.samples = self._bounds_sampler(self.bounds, self.samples)

        return self

    def _bounds_sampler(self, bounds, samples):
        """Recursively up-sample and then subsample.
        Not the best implementation I've ever done, but she works.

        Args:
            bounds (list): [low,high] bounds to constrain the
                distribution function.
            samples (list): A list of samples.

        Returns:
            list: A list of samples.

        """

        samples = samples[samples > bounds[0]]
        samples = samples[samples < bounds[1]]

        if len(samples) >= self.samples_len:
            # Uniformly random sample from your distribution
            samples = np.random.choice(samples, size=self.samples_len)
            return samples

        elif len(samples) < self.samples_len:

            # Increase sample 'size' by 1/(1-fraction missing - 10%) and resample
            frac_miss = len(samples) / self.samples_len
            new_size = int(self.samples_len / (frac_miss))

            # Update kwarg
            self.kwargs[self.size_kwd] = new_size

            # Regen samples
            samples = self.distfunc(**self.kwargs)

        return self._bounds_sampler(bounds, samples)

    def histplot(self, **args):
        """Plot the distribution.

        Args:
            **args: Variable length argument dict
                for seaborn's `sns.histplot` object

        Returns:
            object: sns.histplot object.

        """
        return sns.histplot(self.samples, **args)

    def kdeplot(self, **args):
        """Plot the distribution.

        Args:
            **args: Variable length argument dict
                for seaborn's `sns.kdeplot` object

        Returns:
            object: sns.kdeplot object.

        """
        return sns.kdeplot(self.samples, **args)

    def update(self, key, value):
        if key in self.__dict__.keys():
            self.__dict__.update({key: value})

        return self
