# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['openeo_pg_parser_networkx']

package_data = \
{'': ['*']}

install_requires = \
['geojson-pydantic>=0.5.0,<0.6.0',
 'networkx[default]>=2.8.6,<3.0.0',
 'numpy>=1.24.1,<2.0.0',
 'pendulum>=2.1.2,<3.0.0',
 'pydantic>=1.9.1,<2.0.0',
 'pyproj>=3.4.0,<4.0.0',
 'shapely>=1.8']

setup_kwargs = {
    'name': 'openeo-pg-parser-networkx',
    'version': '2023.1.2',
    'description': 'Parse OpenEO process graphs from JSON to traversible Python objects.',
    'long_description': '# OpenEO Process Graph Parser (Python & networkx)\n![PyPI - Status](https://img.shields.io/pypi/status/openeo-pg-parser-networkx)\n![PyPI](https://img.shields.io/pypi/v/openeo-pg-parser-networkx)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/openeo-pg-parser-networkx)\n[![codecov](https://codecov.io/github/Open-EO/openeo-pg-parser-networkx/branch/main/graph/badge.svg?token=KEAKFB8AFX)](https://codecov.io/github/Open-EO/openeo-pg-parser-networkx)\n\nPython package to parse OpenEO process graphs from raw JSON into fully traversible [`networkx`](https://github.com/networkx/networkx) graph objects.\nThis package is an evolution of the [openeo-pg-parser-python](https://github.com/Open-EO/openeo-pg-parser-python) package.\n\n## Installation\nThis package can be installed with pip:\n\n```\npip install openeo-pg-parser-networkx\n```\n\nCurrently Python versions 3.9 and 3.10 are supported.\n\n## Basic usage\n(An example notebook of using `openeo-pg-parser-networkx` together with a process implementation source like [`openeo-processes-dask`](https://github.com/Open-EO/openeo-processes-dask) can be found in `openeo-pg-parser-networkx/examples/01_minibackend_demo.ipynb`.)\n\n**Parse a JSON OpenEO process graph:**\n\n```\nfrom openeo_pg_parser_networkx.graph import OpenEOProcessGraph\n\nNDVI_GRAPH_PATH = "../tests/data/graphs/apply.json"\n\nparsed_graph = OpenEOProcessGraph.from_file(NDVI_GRAPH_PATH)\n```\n\n```\n> Deserialised process graph into nested structure\n> Walking node root-fd8ae3b4-8cb8-46c8-a5cd-c8ee552d1945\n> Walking node apply2-fd8ae3b4-8cb8-46c8-a5cd-c8ee552d1945\n> Walking node multiply1-f8644201-32a8-4283-8814-a577c4e28226\n> Walking node apply1-fd8ae3b4-8cb8-46c8-a5cd-c8ee552d1945\n> Walking node ndvi1-06a8d8af-296a-4960-a1cb-06dcd251b6bb\n> Walking node loadcollection1-fd8ae3b4-8cb8-46c8-a5cd-c8ee552d1945\n```\n\n**Plot it:**\n\n```\nparsed_graph.plot()\n```\n\n![example process graph](./examples/images/apply_ndvi.png)\n\nTo execute a process graph, `OpenEOProcessGraph` needs to know which Python code to call for each of the nodes in the graph. This information is provided by a "process registry", which is basically a dictionary that maps each `process_id` to its actual Python implementation as a `Callable`.\n\n**Register process implementations to a "process registry":**\n\n```\nfrom openeo_pg_parser_networkx import ProcessRegistry\nprocess_registry = ProcessRegistry()\n\nfrom openeo_processes_dask.process_implementations import apply, ndvi, multiply, load_collection, save_result\n\nprocess_registry["apply"] =  apply\nprocess_registry["ndvi"] =  ndvi\nprocess_registry["multiply"] =  multiply\nprocess_registry["load_collection"] =  load_collection\nprocess_registry["save_result"] =  save_result\n```\n\n**Build an executable callable from the process graph:**\n\n```\npg_callable = parsed_graph.to_callable(process_registry=process_registry)\n```\n\n**Execute that callable like a normal Python function:**\n\n```\npg_callable\n```\n\n```\n> Running process load_collection\n> Running process apply\n> ...\n```\n\n## Development environment\n`openeo-pg-parser-networkx` requires poetry `>1.2`, see their [docs](https://python-poetry.org/docs/#installation) for installation instructions.\n\nTo setup the python venv and install this project into it run:\n```\npoetry install\n```\n\nTo add a new core dependency run:\n```\npoetry add some_new_dependency\n```\n\nTo add a new development dependency run:\n```\npoetry add some_new_dependency --group dev\n```\n\nTo run the test suite run:\n```\npoetry run python -m pytest\n```\n\nNote that you can also use the virtual environment that\'s generated by poetry as the kernel for the ipynb notebooks.\n\n### Pre-commit hooks\nThis repo makes use of [pre-commit](https://pre-commit.com/) hooks to enforce linting & a few sanity checks.\nIn a fresh development setup, install the hooks using `poetry run pre-commit install`.\nThese will then automatically be checked against your changes before making the commit.\n',
    'author': 'Lukas Weidenholzer',
    'author_email': 'lukas.weidenholzer@eodc.eu',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/Open-EO/openeo-pg-parser-networkx',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<3.11',
}


setup(**setup_kwargs)
