# (Be in -*- python -*- mode.)
#
# ====================================================================
# Copyright (c) 2009 CollabNet.  All rights reserved.
#
# This software is licensed as described in the file COPYING, which
# you should have received as part of this distribution.  The terms
# are also available at http://subversion.tigris.org/license-1.html.
# If newer versions of this license are posted there, you may use a
# newer version instead, at your option.
#
# This software consists of voluntary contributions made by many
# individuals.  For exact contribution history, see the revision
# history and logs, available at http://cvs2svn.tigris.org/.
# ====================================================================

from cvs2hg_lib.common import FatalError
from cvs2hg_lib.context import Ctx
from cvs2hg_lib.log import logger
from cvs2hg_lib.run_options import IncompatibleOption
from cvs2hg_lib.dvcs_common import DVCSRunOptions
from cvs2hg_lib.hg_output_option import HgOutputOption
from cvs2hg_lib.hg_output_option import ClobberHgOutputOption
from cvs2hg_lib.hg_output_option import ExistingHgOutputOption


class HgRunOptions(DVCSRunOptions):

    short_desc = 'convert a cvs repository into a Mercurial repository'

    synopsis = """\
.B cvs2hg
[\\fIOPTION\\fR]... \\fIOUTPUT-OPTION CVS-REPOS-PATH\\fR
.br
.B cvs2hg
[\\fIOPTION\\fR]... \\fI--options=PATH\\fR
"""

    # XXX paragraph 2 copied straight from svn_run_options.py
    long_desc = """\
Create a new Mercurial repository based on the version history stored in
a CVS repository. Each CVS commit will be mirrored in the Mercurial
repository, including commit time and author (with optional remapping to
Mercurial-style long usernames).
.P
\\fICVS-REPOS-PATH\\fR is the filesystem path of the part of the CVS
repository that you want to convert.  It is not possible to convert a
CVS repository to which you only have remote access; see the FAQ for
more information.  This path doesn't have to be the top level
directory of a CVS repository; it can point at a project within a
repository, in which case only that project will be converted.  This
path or one of its parent directories has to contain a subdirectory
called CVSROOT (though the CVSROOT directory can be empty).
.P
Unlike CVS or Subversion, Mercurial expects each repository to hold
one independent project.  If your CVS repository contains multiple
independent projects, you should probably convert them to multiple
independent Mercurial repositories with multiple runs of
.B cvs2hg.
"""

    # XXX copied from svn_run_options.py
    files = """\
A directory called \\fItest-temporary-tmp-dir\\fR (or the directory specified by
\\fB--tmpdir\\fR) is used as scratch space for temporary data files.
"""

    # XXX the cvs2{svn,git,bzr,hg} man pages should probably reference
    # each other
    see_also = [
    ('cvs', '1'),
    ('hg', '1'),
    ]

    def __init__(self, *args, **kwargs):
        # Override some default values
        ctx = Ctx()
        ctx.username = "cvs2hg"
        ctx.symbol_commit_message = (
          "fixup commit for %(symbol_type)s '%(symbol_name)s'")
        ctx.post_commit_message = (
          "artificial commit to compensate for changes in %(revnum)s from "
          "a CVS vendor branch")

        super(HgRunOptions, self).__init__(*args, **kwargs)

    # This is a straight copy of SVNRunOptions._get_extraction_options_group();
    # would be nice to refactor, but it's a bit awkward because GitRunOptions
    # doesn't support --use-internal-co option.
    def _get_extraction_options_group(self):
        group = DVCSRunOptions._get_extraction_options_group(self)
        self._add_use_internal_co_option(group)
        self._add_use_cvs_option(group)
        self._add_use_rcs_option(group)
        return group

    def _get_output_options_group(self):
        group = super(HgRunOptions, self)._get_output_options_group()

        # XXX what if the hg repo already exists? die, clobber, or append?
        # (currently we die at the start of OutputPass)
        group.add_option(IncompatibleOption(
          '--hgrepos', type='string',
          action='store',
          help='create Mercurial repository in PATH',
          man_help=(
              'Convert to a Mercurial repository in \\fIpath\\fR.  This creates '
              'a new Mercurial repository at \\fIpath\\fR.  \\fIpath\\fR must '
              'not already exist (unless you pass \\fB--clobber\\fR).'
              ),
          metavar='PATH',
          ))
        group.add_option(IncompatibleOption(
          '--clobber', action='store_true', default=False,
          help='if output repository already exists, remove it',
          man_help=(
              'If the output repository passed to \\fB--hgrepos\\fR '
              'already exists, remove it and create a new repository.'
              ),
          ))
        group.add_option(IncompatibleOption(
          '--existing-hgrepos', type='string', action='store',
          help='add to existing Mercurial repository in PATH',
          man_help=(
              'Convert and append changesets to the Mercurial repository '
              'in \\fIpath\\fR. \\fIpath\\fR must already exist.'),
          metavar='PATH',
          ))

        # XXX --dry-run?

        return group

    def process_extraction_options(self):
        """Process options related to extracting data from the CVS repository."""
        self.process_all_extraction_options()

    def process_output_options(self):
        if self.options.hgrepos:
            repo_dir = self.options.hgrepos
            if self.options.clobber:
                klass = ClobberHgOutputOption
            else:
                klass = HgOutputOption
        elif self.options.existing_hgrepos:
            repo_dir = self.options.existing_hgrepos
            if self.options.clobber:
                logger.warn('option --clobber makes no sense combined with '
                            '--existing-hgrepos (ignoring it)')
            klass = ExistingHgOutputOption
        else:
            raise FatalError('must pass either --hgrepos or --existing-hgrepos')

        Ctx().output_option = klass(
          repo_dir,
          author_transforms={},
          )
