"""Module for wrinting the Level 1 data attributes

"""

from .lstConfig import configurations

class loadAttributes:

    """Level 1 Attribute generator

    It defines and writes the attributes from the Level 1

    Parameters
    ----------
    data : xarray.Dataset
        A xarray dataset containing the Level 1 data
        generated by the lidarsuit
        
    Returns
    -------
    data : xarray.Dataset 
        
        A dataset with corrected attributes

    """

    def __init__(self, data):

        self.data = data
        self.writeGlobalAttrs()
        self.variablesAttrs()
        self.writeCoordsAttrs()
        self.writeVariablesAttrs()

        return None

    def writeGlobalAttrs(self):

        """Global attribute writer

        It loads and writes the global attributes from the Level 1 dataset.
        The global attributes are defiened in the configuration file.
        See lstConfig.generateConf documentation for information about generating
        the configuration file.
        
        """

        configInfo = configurations(lst=None).loadConfFile()

        tmpAtt = {'Conventions':'Cf/Radial 2.0',
                  'title': 'Wind properties',
                  'references':configInfo.references,
                  'institution':configInfo.institution,
                  'instrument_name':configInfo.instrument,
                  'comments':configInfo.comments,
                  'site_name':configInfo.site,
                  'contact_person':configInfo.contact,
                  'email':configInfo.email}

        self.data.attrs = tmpAtt
        
        return self
    
    def variablesAttrs(self):

        """Variable attributes definitions

        All variables and coordinates attributes are defined here.

        """

        attrsDic = {}

        attrsDic['range']=\
            {'standard_name':'range',
             'units':'m',
             'comments': 'Distance between the instrument and the center of each range gate'}

        attrsDic['time']=\
            {'standard_name':'time',
             'reference':'seconds since 1970-01-01 00:00:00',
             'comments': 'time of the horizotal observations'}

        attrsDic['time90']=\
            {'standard_name':'time90',
             'reference':'seconds since 1970-01-01 00:00:00',
             'comments': 'time of the vertical observations'}

        attrsDic['horizontal_wind_speed']=\
            {'standard_name':'wind_speed',
             'units':'m/s',
             'comments': 'horizontal wind speed retrived using the FFT method'}

        attrsDic['horizontal_wind_direction']=\
            {'standard_name':'wind_direction',
             'units':'degrees',
             'comments': 'horizontal wind direction retrived using the FFT method with respect to true north',
             'info': '0=wind coming from the north, 90=east, 180=south, 270=west'}

        attrsDic['zonal_wind']=\
            {'standard_name':'zonal_wind',
             'units':'m/s',
             'comments': 'zonal wind retrived using the FFT method'}

        attrsDic['meridional_wind']=\
            {'standard_name':'meridional_wind',
             'units':'m/s',
             'comments': 'meridional wind retrived using the FFT method'}

        attrsDic['vertical_wind_speed']=\
            {'standard_name':'vertical_wind_speed',
             'units':'m/s',
             'comments': 'observed vertical wind speed (negative towards the ground)'}

        attrsDic['lidar_relative_beta']=\
            {'standard_name':'volume_attenuated_backwards_scattering_function_in_air',
             'units':'m-1 sr-1',
             'comments': 'Attenuated relative backscatter coefficient from the vertical beam'}

        self.attrsDic = attrsDic

        return self
            
    def writeCoordsAttrs(self):

        """Coordinate attribute writer

        It loops over the dataset coordinate and writes their attributes

        """

        for key in self.data.coords:

            try:
                self.data[key].attrs = self.attrsDic[key]

            except:
                print('coord not found: {0}'.format(key))

        return self
            
    def writeVariablesAttrs(self):

        """Variable attribute writer

        It loops over the dataset variables and writes their attributes

        """

        for key in self.data.keys():

            try:
                self.data[key].attrs = self.attrsDic[key]

            except:
                print('variable not found: {0}'.format(key))

        return self