# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pytest_vcr_delete_on_fail']

package_data = \
{'': ['*']}

install_requires = \
['pytest>=6.2.2,<7.0.0']

entry_points = \
{'pytest11': ['vcr_delete_on_fail = pytest_vcr_delete_on_fail.main']}

setup_kwargs = {
    'name': 'pytest-vcr-delete-on-fail',
    'version': '1.0.0',
    'description': 'A pytest plugin that automates vcrpy cassettes deletion on test failure.',
    'long_description': '[![PyPI](https://img.shields.io/pypi/v/pytest-vcr-delete-on-fail)](https://pypi.org/project/pytest-vcr-delete-on-fail/) [![PyPI - Python Version](https://img.shields.io/pypi/pyversions/pytest-vcr-delete-on-fail)](https://pypi.org/project/pytest-vcr-delete-on-fail/) [![CI Status](https://img.shields.io/github/workflow/status/CarloDePieri/pytest-vcr-delete-on-fail/prod?logo=github)](https://github.com/CarloDePieri/pytest-vcr-delete-on-fail/actions/workflows/prod.yml) [![Coverage Status](https://coveralls.io/repos/github/CarloDePieri/pytest-vcr-delete-on-fail/badge.svg?branch=main)](https://coveralls.io/github/CarloDePieri/pytest-vcr-delete-on-fail?branch=main) [![Maintenance](https://img.shields.io/maintenance/yes/2022)](https://github.com/CarloDePieri/pytest-vcr-delete-on-fail/)\n\nA pytest plugin that automates vcrpy cassettes deletion on test failure.\n\n## Rationale\n\nSometimes when testing a function containing multiple http requests a failure will occur halfway through (this happens\nall the time when doing TDD). When using [vcrpy](https://github.com/kevin1024/vcrpy) to cache http requests, this could\nresult in a test cache that only cover a fraction of the function under test, which in turn could prevent the function\nto ever succeed or the test to pass in subsequent run if the http requests that didn\'t get cached depended on a\nfresh context (maybe they are time sensitive or there\'s randomness involved).\n\nThis possibility leads to doubt and untrust towards the test suite, which is wrong on too many level.\n\nThis plugin provides a pytest marker that solves the uncertainty: when a marked test fails its http requests cache will\nbe purged, so that it can start fresh on the next run.\n\n## Install\n\nSimply run:\n\n```bash\npip install pytest-vcr-delete-on-fail\n```\n\n## Usage\n\nOnce the plugin is installed, mark the designated test like this:\n\n```python\nimport pytest\nimport requests\nimport vcr\n\nmy_vcr = vcr.VCR(record_mode="once")\n\ncassette_path = "tests/cassettes/this.yaml"\n\n@pytest.mark.vcr_delete_on_fail([cassette_path])\ndef test_this():\n    with my_vcr.use_cassette(cassette_path):\n        requests.get("https://github.com")\n    assert False\n```\n\nRunning the test will result in no cassettes on disk: the http request first got cached but since the test failed the\ncassette was later deleted.\n\nWhen using [pytest-recording](https://github.com/kiwicom/pytest-recording) to automatically save cassettes (or when\nusing its path and naming conventions manually), this plugin\'s marker can be used without arguments since it will figure\nout the cassette path on its own:\n\n```python\nimport pytest\nimport requests\n\n# Configure pytest_recording\n@pytest.fixture(scope="module")\ndef vcr_config():\n    return {"record_mode": ["once"]}\n\n@pytest.mark.vcr_delete_on_fail\ndef test_this():\n    requests.get("https://github.com")\n    assert False\n```\n\nThe marker is actually quite flexible; this is the full signature:\n\n```python\npytest.mark.vcr_delete_on_fail(\n    cassette_path_list: Optional[List[Union[str, Callable[[Item], str]]]],\n    delete_default: Optional[bool],\n    skip: Optional[bool])\n```\n\n###### cassette_path_list\n\nEither the first unnamed argument or a named one; when both are missing, the cassette path will be automatically\ndetermined. This list\'s elements can either be `str` or functions that take a pytest `nodes.Item` object and return a\n`str`: these are the to-be-deleted cassettes\' full path. If `cassette_path_list is None`, no cassette will be deleted\nfor that test (which is equivalent to pass `skip=True`); if an empty list is passed instead, that marker won\'t result\nin a cassette deletion but it won\'t prevent other markers to delete cassettes.\n\n###### delete_default\n\nOnly valid as named argument. It\'s `True` by default if no `cassette_path_list` is passed to the marker, `False`\notherwise. If `True` the cassette with the automatically computed path will be deleted.\n\n###### skip\n\nOnly valid as named argument. It\'s `False` by default. If `True` no cassette will be deleted for that test. It\'s\nequivalent to passing `cassette_path_list=None`.\n\n### Utilities\n\nWhen writing a function to determine a cassette path here are some useful imports from `pytest_vcr_delete_on_fail`:\n\n###### get_default_cassette_path(item: nodes.Item) -> str\n\nA function that return the path of the default cassette.\n\n###### has_class_scoped_setup_failed(item: nodes.Item) -> bool\n\nIt returns True if a class scoped fixture failed in the setup phase. This could come in handy when using class scoped setup: an example\nof this pattern can be found in [test_integrations.py](https://github.com/CarloDePieri/pytest-vcr-delete-on-fail/blob/main/tests/test_integrations.py).\n\n###### has_class_scoped_teardown_failed(item: nodes.Item) -> bool\n\nIt returns True if a class scoped fixture failed in the teardown phase.\n\n## Development\n\nInstall [invoke](http://pyinvoke.org/) and [poetry](https://python-poetry.org/):\n\n```bash\npip install invoke poetry\n```\n\nNow clone the git repo:\n\n```bash\ngit clone https://github.com/CarloDePieri/pytest-vcr-delete-on-fail.git\ncd pytest-vcr-delete-on-fail\ninv install\n```\n\nThis will try to create a virtualenv based on `python3.7` and install there all\nproject\'s dependencies. If a different python version is preferred, it can be\nselected by specifying  the `--python` (`-p`) flag like this:\n\n```bash\ninv install -p python3.8\n```\n\nThe test suite can be run with commands:\n\n```bash\ninv test         # run the test suite\ninv test-spec    # run the tests while showing the output as a spec document\ninv test-cov     # run the tests suite and produce a coverage report\n```\n\nTo run the test suite against all supported python version (they must be in path!) run:\n\n```bash\ninv test-all-python-version\n```\n\nTo test the github workflow with [act](https://github.com/nektos/act):\n\n```bash\ninv act-dev           # test the dev workflow\ninv act-dev -c shell  # open a shell in the act container (the above must fail first!)\ninv act-dev -c clean  # stop and delete a failed act container\n```\n',
    'author': 'Carlo De Pieri',
    'author_email': 'depieri.carlo@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/CarloDePieri/pytest-vcr-delete-on-fail',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
