# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['decoy', 'decoy.mypy']

package_data = \
{'': ['*']}

entry_points = \
{'pytest11': ['decoy = decoy.pytest_plugin']}

setup_kwargs = {
    'name': 'decoy',
    'version': '1.10.3',
    'description': 'Opinionated mocking library for Python',
    'long_description': '<div align="center">\n    <img alt="Decoy logo" src="https://mike.cousins.io/decoy/img/decoy.png" width="256px">\n    <h1 class="decoy-title">Decoy</h1>\n    <p>Opinionated mocking library for Python</p>\n    <p>\n        <a title="CI Status" href="https://github.com/mcous/decoy/actions">\n        <img src="https://img.shields.io/github/workflow/status/mcous/decoy/Continuous%20integration/main?style=flat-square"></a>\n        <a title="Code Coverage" href="https://app.codecov.io/gh/mcous/decoy/"><img src="https://img.shields.io/codecov/c/github/mcous/decoy?style=flat-square"></a>\n        <a title="License" href="https://github.com/mcous/decoy/blob/main/LICENSE"><img src="https://img.shields.io/github/license/mcous/decoy?style=flat-square"></a>\n        <a title="PyPI Version"href="https://pypi.org/project/decoy/"><img src="https://img.shields.io/pypi/v/decoy?style=flat-square"></a>\n        <a title="Supported Python Versions" href="https://pypi.org/project/decoy/"><img src="https://img.shields.io/pypi/pyversions/decoy?style=flat-square"></a>\n    </p>\n    <p>\n        <a href="https://mike.cousins.io/decoy/" class="decoy-hidden">Usage guide and documentation</a>\n    </p>\n</div>\n\nDecoy is a mocking library designed for **effective and productive test-driven development** in Python. If you want to use tests to guide the structure of your code, Decoy might be for you!\n\nDecoy mocks are **async/await** and **type-checking** friendly. Decoy is heavily inspired by (and/or stolen from) the excellent [testdouble.js][] and [Mockito][] projects. The Decoy API is powerful, easy to read, and strives to help you make good decisions about your code.\n\n## Install\n\n```bash\n# pip\npip install decoy\n\n# poetry\npoetry add --dev decoy\n\n# pipenv\npipenv install --dev decoy\n```\n\n## Setup\n\n### Pytest setup\n\nDecoy ships with its own [pytest][] plugin, so once Decoy is installed, you\'re ready to start using it via its pytest fixture, called `decoy`.\n\n```python\n# test_my_thing.py\nfrom decoy import Decoy\n\ndef test_my_thing_works(decoy: Decoy) -> None:\n    ...\n```\n\n### Mypy setup\n\nBy default, Decoy is compatible with Python [typing][] and type-checkers like [mypy][]. However, stubbing functions that return `None` can trigger a [type checking error](https://mypy.readthedocs.io/en/stable/error_code_list.html#check-that-called-function-returns-a-value-func-returns-value) during correct usage of the Decoy API. To suppress these errors, add Decoy\'s plugin to your mypy configuration.\n\n```ini\n# mypy.ini\nplugins = decoy.mypy\n```\n\n### Other testing libraries\n\nDecoy works well with [pytest][], but if you use another testing library or framework, you can still use Decoy! You just need to do two things:\n\n1. Create a new instance of [`Decoy()`](https://mike.cousins.io/decoy/api/#decoy.Decoy) before each test\n2. Call [`decoy.reset()`](https://mike.cousins.io/decoy/api/#decoy.Decoy.reset) after each test\n\nFor example, using the built-in [unittest][] framework, you would use the `setUp` fixture method to do `self.decoy = Decoy()` and the `tearDown` method to call `self.decoy.reset()`. For a working example, see [`tests/test_unittest.py`](https://github.com/mcous/decoy/blob/main/tests/test_unittest.py).\n\n## Basic Usage\n\nThis basic example assumes you are using [pytest][]. For more detailed documentation, see Decoy\'s [usage guide][] and [API reference][].\n\n### Define your test\n\nDecoy will add a `decoy` fixture that provides its mock creation API.\n\n```python\nfrom decoy import Decoy\nfrom todo import TodoAPI, TodoItem\nfrom todo.store TodoStore\n\ndef test_add_todo(decoy: Decoy) -> None:\n    ...\n```\n\n### Create a mock\n\nUse `decoy.mock` to create a mock based on some specification. From there, inject the mock into your test subject.\n\n```python\ndef test_add_todo(decoy: Decoy) -> None:\n    todo_store = decoy.mock(cls=TodoStore)\n    subject = TodoAPI(store=todo_store)\n    ...\n```\n\nSee [creating mocks][] for more details.\n\n### Stub a behavior\n\nUse `decoy.when` to configure your mock\'s behaviors. For example, you can set the mock to return a certain value when called in a certain way using `then_return`:\n\n```python\ndef test_add_todo(decoy: Decoy) -> None:\n    """Adding a todo should create a TodoItem in the TodoStore."""\n    todo_store = decoy.mock(cls=TodoStore)\n    subject = TodoAPI(store=todo_store)\n\n    decoy.when(\n        todo_store.add(name="Write a test for adding a todo")\n    ).then_return(\n        TodoItem(id="abc123", name="Write a test for adding a todo")\n    )\n\n    result = subject.add("Write a test for adding a todo")\n    assert result == TodoItem(id="abc123", name="Write a test for adding a todo")\n```\n\nSee [stubbing with when][] for more details.\n\n### Verify a call\n\nUse `decoy.verify` to assert that a mock was called in a certain way. This is best used with dependencies that are being used for their side-effects and don\'t return a useful value.\n\n```python\ndef test_remove_todo(decoy: Decoy) -> None:\n    """Removing a todo should remove the item from the TodoStore."""\n    todo_store = decoy.mock(cls=TodoStore)\n    subject = TodoAPI(store=todo_store)\n\n    subject.remove("abc123")\n\n    decoy.verify(todo_store.remove(id="abc123"))\n```\n\nSee [spying with verify][] for more details.\n\n[testdouble.js]: https://github.com/testdouble/testdouble.js\n[mockito]: https://site.mockito.org/\n[pytest]: https://docs.pytest.org/\n[unittest]: https://docs.python.org/3/library/unittest.html\n[typing]: https://docs.python.org/3/library/typing.html\n[mypy]: https://mypy.readthedocs.io/\n[api reference]: https://mike.cousins.io/decoy/api/\n[usage guide]: https://mike.cousins.io/decoy/usage/create/\n[creating mocks]: https://mike.cousins.io/decoy/usage/create/\n[stubbing with when]: https://mike.cousins.io/decoy/usage/when/\n[spying with verify]: https://mike.cousins.io/decoy/usage/verify/\n',
    'author': 'Mike Cousins',
    'author_email': 'mike@cousins.io',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://mike.cousins.io/decoy/',
    'packages': packages,
    'package_data': package_data,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
