"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ErrorCode = void 0;
exports.addGravatarSupport = addGravatarSupport;
exports.addScope = addScope;
exports.buildToken = buildToken;
exports.combineBaseUrl = combineBaseUrl;
exports.convertDistRemoteToLocalTarballUrls = convertDistRemoteToLocalTarballUrls;
exports.convertPayloadToBase64 = convertPayloadToBase64;
exports.deleteProperties = deleteProperties;
exports.encodeScopedUri = encodeScopedUri;
exports.extractTarballFromUrl = extractTarballFromUrl;
exports.fileExists = fileExists;
exports.folderExists = folderExists;
exports.formatAuthor = formatAuthor;
exports.getLatestVersion = getLatestVersion;
exports.getLocalRegistryTarballUri = getLocalRegistryTarballUri;
exports.getPublicUrl = getPublicUrl;
exports.getUserAgent = getUserAgent;
exports.getVersion = getVersion;
exports.getVersionFromTarball = getVersionFromTarball;
exports.getWebProtocol = getWebProtocol;
exports.hasDiffOneKey = hasDiffOneKey;
exports.hasLogin = hasLogin;
exports.isHTTPProtocol = isHTTPProtocol;
exports.isHost = isHost;
exports.isObject = isObject;
exports.isObjectOrArray = isObjectOrArray;
exports.isRelatedToDeprecation = isRelatedToDeprecation;
exports.isVersionValid = isVersionValid;
exports.mask = mask;
exports.normalizeDistTags = normalizeDistTags;
exports.pad = pad;
exports.parseAddress = parseAddress;
exports.parseConfigFile = parseConfigFile;
exports.parseInterval = parseInterval;
exports.parseReadme = parseReadme;
exports.semverSort = semverSort;
exports.sortByName = sortByName;
exports.tagVersion = tagVersion;
exports.validateMetadata = validateMetadata;
exports.validateName = validateName;
exports.validatePackage = validatePackage;
exports.validateURL = validateURL;
exports.wrapPrefix = wrapPrefix;

var _assert = _interopRequireDefault(require("assert"));

var _debug = _interopRequireDefault(require("debug"));

var _fs = _interopRequireDefault(require("fs"));

var _jsYaml = _interopRequireDefault(require("js-yaml"));

var _lodash = _interopRequireDefault(require("lodash"));

var _memoizee = _interopRequireDefault(require("memoizee"));

var _semver = _interopRequireDefault(require("semver"));

var _url = _interopRequireWildcard(require("url"));

var _validator = _interopRequireDefault(require("validator"));

var _commonsApi = require("@verdaccio/commons-api");

var _readme = _interopRequireDefault(require("@verdaccio/readme"));

var _user = require("../utils/user");

var _constants = require("./constants");

var _logger = require("./logger");

var _storageUtils = require("./storage-utils");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const debug = (0, _debug.default)('verdaccio'); // eslint-disable-next-line @typescript-eslint/no-unused-vars
// eslint-disable-next-line @typescript-eslint/no-var-requires

require('pkginfo')(module);

const pkgVersion = module.exports.version;
const pkgName = module.exports.name;
const validProtocols = ['https', 'http'];

function getUserAgent(customUserAgent) {
  (0, _assert.default)(_lodash.default.isString(pkgName));
  (0, _assert.default)(_lodash.default.isString(pkgVersion));

  if (customUserAgent === true) {
    return `${pkgName}/${pkgVersion}`;
  } else if (_lodash.default.isString(customUserAgent) && _lodash.default.isEmpty(customUserAgent) === false) {
    return customUserAgent;
  } else if (customUserAgent === false) {
    return '';
  }

  return `${pkgName}/${pkgVersion}`;
}

function convertPayloadToBase64(payload) {
  return Buffer.from(payload, 'base64');
}
/**
 * From normalize-package-data/lib/fixer.js
 * @param {*} name  the package name
 * @return {Boolean} whether is valid or not
 */


function validateName(name) {
  if (_lodash.default.isString(name) === false) {
    return false;
  }

  const normalizedName = name.toLowerCase();
  /**
   * Some context about the first regex
   * - npm used to have a different tarball naming system.
   * eg: http://registry.npmjs.com/thirty-two
   * https://registry.npmjs.org/thirty-two/-/thirty-two@0.0.1.tgz
   * The file name thirty-two@0.0.1.tgz, the version and the pkg name was separated by an at (@)
   * while nowadays the naming system is based in dashes
   * https://registry.npmjs.org/verdaccio/-/verdaccio-1.4.0.tgz
   *
   * more info here: https://github.com/rlidwka/sinopia/issues/75
   */

  return !(!normalizedName.match(/^[-a-zA-Z0-9_.!~*'()@]+$/) || normalizedName.startsWith('.') || // ".bin", etc.
  ['node_modules', '__proto__', 'favicon.ico'].includes(normalizedName));
}
/**
 * Validate a package.
 * @return {Boolean} whether the package is valid or not
 */


function validatePackage(name) {
  const nameList = name.split('/', 2);

  if (nameList.length === 1) {
    // normal package
    return validateName(nameList[0]);
  } // scoped package


  return nameList[0][0] === '@' && validateName(nameList[0].slice(1)) && validateName(nameList[1]);
}
/**
 * Check whether an element is an Object
 * @param {*} obj the element
 * @return {Boolean}
 */


function isObject(obj) {
  return _lodash.default.isObject(obj) && _lodash.default.isNull(obj) === false && _lodash.default.isArray(obj) === false;
}

function isObjectOrArray(obj) {
  return _lodash.default.isObject(obj) && _lodash.default.isNull(obj) === false;
}
/**
 * Validate the package metadata, add additional properties whether are missing within
 * the metadata properties.
 * @param {*} object
 * @param {*} name
 * @return {Object} the object with additional properties as dist-tags ad versions
 */


function validateMetadata(object, name) {
  (0, _assert.default)(isObject(object), 'not a json object');

  _assert.default.strictEqual(object.name, name);

  if (!isObject(object[_constants.DIST_TAGS])) {
    object[_constants.DIST_TAGS] = {};
  }

  if (!isObject(object['versions'])) {
    object['versions'] = {};
  }

  if (!isObject(object['time'])) {
    object['time'] = {};
  }

  return object;
}

function extractTarballFromUrl(url) {
  // @ts-ignore
  return _url.default.parse(url).pathname.replace(/^.*\//, '');
}
/**
 * Iterate a packages's versions and filter each original tarball url.
 * @param {*} pkg
 * @param {*} req
 * @param {*} config
 * @return {String} a filtered package
 */


function convertDistRemoteToLocalTarballUrls(pkg, req, urlPrefix) {
  for (const ver in pkg.versions) {
    if (Object.prototype.hasOwnProperty.call(pkg.versions, ver)) {
      const distName = pkg.versions[ver].dist;

      if (_lodash.default.isNull(distName) === false && _lodash.default.isNull(distName.tarball) === false) {
        distName.tarball = getLocalRegistryTarballUri(distName.tarball, pkg.name, req, urlPrefix);
      }
    }
  }

  return pkg;
}

const memoizedgetPublicUrl = (0, _memoizee.default)(getPublicUrl);
/**
 * Filter a tarball url.
 * @param {*} uri
 * @return {String} a parsed url
 */

function getLocalRegistryTarballUri(uri, pkgName, req, urlPrefix) {
  const currentHost = req.get('host');

  if (!currentHost) {
    return uri;
  }

  const tarballName = extractTarballFromUrl(uri);
  const domainRegistry = memoizedgetPublicUrl(urlPrefix || '', req);
  return `${domainRegistry}${encodeScopedUri(pkgName)}/-/${tarballName}`;
}

function tagVersion(data, version, tag) {
  if (tag && data[_constants.DIST_TAGS][tag] !== version && _semver.default.parse(version, true)) {
    // valid version - store
    data[_constants.DIST_TAGS][tag] = version;
    return true;
  }

  return false;
}
/**
 * Gets version from a package object taking into account semver weirdness.
 * @return {String} return the semantic version of a package
 */


function getVersion(pkg, version) {
  // this condition must allow cast
  if (_lodash.default.isNil(pkg.versions[version]) === false) {
    return pkg.versions[version];
  }

  try {
    version = _semver.default.parse(version, true);

    for (const versionItem in pkg.versions) {
      // $FlowFixMe
      if (version.compare(_semver.default.parse(versionItem, true)) === 0) {
        return pkg.versions[versionItem];
      }
    }
  } catch (err) {
    return undefined;
  }
}
/**
 * Parse an internet address
 * Allow:
 - https:localhost:1234        - protocol + host + port
 - localhost:1234              - host + port
 - 1234                        - port
 - http::1234                  - protocol + port
 - https://localhost:443/      - full url + https
 - http://[::1]:443/           - ipv6
 - unix:/tmp/http.sock         - unix sockets
 - https://unix:/tmp/http.sock - unix sockets (https)
 * @param {*} urlAddress the internet address definition
 * @return {Object|Null} literal object that represent the address parsed
 */


function parseAddress(urlAddress) {
  //
  // TODO: refactor it to something more reasonable?
  //
  //        protocol :  //      (  host  )|(    ipv6     ):  port  /
  let urlPattern = /^((https?):(\/\/)?)?((([^\/:]*)|\[([^\[\]]+)\]):)?(\d+)\/?$/.exec(urlAddress);

  if (urlPattern) {
    return {
      proto: urlPattern[2] || _constants.DEFAULT_PROTOCOL,
      host: urlPattern[6] || urlPattern[7] || _constants.DEFAULT_DOMAIN,
      port: urlPattern[8] || _constants.DEFAULT_PORT
    };
  }

  urlPattern = /^((https?):(\/\/)?)?unix:(.*)$/.exec(urlAddress);

  if (urlPattern) {
    return {
      proto: urlPattern[2] || _constants.DEFAULT_PROTOCOL,
      path: urlPattern[4]
    };
  }

  return null;
}
/**
 * Function filters out bad semver versions and sorts the array.
 * @return {Array} sorted Array
 */


function semverSort(listVersions) {
  return listVersions.filter(function (x) {
    if (!_semver.default.parse(x, true)) {
      _logger.logger.warn({
        ver: x
      }, 'ignoring bad version @{ver}');

      return false;
    }

    return true;
  }) // FIXME: it seems the @types/semver do not handle a legitimate method named 'compareLoose'
  // @ts-ignore
  .sort(_semver.default.compareLoose).map(String);
}
/**
 * Flatten arrays of tags.
 * @param {*} data
 */


function normalizeDistTags(pkg) {
  let sorted;

  if (!pkg[_constants.DIST_TAGS].latest) {
    // overwrite latest with highest known version based on semver sort
    sorted = semverSort(Object.keys(pkg.versions));

    if (sorted && sorted.length) {
      pkg[_constants.DIST_TAGS].latest = sorted.pop();
    }
  }

  for (const tag in pkg[_constants.DIST_TAGS]) {
    if (_lodash.default.isArray(pkg[_constants.DIST_TAGS][tag])) {
      if (pkg[_constants.DIST_TAGS][tag].length) {
        // sort array
        // FIXME: this is clearly wrong, we need to research why this is like this.
        // @ts-ignore
        sorted = semverSort(pkg[_constants.DIST_TAGS][tag]);

        if (sorted.length) {
          // use highest version based on semver sort
          pkg[_constants.DIST_TAGS][tag] = sorted.pop();
        }
      } else {
        delete pkg[_constants.DIST_TAGS][tag];
      }
    } else if (_lodash.default.isString(pkg[_constants.DIST_TAGS][tag])) {
      if (!_semver.default.parse(pkg[_constants.DIST_TAGS][tag], true)) {
        // if the version is invalid, delete the dist-tag entry
        delete pkg[_constants.DIST_TAGS][tag];
      }
    }
  }
}

const parseIntervalTable = {
  '': 1000,
  ms: 1,
  s: 1000,
  m: 60 * 1000,
  h: 60 * 60 * 1000,
  d: 86400000,
  w: 7 * 86400000,
  M: 30 * 86400000,
  y: 365 * 86400000
};
/**
 * Parse an internal string to number
 * @param {*} interval
 * @return {Number}
 */

function parseInterval(interval) {
  if (typeof interval === 'number') {
    return interval * 1000;
  }

  let result = 0;
  let last_suffix = Infinity;
  interval.split(/\s+/).forEach(function (x) {
    if (!x) {
      return;
    }

    const m = x.match(/^((0|[1-9][0-9]*)(\.[0-9]+)?)(ms|s|m|h|d|w|M|y|)$/);

    if (!m || parseIntervalTable[m[4]] >= last_suffix || m[4] === '' && last_suffix !== Infinity) {
      throw Error('invalid interval: ' + interval);
    }

    last_suffix = parseIntervalTable[m[4]];
    result += Number(m[1]) * parseIntervalTable[m[4]];
  });
  return result;
}
/**
 * Detect running protocol (http or https)
 */


function getWebProtocol(headerProtocol, protocol) {
  let returnProtocol;
  const [, defaultProtocol] = validProtocols; // HAProxy variant might return http,http with X-Forwarded-Proto

  if (typeof headerProtocol === 'string' && headerProtocol !== '') {
    debug('header protocol: %o', protocol);
    const commaIndex = headerProtocol.indexOf(',');
    returnProtocol = commaIndex > 0 ? headerProtocol.substr(0, commaIndex) : headerProtocol;
  } else {
    debug('req protocol: %o', headerProtocol);
    returnProtocol = protocol;
  }

  return validProtocols.includes(returnProtocol) ? returnProtocol : defaultProtocol;
}

function getLatestVersion(pkgInfo) {
  return pkgInfo[_constants.DIST_TAGS].latest;
}

const ErrorCode = {
  getConflict: _commonsApi.getConflict,
  getBadData: _commonsApi.getBadData,
  getBadRequest: _commonsApi.getBadRequest,
  getInternalError: _commonsApi.getInternalError,
  getUnauthorized: _commonsApi.getUnauthorized,
  getForbidden: _commonsApi.getForbidden,
  getServiceUnavailable: _commonsApi.getServiceUnavailable,
  getNotFound: _commonsApi.getNotFound,
  getCode: _commonsApi.getCode
};
exports.ErrorCode = ErrorCode;

function parseConfigFile(configPath) {
  try {
    if (/\.ya?ml$/i.test(configPath)) {
      return _jsYaml.default.load(_fs.default.readFileSync(configPath, 'utf-8'));
    }

    debug('yaml parsed');
    return require(configPath);
  } catch (e) {
    debug('yaml parse failed');

    if (e.code !== 'MODULE_NOT_FOUND') {
      e.message = _constants.APP_ERROR.CONFIG_NOT_VALID;
    }

    throw new Error(e);
  }
}
/**
 * Check whether the path already exist.
 * @param {String} path
 * @return {Boolean}
 */


function folderExists(path) {
  try {
    const stat = _fs.default.statSync(path);

    return stat.isDirectory();
  } catch (_) {
    return false;
  }
}
/**
 * Check whether the file already exist.
 * @param {String} path
 * @return {Boolean}
 */


function fileExists(path) {
  try {
    const stat = _fs.default.statSync(path);

    return stat.isFile();
  } catch (_) {
    return false;
  }
}

function sortByName(packages, orderAscending = true) {
  return packages.slice().sort(function (a, b) {
    const comparatorNames = a.name.toLowerCase() < b.name.toLowerCase();
    return orderAscending ? comparatorNames ? -1 : 1 : comparatorNames ? 1 : -1;
  });
}

function addScope(scope, packageName) {
  return `@${scope}/${packageName}`;
}

function deleteProperties(propertiesToDelete, objectItem) {
  _lodash.default.forEach(propertiesToDelete, property => {
    delete objectItem[property];
  });

  return objectItem;
}

function addGravatarSupport(pkgInfo, online = true) {
  const pkgInfoCopy = _objectSpread({}, pkgInfo);

  const author = _lodash.default.get(pkgInfo, 'latest.author', null);

  const contributors = (0, _storageUtils.normalizeContributors)(_lodash.default.get(pkgInfo, 'latest.contributors', []));

  const maintainers = _lodash.default.get(pkgInfo, 'latest.maintainers', []); // for author.


  if (author && _lodash.default.isObject(author)) {
    const {
      email
    } = author;
    pkgInfoCopy.latest.author.avatar = (0, _user.generateGravatarUrl)(email, online);
  }

  if (author && _lodash.default.isString(author)) {
    pkgInfoCopy.latest.author = {
      avatar: _user.GENERIC_AVATAR,
      email: '',
      author
    };
  } // for contributors


  if (_lodash.default.isEmpty(contributors) === false) {
    pkgInfoCopy.latest.contributors = contributors.map(contributor => {
      if (isObject(contributor)) {
        contributor.avatar = (0, _user.generateGravatarUrl)(contributor.email, online);
      } else if (_lodash.default.isString(contributor)) {
        contributor = {
          avatar: _user.GENERIC_AVATAR,
          email: contributor,
          name: contributor
        };
      }

      return contributor;
    });
  } // for maintainers


  if (_lodash.default.isEmpty(maintainers) === false) {
    pkgInfoCopy.latest.maintainers = maintainers.map(maintainer => {
      maintainer.avatar = (0, _user.generateGravatarUrl)(maintainer.email, online);
      return maintainer;
    });
  }

  return pkgInfoCopy;
}
/**
 * parse package readme - markdown/ascii
 * @param {String} packageName name of package
 * @param {String} readme package readme
 * @param {Object} options sanitizyReadme options
 * @return {String} converted html template
 */


function parseReadme(packageName, readme, options = {}) {
  if (_lodash.default.isEmpty(readme) === false) {
    return (0, _readme.default)(readme, options);
  } // logs readme not found error


  _logger.logger.error({
    packageName
  }, '@{packageName}: No readme found');

  return (0, _readme.default)('ERROR: No README data found!');
}

function buildToken(type, token) {
  return `${_lodash.default.capitalize(type)} ${token}`;
}
/**
 * return package version from tarball name
 * @param {String} name
 * @returns {String}
 */


function getVersionFromTarball(name) {
  // FIXME: we know the regex is valid, but we should improve this part as ts suggest
  // @ts-ignore
  return /.+-(\d.+)\.tgz/.test(name) ? name.match(/.+-(\d.+)\.tgz/)[1] : undefined;
}

/**
 * Formats author field for webui.
 * @see https://docs.npmjs.com/files/package.json#author
 * @param {string|object|undefined} author
 */
function formatAuthor(author) {
  let authorDetails = {
    name: _constants.DEFAULT_USER,
    email: '',
    url: ''
  };

  if (_lodash.default.isNil(author)) {
    return authorDetails;
  }

  if (_lodash.default.isString(author)) {
    authorDetails = _objectSpread(_objectSpread({}, authorDetails), {}, {
      name: author
    });
  }

  if (_lodash.default.isObject(author)) {
    authorDetails = _objectSpread(_objectSpread({}, authorDetails), author);
  }

  return authorDetails;
}
/**
 * Check if URI is starting with "http://", "https://" or "//"
 * @param {string} uri
 */


function isHTTPProtocol(uri) {
  return /^(https?:)?\/\//.test(uri);
}
/**
 * Apply whitespaces based on the length
 * @param {*} str the log message
 * @return {String}
 */


function pad(str, max) {
  if (str.length < max) {
    return str + ' '.repeat(max - str.length);
  }

  return str;
}
/**
 * return a masquerade string with its first and last {charNum} and three dots in between.
 * @param {String} str
 * @param {Number} charNum
 * @returns {String}
 */


function mask(str, charNum = 3) {
  return `${str.substr(0, charNum)}...${str.substr(-charNum)}`;
}

function encodeScopedUri(packageName) {
  return packageName.replace(/\//g, '%2f');
}

function hasDiffOneKey(versions) {
  return Object.keys(versions).length !== 1;
}

function isVersionValid(packageMeta, packageVersion) {
  const hasVersion = typeof packageVersion !== 'undefined';

  if (!hasVersion) {
    return false;
  }

  const hasMatchVersion = Object.keys(packageMeta.versions).includes(packageVersion);
  return hasMatchVersion;
}

function isRelatedToDeprecation(pkgInfo) {
  const {
    versions
  } = pkgInfo;

  for (const version in versions) {
    if (Object.prototype.hasOwnProperty.call(versions[version], 'deprecated')) {
      return true;
    }
  }

  return false;
}

function validateURL(publicUrl) {
  try {
    const parsed = new _url.URL(publicUrl);

    if (!validProtocols.includes(parsed.protocol.replace(':', ''))) {
      throw Error('invalid protocol');
    }

    return true;
  } catch (err) {
    // TODO: add error logger here
    return false;
  }
}

function isHost(url = '', options = {}) {
  return _validator.default.isURL(url, _objectSpread({
    require_host: true,
    allow_trailing_dot: false,
    require_valid_protocol: false,
    // @ts-ignore
    require_port: false,
    require_tld: false
  }, options));
}

function getPublicUrl(url_prefix = '', req) {
  if (validateURL(process.env.VERDACCIO_PUBLIC_URL)) {
    const envURL = new _url.URL(wrapPrefix(url_prefix), process.env.VERDACCIO_PUBLIC_URL).href;
    debug('public url by env %o', envURL);
    return envURL;
  } else if (req.get('host')) {
    var _process$env$VERDACCI;

    const host = req.get('host');

    if (!isHost(host)) {
      throw new Error('invalid host');
    }

    const protoHeader = (_process$env$VERDACCI = process.env.VERDACCIO_FORWARDED_PROTO) !== null && _process$env$VERDACCI !== void 0 ? _process$env$VERDACCI : _constants.HEADERS.FORWARDED_PROTO;
    const protocol = getWebProtocol(req.get(protoHeader.toLowerCase()), req.protocol);
    const combinedUrl = combineBaseUrl(protocol, host, url_prefix);
    debug('public url by request %o', combinedUrl);
    return combinedUrl;
  } else {
    return '/';
  }
}
/**
 * Create base url for registry.
 * @return {String} base registry url
 */


function combineBaseUrl(protocol, host, prefix = '') {
  debug('combined protocol %o', protocol);
  debug('combined host %o', host);
  const newPrefix = wrapPrefix(prefix);
  debug('combined prefix %o', newPrefix);
  const groupedURI = new _url.URL(wrapPrefix(prefix), `${protocol}://${host}`);
  const result = groupedURI.href;
  debug('combined url %o', result);
  return result;
}

function wrapPrefix(prefix) {
  if (prefix === '' || typeof prefix === 'undefined' || prefix === null) {
    return '';
  } else if (!prefix.startsWith('/') && prefix.endsWith('/')) {
    return `/${prefix}`;
  } else if (!prefix.startsWith('/') && !prefix.endsWith('/')) {
    return `/${prefix}/`;
  } else if (prefix.startsWith('/') && !prefix.endsWith('/')) {
    return `${prefix}/`;
  } else {
    return prefix;
  }
}

function hasLogin(config) {
  var _config$web, _config$web2;

  // FIXME: types are not yet on the library verdaccio/monorepo
  // @ts-ignore
  return _lodash.default.isNil(config === null || config === void 0 ? void 0 : (_config$web = config.web) === null || _config$web === void 0 ? void 0 : _config$web.login) || (config === null || config === void 0 ? void 0 : (_config$web2 = config.web) === null || _config$web2 === void 0 ? void 0 : _config$web2.login) === true;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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