"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _assert = _interopRequireDefault(require("assert"));

var _async = _interopRequireDefault(require("async"));

var _lodash = _interopRequireDefault(require("lodash"));

var _stream = _interopRequireDefault(require("stream"));

var _streams = require("@verdaccio/streams");

var _logger = require("../lib/logger");

var _configUtils = require("./config-utils");

var _constants = require("./constants");

var _localStorage = _interopRequireDefault(require("./local-storage"));

var _metadataUtils = require("./metadata-utils");

var _search = _interopRequireDefault(require("./search"));

var _storageUtils = require("./storage-utils");

var _upStorage = _interopRequireDefault(require("./up-storage"));

var _uplinkUtil = require("./uplink-util");

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class Storage {
  constructor(config) {
    _defineProperty(this, "localStorage", void 0);

    _defineProperty(this, "config", void 0);

    _defineProperty(this, "logger", void 0);

    _defineProperty(this, "uplinks", void 0);

    _defineProperty(this, "filters", void 0);

    this.config = config;
    this.uplinks = (0, _uplinkUtil.setupUpLinks)(config);
    this.logger = _logger.logger;
    this.filters = []; // @ts-ignore

    this.localStorage = null;
  }

  init(config, filters = []) {
    this.filters = filters;
    this.localStorage = new _localStorage.default(this.config, _logger.logger);
    return this.localStorage.getSecret(config);
  }
  /**
   *  Add a {name} package to a system
   Function checks if package with the same name is available from uplinks.
   If it isn't, we create package locally
   Used storages: local (write) && uplinks
   */


  async addPackage(name, metadata, callback) {
    try {
      await (0, _storageUtils.checkPackageLocal)(name, this.localStorage);
      await (0, _storageUtils.checkPackageRemote)(name, this._isAllowPublishOffline(), this._syncUplinksMetadata.bind(this));
      await (0, _storageUtils.publishPackage)(name, metadata, this.localStorage);
      callback();
    } catch (err) {
      callback(err);
    }
  }

  _isAllowPublishOffline() {
    return typeof this.config.publish !== 'undefined' && _lodash.default.isBoolean(this.config.publish.allow_offline) && this.config.publish.allow_offline;
  }

  readTokens(filter) {
    return this.localStorage.readTokens(filter);
  }

  saveToken(token) {
    return this.localStorage.saveToken(token);
  }

  deleteToken(user, tokenKey) {
    return this.localStorage.deleteToken(user, tokenKey);
  }
  /**
   * Add a new version of package {name} to a system
   Used storages: local (write)
   */


  addVersion(name, version, metadata, tag, callback) {
    this.localStorage.addVersion(name, version, metadata, tag, callback);
  }
  /**
   * Tags a package version with a provided tag
   Used storages: local (write)
   */


  mergeTags(name, tagHash, callback) {
    this.localStorage.mergeTags(name, tagHash, callback);
  }
  /**
   * Change an existing package (i.e. unpublish one version)
   Function changes a package info from local storage and all uplinks with write access./
   Used storages: local (write)
   */


  changePackage(name, metadata, revision, callback) {
    this.localStorage.changePackage(name, metadata, revision, callback);
  }
  /**
   * Remove a package from a system
   Function removes a package from local storage
   Used storages: local (write)
   */


  removePackage(name, callback) {
    this.localStorage.removePackage(name, callback); // update the indexer

    _search.default.remove(name);
  }
  /**
   Remove a tarball from a system
   Function removes a tarball from local storage.
   Tarball in question should not be linked to in any existing
   versions, i.e. package version should be unpublished first.
   Used storage: local (write)
   */


  removeTarball(name, filename, revision, callback) {
    this.localStorage.removeTarball(name, filename, revision, callback);
  }
  /**
   * Upload a tarball for {name} package
   Function is synchronous and returns a WritableStream
   Used storages: local (write)
   */


  addTarball(name, filename) {
    return this.localStorage.addTarball(name, filename);
  }

  hasLocalTarball(name, filename) {
    const self = this;
    return new Promise((resolve, reject) => {
      let localStream = self.localStorage.getTarball(name, filename);
      let isOpen = false;
      localStream.on('error', err => {
        if (isOpen || err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
          reject(err);
        } // local reported 404 or request was aborted already


        if (localStream) {
          localStream.abort();
          localStream = null;
        }

        resolve(false);
      });
      localStream.on('open', function () {
        isOpen = true;
        localStream.abort();
        localStream = null;
        resolve(true);
      });
    });
  }
  /**
   Get a tarball from a storage for {name} package
   Function is synchronous and returns a ReadableStream
   Function tries to read tarball locally, if it fails then it reads package
   information in order to figure out where we can get this tarball from
   Used storages: local || uplink (just one)
   */


  getTarball(name, filename) {
    const readStream = new _streams.ReadTarball({});

    readStream.abort = function () {};

    const self = this; // if someone requesting tarball, it means that we should already have some
    // information about it, so fetching package info is unnecessary
    // trying local first
    // flow: should be IReadTarball

    let localStream = self.localStorage.getTarball(name, filename);
    let isOpen = false;
    localStream.on('error', err => {
      if (isOpen || err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
        return readStream.emit('error', err);
      } // local reported 404


      const err404 = err;
      localStream.abort();
      localStream = null; // we force for garbage collector

      self.localStorage.getPackageMetadata(name, (err, info) => {
        if (_lodash.default.isNil(err) && info._distfiles && _lodash.default.isNil(info._distfiles[filename]) === false) {
          // information about this file exists locally
          serveFile(info._distfiles[filename]);
        } else {
          // we know nothing about this file, trying to get information elsewhere
          self._syncUplinksMetadata(name, info, {}, (err, info) => {
            if (_lodash.default.isNil(err) === false) {
              return readStream.emit('error', err);
            }

            if (_lodash.default.isNil(info._distfiles) || _lodash.default.isNil(info._distfiles[filename])) {
              return readStream.emit('error', err404);
            }

            serveFile(info._distfiles[filename]);
          });
        }
      });
    });
    localStream.on('content-length', function (v) {
      readStream.emit('content-length', v);
    });
    localStream.on('open', function () {
      isOpen = true;
      localStream.pipe(readStream);
    });
    return readStream;
    /**
     * Fetch and cache local/remote packages.
     * @param {Object} file define the package shape
     */

    function serveFile(file) {
      let uplink = null;

      for (const uplinkId in self.uplinks) {
        if ((0, _configUtils.hasProxyTo)(name, uplinkId, self.config.packages)) {
          uplink = self.uplinks[uplinkId];
        }
      }

      if (uplink == null) {
        uplink = new _upStorage.default({
          url: file.url,
          cache: true,
          _autogenerated: true
        }, self.config);
      }

      let savestream = null;

      if (uplink.config.cache) {
        savestream = self.localStorage.addTarball(name, filename);
      }

      let on_open = function () {
        // prevent it from being called twice
        on_open = function () {};

        const rstream2 = uplink.fetchTarball(file.url);
        rstream2.on('error', function (err) {
          if (savestream) {
            savestream.abort();
          }

          savestream = null;
          readStream.emit('error', err);
        });
        rstream2.on('end', function () {
          if (savestream) {
            savestream.done();
          }
        });
        rstream2.on('content-length', function (v) {
          readStream.emit('content-length', v);

          if (savestream) {
            savestream.emit('content-length', v);
          }
        });
        rstream2.pipe(readStream);

        if (savestream) {
          rstream2.pipe(savestream);
        }
      };

      if (savestream) {
        savestream.on('open', function () {
          on_open();
        });
        savestream.on('error', function (err) {
          self.logger.warn({
            err: err,
            fileName: file
          }, 'error saving file @{fileName}: @{err.message}\n@{err.stack}');

          if (savestream) {
            savestream.abort();
          }

          savestream = null;
          on_open();
        });
      } else {
        on_open();
      }
    }
  }
  /**
   Retrieve a package metadata for {name} package
   Function invokes localStorage.getPackage and uplink.get_package for every
   uplink with proxy_access rights against {name} and combines results
   into one json object
   Used storages: local && uplink (proxy_access)
    * @param {object} options
   * @property {string} options.name Package Name
   * @property {object}  options.req Express `req` object
   * @property {boolean} options.keepUpLinkData keep up link info in package meta, last update, etc.
   * @property {function} options.callback Callback for receive data
   */


  getPackage(options) {
    this.localStorage.getPackageMetadata(options.name, (err, data) => {
      if (err && (!err.status || err.status >= _constants.HTTP_STATUS.INTERNAL_ERROR)) {
        // report internal errors right away
        return options.callback(err);
      }

      this._syncUplinksMetadata(options.name, data, {
        req: options.req,
        uplinksLook: options.uplinksLook
      }, function getPackageSynUpLinksCallback(err, result, uplinkErrors) {
        if (err) {
          return options.callback(err);
        }

        (0, _utils.normalizeDistTags)((0, _storageUtils.cleanUpLinksRef)(options.keepUpLinkData, result)); // npm can throw if this field doesn't exist

        result._attachments = {};
        options.callback(null, result, uplinkErrors);
      });
    });
  }
  /**
   Retrieve remote and local packages more recent than {startkey}
   Function streams all packages from all uplinks first, and then
   local packages.
   Note that local packages could override registry ones just because
   they appear in JSON last. That's a trade-off we make to avoid
   memory issues.
   Used storages: local && uplink (proxy_access)
   * @param {*} startkey
   * @param {*} options
   * @return {Stream}
   */


  search(startkey, options) {
    const self = this;
    const searchStream = new _stream.default.PassThrough({
      objectMode: true
    });

    _async.default.eachSeries(Object.keys(this.uplinks), function (up_name, cb) {
      var _options$req, _options$req$query;

      // shortcut: if `local=1` is supplied, don't call uplinks
      if (((_options$req = options.req) === null || _options$req === void 0 ? void 0 : (_options$req$query = _options$req.query) === null || _options$req$query === void 0 ? void 0 : _options$req$query.local) !== undefined) {
        return cb();
      }

      _logger.logger.info(`search for uplink ${up_name}`); // search by keyword for each uplink


      const uplinkStream = self.uplinks[up_name].search(options); // join uplink stream with streams PassThrough

      uplinkStream.pipe(searchStream, {
        end: false
      });
      uplinkStream.on('error', function (err) {
        self.logger.error({
          err: err
        }, 'uplink error: @{err.message}');
        cb(); // to avoid call callback more than once

        cb = function () {};
      });
      uplinkStream.on('end', function () {
        cb(); // to avoid call callback more than once

        cb = function () {};
      });

      searchStream.abort = function () {
        if (uplinkStream.abort) {
          uplinkStream.abort();
        }

        cb(); // to avoid call callback more than once

        cb = function () {};
      };
    }, // executed after all series
    function () {
      // attach a local search results
      const localSearchStream = self.localStorage.search(startkey, options);

      searchStream.abort = function () {
        localSearchStream.abort();
      };

      localSearchStream.pipe(searchStream, {
        end: true
      });
      localSearchStream.on('error', function (err) {
        self.logger.error({
          err: err
        }, 'search error: @{err.message}');
        searchStream.end();
      });
    });

    return searchStream;
  }
  /**
   * Retrieve only private local packages
   * @param {*} callback
   */


  getLocalDatabase(callback) {
    const self = this;
    this.localStorage.storagePlugin.get((err, locals) => {
      if (err) {
        callback(err);
      }

      const packages = [];

      const getPackage = function (itemPkg) {
        self.localStorage.getPackageMetadata(locals[itemPkg], function (err, pkgMetadata) {
          if (_lodash.default.isNil(err)) {
            const latest = pkgMetadata[_constants.DIST_TAGS].latest;

            if (latest && pkgMetadata.versions[latest]) {
              const version = pkgMetadata.versions[latest];
              const timeList = pkgMetadata.time;
              const time = timeList[latest]; // @ts-ignore

              version.time = time; // Add for stars api
              // @ts-ignore

              version.users = pkgMetadata.users;
              packages.push(version);
            } else {
              self.logger.warn({
                package: locals[itemPkg]
              }, 'package @{package} does not have a "latest" tag?');
            }
          }

          if (itemPkg >= locals.length - 1) {
            callback(null, packages);
          } else {
            getPackage(itemPkg + 1);
          }
        });
      };

      if (locals.length) {
        getPackage(0);
      } else {
        callback(null, []);
      }
    });
  }
  /**
   * Function fetches package metadata from uplinks and synchronizes it with local data
   if package is available locally, it MUST be provided in pkginfo
   returns callback(err, result, uplink_errors)
   */


  _syncUplinksMetadata(name, packageInfo, options, callback) {
    let found = true;
    const self = this;
    const upLinks = [];
    const hasToLookIntoUplinks = _lodash.default.isNil(options.uplinksLook) || options.uplinksLook;

    if (!packageInfo) {
      found = false;
      packageInfo = (0, _storageUtils.generatePackageTemplate)(name);
    }

    for (const uplink in this.uplinks) {
      if ((0, _configUtils.hasProxyTo)(name, uplink, this.config.packages) && hasToLookIntoUplinks) {
        upLinks.push(this.uplinks[uplink]);
      }
    }

    _async.default.map(upLinks, (upLink, cb) => {
      const _options = Object.assign({}, options);

      const upLinkMeta = packageInfo._uplinks[upLink.upname];

      if ((0, _utils.isObject)(upLinkMeta)) {
        const fetched = upLinkMeta.fetched;

        if (fetched && Date.now() - fetched < upLink.maxage) {
          return cb();
        }

        _options.etag = upLinkMeta.etag;
      }

      upLink.getRemoteMetadata(name, _options, (err, upLinkResponse, eTag) => {
        if (err && err.remoteStatus === 304) {
          upLinkMeta.fetched = Date.now();
        }

        if (err || !upLinkResponse) {
          return cb(null, [err || _utils.ErrorCode.getInternalError('no data')]);
        }

        try {
          (0, _utils.validateMetadata)(upLinkResponse, name);
        } catch (err) {
          self.logger.error({
            sub: 'out',
            err: err
          }, 'package.json validating error @{!err.message}\n@{err.stack}');
          return cb(null, [err]);
        }

        packageInfo._uplinks[upLink.upname] = {
          etag: eTag,
          fetched: Date.now()
        };
        packageInfo.time = (0, _storageUtils.mergeUplinkTimeIntoLocal)(packageInfo, upLinkResponse);
        (0, _uplinkUtil.updateVersionsHiddenUpLink)(upLinkResponse.versions, upLink);

        try {
          (0, _metadataUtils.mergeVersions)(packageInfo, upLinkResponse);
        } catch (err) {
          self.logger.error({
            sub: 'out',
            err: err
          }, 'package.json parsing error @{!err.message}\n@{err.stack}');
          return cb(null, [err]);
        } // if we got to this point, assume that the correct package exists
        // on the uplink


        found = true;
        cb();
      });
    }, // @ts-ignore
    (err, upLinksErrors) => {
      (0, _assert.default)(!err && Array.isArray(upLinksErrors)); // Check for connection timeout or reset errors with uplink(s)
      // (these should be handled differently from the package not being found)

      if (!found) {
        let uplinkTimeoutError;

        for (let i = 0; i < upLinksErrors.length; i++) {
          if (upLinksErrors[i]) {
            for (let j = 0; j < upLinksErrors[i].length; j++) {
              if (upLinksErrors[i][j]) {
                const code = upLinksErrors[i][j].code;

                if (code === 'ETIMEDOUT' || code === 'ESOCKETTIMEDOUT' || code === 'ECONNRESET') {
                  uplinkTimeoutError = true;
                  break;
                }
              }
            }
          }
        }

        if (uplinkTimeoutError) {
          return callback(_utils.ErrorCode.getServiceUnavailable(), null, upLinksErrors);
        }

        return callback(_utils.ErrorCode.getNotFound(_constants.API_ERROR.NO_PACKAGE), null, upLinksErrors);
      }

      if (upLinks.length === 0) {
        return callback(null, packageInfo);
      }

      self.localStorage.updateVersions(name, packageInfo, async (err, packageJsonLocal) => {
        if (err) {
          return callback(err);
        } // Any error here will cause a 404, like an uplink error. This is likely the right thing to do
        // as a broken filter is a security risk.


        const filterErrors = []; // This MUST be done serially and not in parallel as they modify packageJsonLocal

        for (const filter of self.filters) {
          try {
            // These filters can assume it's save to modify packageJsonLocal and return it directly for
            // performance (i.e. need not be pure)
            packageJsonLocal = await filter.filter_metadata(packageJsonLocal);
          } catch (err) {
            filterErrors.push(err);
          }
        }

        callback(null, packageJsonLocal, _lodash.default.concat(upLinksErrors, filterErrors));
      });
    });
  }
  /**
   * Set a hidden value for each version.
   * @param {Array} versions list of version
   * @param {String} upLink uplink name
   * @private
   */


  _updateVersionsHiddenUpLink(versions, upLink) {
    for (const i in versions) {
      if (Object.prototype.hasOwnProperty.call(versions, i)) {
        const version = versions[i]; // holds a "hidden" value to be used by the package storage.
        // $FlowFixMe

        version[Symbol.for('__verdaccio_uplink')] = upLink.upname;
      }
    }
  }

}

var _default = Storage;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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