"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WHITELIST = void 0;
exports.checkPackageLocal = checkPackageLocal;
exports.checkPackageRemote = checkPackageRemote;
exports.cleanUpLinksRef = cleanUpLinksRef;
exports.cleanUpReadme = cleanUpReadme;
exports.generatePackageTemplate = generatePackageTemplate;
exports.generateRevision = generateRevision;
exports.getLatestReadme = getLatestReadme;
exports.isPublishablePackage = isPublishablePackage;
exports.mergeUplinkTimeIntoLocal = mergeUplinkTimeIntoLocal;
exports.normalizeContributors = normalizeContributors;
exports.normalizePackage = normalizePackage;
exports.prepareSearchPackage = prepareSearchPackage;
exports.publishPackage = publishPackage;

var _lodash = _interopRequireDefault(require("lodash"));

var _cryptoUtils = require("../lib/crypto-utils");

var _constants = require("./constants");

var _search = _interopRequireDefault(require("./search"));

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function generatePackageTemplate(name) {
  return {
    // standard things
    name,
    versions: {},
    time: {},
    [_constants.USERS]: {},
    [_constants.DIST_TAGS]: {},
    _uplinks: {},
    _distfiles: {},
    _attachments: {},
    _rev: ''
  };
}
/**
 * Normalize package properties, tags, revision id.
 * @param {Object} pkg package reference.
 */


function normalizePackage(pkg) {
  const pkgProperties = ['versions', 'dist-tags', '_distfiles', '_attachments', '_uplinks', 'time'];
  pkgProperties.forEach(key => {
    const pkgProp = pkg[key];

    if (_lodash.default.isNil(pkgProp) || (0, _utils.isObject)(pkgProp) === false) {
      pkg[key] = {};
    }
  });

  if (_lodash.default.isString(pkg._rev) === false) {
    pkg._rev = _constants.STORAGE.DEFAULT_REVISION;
  }

  if (_lodash.default.isString(pkg._id) === false) {
    pkg._id = pkg.name;
  } // normalize dist-tags


  (0, _utils.normalizeDistTags)(pkg);
  return pkg;
}

function generateRevision(rev) {
  const _rev = rev.split('-');

  return (+_rev[0] || 0) + 1 + '-' + (0, _cryptoUtils.generateRandomHexString)();
}

function getLatestReadme(pkg) {
  const versions = pkg['versions'] || {};
  const distTags = pkg[_constants.DIST_TAGS] || {}; // FIXME: here is a bit tricky add the types

  const latestVersion = distTags['latest'] ? versions[distTags['latest']] || {} : {};

  let readme = _lodash.default.trim(pkg.readme || latestVersion.readme || '');

  if (readme) {
    return readme;
  } // In case of empty readme - trying to get ANY readme in the following order: 'next','beta','alpha','test','dev','canary'


  const readmeDistTagsPriority = ['next', 'beta', 'alpha', 'test', 'dev', 'canary'];
  readmeDistTagsPriority.map(function (tag) {
    if (readme) {
      return readme;
    }

    const version = distTags[tag] ? versions[distTags[tag]] || {} : {};
    readme = _lodash.default.trim(version.readme || readme);
  });
  return readme;
}

function cleanUpReadme(version) {
  if (_lodash.default.isNil(version) === false) {
    // @ts-ignore
    delete version.readme;
  }

  return version;
}

function normalizeContributors(contributors) {
  if (_lodash.default.isNil(contributors)) {
    return [];
  } else if (contributors && _lodash.default.isArray(contributors) === false) {
    // FIXME: this branch is clearly no an array, still tsc complains
    // @ts-ignore
    return [contributors];
  } else if (_lodash.default.isString(contributors)) {
    return [{
      name: contributors
    }];
  }

  return contributors;
}

const WHITELIST = ['_rev', 'name', 'versions', 'dist-tags', 'readme', 'time', '_id', 'users'];
exports.WHITELIST = WHITELIST;

function cleanUpLinksRef(keepUpLinkData, result) {
  const propertyToKeep = [...WHITELIST];

  if (keepUpLinkData === true) {
    propertyToKeep.push('_uplinks');
  }

  for (const i in result) {
    if (propertyToKeep.indexOf(i) === -1) {
      // Remove sections like '_uplinks' from response
      delete result[i];
    }
  }

  return result;
}
/**
 * Check whether a package it is already a local package
 * @param {*} name
 * @param {*} localStorage
 */


function checkPackageLocal(name, localStorage) {
  return new Promise((resolve, reject) => {
    localStorage.getPackageMetadata(name, (err, results) => {
      if (!_lodash.default.isNil(err) && err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
        return reject(err);
      }

      if (results) {
        return reject(_utils.ErrorCode.getConflict(_constants.API_ERROR.PACKAGE_EXIST));
      }

      return resolve();
    });
  });
}

function publishPackage(name, metadata, localStorage) {
  return new Promise((resolve, reject) => {
    localStorage.addPackage(name, metadata, (err, latest) => {
      if (!_lodash.default.isNull(err)) {
        return reject(err);
      } else if (!_lodash.default.isUndefined(latest)) {
        _search.default.add(latest);
      }

      return resolve();
    });
  });
}

function checkPackageRemote(name, isAllowPublishOffline, syncMetadata) {
  return new Promise((resolve, reject) => {
    syncMetadata(name, null, {}, (err, packageJsonLocal, upLinksErrors) => {
      // something weird
      if (err && err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
        return reject(err);
      } // checking package exist already


      if (_lodash.default.isNil(packageJsonLocal) === false) {
        return reject(_utils.ErrorCode.getConflict(_constants.API_ERROR.PACKAGE_EXIST));
      }

      for (let errorItem = 0; errorItem < upLinksErrors.length; errorItem++) {
        // checking error
        // if uplink fails with a status other than 404, we report failure
        if (_lodash.default.isNil(upLinksErrors[errorItem][0]) === false) {
          if (upLinksErrors[errorItem][0].status !== _constants.HTTP_STATUS.NOT_FOUND) {
            if (isAllowPublishOffline) {
              return resolve();
            }

            return reject(_utils.ErrorCode.getServiceUnavailable(_constants.API_ERROR.UPLINK_OFFLINE_PUBLISH));
          }
        }
      }

      return resolve();
    });
  });
}

function mergeUplinkTimeIntoLocal(localMetadata, remoteMetadata) {
  if ('time' in remoteMetadata) {
    return Object.assign({}, localMetadata.time, remoteMetadata.time);
  }

  return localMetadata.time;
}

function prepareSearchPackage(data, time) {
  const listVersions = Object.keys(data.versions);
  const versions = (0, _utils.semverSort)(listVersions);
  const latest = data[_constants.DIST_TAGS] && data[_constants.DIST_TAGS].latest ? data[_constants.DIST_TAGS].latest : versions.pop();

  if (latest && data.versions[latest]) {
    const version = data.versions[latest];
    const versions = {
      [latest]: 'latest'
    };
    const pkg = {
      name: version.name,
      description: version.description,
      [_constants.DIST_TAGS]: {
        latest
      },
      maintainers: version.maintainers || [version.author].filter(Boolean),
      author: version.author,
      repository: version.repository,
      readmeFilename: version.readmeFilename || '',
      homepage: version.homepage,
      keywords: version.keywords,
      bugs: version.bugs,
      license: version.license,
      time: {
        modified: time
      },
      versions
    };
    return pkg;
  }
}
/**
 * Check whether the package metadta has enough data to be published
 * @param pkg metadata
 */


function isPublishablePackage(pkg) {
  const keys = Object.keys(pkg);
  return _lodash.default.includes(keys, 'versions');
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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