"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _assert = _interopRequireDefault(require("assert"));

var _debug = _interopRequireDefault(require("debug"));

var _lodash = _interopRequireDefault(require("lodash"));

var _url = _interopRequireDefault(require("url"));

var _localStorage = _interopRequireDefault(require("@verdaccio/local-storage"));

var _streams = require("@verdaccio/streams");

var _pluginLoader = _interopRequireDefault(require("../lib/plugin-loader"));

var _constants = require("./constants");

var _cryptoUtils = require("./crypto-utils");

var _storageUtils = require("./storage-utils");

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const debug = (0, _debug.default)('verdaccio:local-storage');
/**
 * Implements Storage interface (same for storage.js, local-storage.js, up-storage.js).
 */

class LocalStorage {
  constructor(config, logger) {
    _defineProperty(this, "config", void 0);

    _defineProperty(this, "storagePlugin", void 0);

    _defineProperty(this, "logger", void 0);

    this.logger = logger;
    this.config = config;
    this.storagePlugin = this._loadStorage(config, logger);
  }

  addPackage(name, pkg, callback) {
    const storage = this._getLocalStorage(name);

    if (_lodash.default.isNil(storage)) {
      return callback(_utils.ErrorCode.getNotFound('this package cannot be added'));
    }

    storage.createPackage(name, (0, _storageUtils.generatePackageTemplate)(name), err => {
      // FIXME: it will be fixed here https://github.com/verdaccio/verdaccio/pull/1360
      // @ts-ignore
      if (_lodash.default.isNull(err) === false && (err.code === _constants.STORAGE.FILE_EXIST_ERROR || err.code === _constants.HTTP_STATUS.CONFLICT)) {
        return callback(_utils.ErrorCode.getConflict());
      }

      const latest = (0, _utils.getLatestVersion)(pkg);

      if (_lodash.default.isNil(latest) === false && pkg.versions[latest]) {
        return callback(null, pkg.versions[latest]);
      }

      return callback();
    });
  }
  /**
   * Remove package.
   * @param {*} name
   * @param {*} callback
   * @return {Function}
   */


  removePackage(name, callback) {
    const storage = this._getLocalStorage(name);

    debug('[storage] removing package %o', name);

    if (_lodash.default.isNil(storage)) {
      return callback(_utils.ErrorCode.getNotFound());
    }

    storage.readPackage(name, (err, data) => {
      if (_lodash.default.isNil(err) === false) {
        if (err.code === _constants.STORAGE.NO_SUCH_FILE_ERROR || err.code === _constants.HTTP_STATUS.NOT_FOUND) {
          return callback(_utils.ErrorCode.getNotFound());
        }

        return callback(err);
      }

      data = (0, _storageUtils.normalizePackage)(data);
      this.storagePlugin.remove(name, removeFailed => {
        if (removeFailed) {
          // This will happen when database is locked
          this.logger.error({
            name
          }, `[storage/removePackage] the database is locked, removed has failed for @{name}`);
          return callback(_utils.ErrorCode.getBadData(removeFailed.message));
        }

        storage.deletePackage(_constants.STORAGE.PACKAGE_FILE_NAME, err => {
          if (err) {
            return callback(err);
          }

          const attachments = Object.keys(data._attachments);

          this._deleteAttachments(storage, attachments, callback);
        });
      });
    });
  }
  /**
   * Synchronize remote package info with the local one
   * @param {*} name
   * @param {*} packageInfo
   * @param {*} callback
   */


  updateVersions(name, packageInfo, callback) {
    this._readCreatePackage(name, (err, packageLocalJson) => {
      if (err) {
        return callback(err);
      }

      let change = false; // updating readme

      packageLocalJson.readme = (0, _storageUtils.getLatestReadme)(packageInfo);

      if (packageInfo.readme !== packageLocalJson.readme) {
        change = true;
      }

      for (const versionId in packageInfo.versions) {
        if (_lodash.default.isNil(packageLocalJson.versions[versionId])) {
          let version = packageInfo.versions[versionId]; // we don't keep readme for package versions,
          // only one readme per package

          version = (0, _storageUtils.cleanUpReadme)(version);
          version.contributors = (0, _storageUtils.normalizeContributors)(version.contributors);
          change = true;
          packageLocalJson.versions[versionId] = version;

          if (version.dist && version.dist.tarball) {
            const urlObject = _url.default.parse(version.dist.tarball);

            const filename = urlObject.pathname.replace(/^.*\//, ''); // we do NOT overwrite any existing records

            if (_lodash.default.isNil(packageLocalJson._distfiles[filename])) {
              const hash = packageLocalJson._distfiles[filename] = {
                url: version.dist.tarball,
                sha: version.dist.shasum
              };
              /* eslint spaced-comment: 0 */
              // $FlowFixMe

              const upLink = version[Symbol.for('__verdaccio_uplink')];

              if (_lodash.default.isNil(upLink) === false) {
                this._updateUplinkToRemoteProtocol(hash, upLink);
              }
            }
          }
        }
      }

      for (const tag in packageInfo[_constants.DIST_TAGS]) {
        if (!packageLocalJson[_constants.DIST_TAGS][tag] || packageLocalJson[_constants.DIST_TAGS][tag] !== packageInfo[_constants.DIST_TAGS][tag]) {
          change = true;
          packageLocalJson[_constants.DIST_TAGS][tag] = packageInfo[_constants.DIST_TAGS][tag];
        }
      }

      for (const up in packageInfo._uplinks) {
        if (Object.prototype.hasOwnProperty.call(packageInfo._uplinks, up)) {
          const need_change = !(0, _utils.isObject)(packageLocalJson._uplinks[up]) || packageInfo._uplinks[up].etag !== packageLocalJson._uplinks[up].etag || packageInfo._uplinks[up].fetched !== packageLocalJson._uplinks[up].fetched;

          if (need_change) {
            change = true;
            packageLocalJson._uplinks[up] = packageInfo._uplinks[up];
          }
        }
      }

      if ('time' in packageInfo && !_lodash.default.isEqual(packageLocalJson.time, packageInfo.time)) {
        packageLocalJson.time = packageInfo.time;
        change = true;
      }

      if (change) {
        debug('updating package %o info', name);

        this._writePackage(name, packageLocalJson, function (err) {
          callback(err, packageLocalJson);
        });
      } else {
        callback(null, packageLocalJson);
      }
    });
  }
  /**
   * Add a new version to a previous local package.
   * @param {*} name
   * @param {*} version
   * @param {*} metadata
   * @param {*} tag
   * @param {*} callback
   */


  addVersion(name, version, metadata, tag, callback) {
    this._updatePackage(name, (data, cb) => {
      // keep only one readme per package
      data.readme = metadata.readme; // TODO: lodash remove

      metadata = (0, _storageUtils.cleanUpReadme)(metadata);
      metadata.contributors = (0, _storageUtils.normalizeContributors)(metadata.contributors);
      const hasVersion = data.versions[version] != null;

      if (hasVersion) {
        return cb(_utils.ErrorCode.getConflict());
      } // if uploaded tarball has a different shasum, it's very likely that we have some kind of error


      if ((0, _utils.isObject)(metadata.dist) && _lodash.default.isString(metadata.dist.tarball)) {
        const tarball = metadata.dist.tarball.replace(/.*\//, '');

        if ((0, _utils.isObject)(data._attachments[tarball])) {
          if (_lodash.default.isNil(data._attachments[tarball].shasum) === false && _lodash.default.isNil(metadata.dist.shasum) === false) {
            if (data._attachments[tarball].shasum != metadata.dist.shasum) {
              const errorMessage = `shasum error, ${data._attachments[tarball].shasum} != ${metadata.dist.shasum}`;
              return cb(_utils.ErrorCode.getBadRequest(errorMessage));
            }
          }

          const currentDate = new Date().toISOString(); // some old storage do not have this field #740

          if (_lodash.default.isNil(data.time)) {
            data.time = {};
          }

          data.time['modified'] = currentDate;

          if ('created' in data.time === false) {
            data.time.created = currentDate;
          }

          data.time[version] = currentDate;
          data._attachments[tarball].version = version;
        }
      }

      data.versions[version] = metadata;
      (0, _utils.tagVersion)(data, version, tag);
      this.storagePlugin.add(name, addFailed => {
        if (addFailed) {
          return cb(_utils.ErrorCode.getBadData(addFailed.message));
        }

        cb();
      });
    }, callback);
  }
  /**
   * Merge a new list of tags for a local packages with the existing one.
   * @param {*} pkgName
   * @param {*} tags
   * @param {*} callback
   */


  mergeTags(pkgName, tags, callback) {
    this._updatePackage(pkgName, (data, cb) => {
      /* eslint guard-for-in: 0 */
      for (const tag in tags) {
        // this handle dist-tag rm command
        if (_lodash.default.isNull(tags[tag])) {
          delete data[_constants.DIST_TAGS][tag];
          continue;
        }

        if (_lodash.default.isNil(data.versions[tags[tag]])) {
          return cb(this._getVersionNotFound());
        }

        const version = tags[tag];
        (0, _utils.tagVersion)(data, version, tag);
      }

      cb(null);
    }, callback);
  }
  /**
   * Return version not found
   * @return {String}
   * @private
   */


  _getVersionNotFound() {
    return _utils.ErrorCode.getNotFound(_constants.API_ERROR.VERSION_NOT_EXIST);
  }
  /**
   * Return file no available
   * @return {String}
   * @private
   */


  _getFileNotAvailable() {
    return _utils.ErrorCode.getNotFound('no such file available');
  }
  /**
   * Update the package metadata, tags and attachments (tarballs).
   * Note: Currently supports unpublishing and deprecation.
   * @param {*} name
   * @param {*} incomingPkg
   * @param {*} revision
   * @param {*} callback
   * @return {Function}
   */


  changePackage(name, incomingPkg, revision, callback) {
    if (!(0, _utils.isObject)(incomingPkg.versions) || !(0, _utils.isObject)(incomingPkg[_constants.DIST_TAGS])) {
      this.logger.error({
        name
      }, `changePackage bad data for @{name}`);
      return callback(_utils.ErrorCode.getBadData());
    }

    debug('changePackage udapting package for %o', name);

    this._updatePackage(name, (localData, cb) => {
      for (const version in localData.versions) {
        const incomingVersion = incomingPkg.versions[version];

        if (_lodash.default.isNil(incomingVersion)) {
          this.logger.info({
            name: name,
            version: version
          }, 'unpublishing @{name}@@{version}'); // FIXME: I prefer return a new object rather mutate the metadata

          delete localData.versions[version];
          delete localData.time[version];

          for (const file in localData._attachments) {
            if (localData._attachments[file].version === version) {
              delete localData._attachments[file].version;
            }
          }
        } else if (Object.prototype.hasOwnProperty.call(incomingVersion, 'deprecated')) {
          const incomingDeprecated = incomingVersion.deprecated;

          if (incomingDeprecated != localData.versions[version].deprecated) {
            if (!incomingDeprecated) {
              this.logger.info({
                name: name,
                version: version
              }, 'undeprecating @{name}@@{version}');
              delete localData.versions[version].deprecated;
            } else {
              this.logger.info({
                name: name,
                version: version
              }, 'deprecating @{name}@@{version}');
              localData.versions[version].deprecated = incomingDeprecated;
            }

            localData.time.modified = new Date().toISOString();
          }
        }
      }

      localData[_constants.USERS] = incomingPkg[_constants.USERS];
      localData[_constants.DIST_TAGS] = incomingPkg[_constants.DIST_TAGS];
      cb(null);
    }, function (err) {
      if (err) {
        return callback(err);
      }

      callback();
    });
  }
  /**
   * Remove a tarball.
   * @param {*} name
   * @param {*} filename
   * @param {*} revision
   * @param {*} callback
   */


  removeTarball(name, filename, revision, callback) {
    (0, _assert.default)((0, _utils.validateName)(filename));

    this._updatePackage(name, (data, cb) => {
      if (data._attachments[filename]) {
        delete data._attachments[filename];
        cb(null);
      } else {
        cb(this._getFileNotAvailable());
      }
    }, err => {
      if (err) {
        return callback(err);
      }

      const storage = this._getLocalStorage(name);

      if (storage) {
        storage.deletePackage(filename, callback);
      }
    });
  }
  /**
   * Add a tarball.
   * @param {String} name
   * @param {String} filename
   * @return {Stream}
   */


  addTarball(name, filename) {
    (0, _assert.default)((0, _utils.validateName)(filename));
    let length = 0;
    const shaOneHash = (0, _cryptoUtils.createTarballHash)();
    const uploadStream = new _streams.UploadTarball({});
    const _transform = uploadStream._transform;

    const storage = this._getLocalStorage(name);

    uploadStream.abort = function () {};

    uploadStream.done = function () {};

    uploadStream._transform = function (data, ...args) {
      shaOneHash.update(data); // measure the length for validation reasons

      length += data.length;
      const appliedData = [data, ...args]; // FIXME: not sure about this approach, tsc complains
      // @ts-ignore

      _transform.apply(uploadStream, appliedData);
    };

    if (name === '__proto__') {
      process.nextTick(() => {
        uploadStream.emit('error', _utils.ErrorCode.getForbidden());
      });
      return uploadStream;
    }

    if (!storage) {
      process.nextTick(() => {
        uploadStream.emit('error', "can't upload this package");
      });
      return uploadStream;
    }

    const writeStream = storage.writeTarball(filename);
    writeStream.on('error', err => {
      // @ts-ignore
      if (err.code === _constants.STORAGE.FILE_EXIST_ERROR || err.code === _constants.HTTP_STATUS.CONFLICT) {
        uploadStream.emit('error', _utils.ErrorCode.getConflict());
        uploadStream.abort(); // @ts-ignore
      } else if (err.code === _constants.STORAGE.NO_SUCH_FILE_ERROR || err.code === _constants.HTTP_STATUS.NOT_FOUND) {
        // check if package exists to throw an appropriate message
        this.getPackageMetadata(name, function (_err, _res) {
          if (_err) {
            uploadStream.emit('error', _err);
          } else {
            uploadStream.emit('error', err);
          }
        });
      } else {
        uploadStream.emit('error', err);
      }
    });
    writeStream.on('open', function () {
      // re-emitting open because it's handled in storage.js
      uploadStream.emit('open');
    });
    writeStream.on('success', () => {
      this._updatePackage(name, function updater(data, cb) {
        data._attachments[filename] = {
          shasum: shaOneHash.digest('hex')
        };
        cb(null);
      }, function (err) {
        if (err) {
          uploadStream.emit('error', err);
        } else {
          uploadStream.emit('success');
        }
      });
    });

    uploadStream.abort = function () {
      writeStream.abort();
    };

    uploadStream.done = function () {
      if (!length) {
        uploadStream.emit('error', _utils.ErrorCode.getBadData('refusing to accept zero-length file'));
        writeStream.abort();
      } else {
        writeStream.done();
      }
    };

    uploadStream.pipe(writeStream);
    return uploadStream;
  }
  /**
   * Get a tarball.
   * @param {*} name
   * @param {*} filename
   * @return {ReadTarball}
   */


  getTarball(name, filename) {
    (0, _assert.default)((0, _utils.validateName)(filename));

    const storage = this._getLocalStorage(name);

    if (_lodash.default.isNil(storage)) {
      return this._createFailureStreamResponse();
    }

    return this._streamSuccessReadTarBall(storage, filename);
  }
  /**
   * Return a stream that emits a read failure.
   * @private
   * @return {ReadTarball}
   */


  _createFailureStreamResponse() {
    const stream = new _streams.ReadTarball({});
    process.nextTick(() => {
      stream.emit('error', this._getFileNotAvailable());
    });
    return stream;
  }
  /**
   * Return a stream that emits the tarball data
   * @param {Object} storage
   * @param {String} filename
   * @private
   * @return {ReadTarball}
   */


  _streamSuccessReadTarBall(storage, filename) {
    const stream = new _streams.ReadTarball({});
    const readTarballStream = storage.readTarball(filename);
    const e404 = _utils.ErrorCode.getNotFound;

    stream.abort = function () {
      if (_lodash.default.isNil(readTarballStream) === false) {
        readTarballStream.abort();
      }
    };

    readTarballStream.on('error', function (err) {
      // @ts-ignore
      if (err.code === _constants.STORAGE.NO_SUCH_FILE_ERROR || err.code === _constants.HTTP_STATUS.NOT_FOUND) {
        stream.emit('error', e404('no such file available'));
      } else {
        stream.emit('error', err);
      }
    });
    readTarballStream.on('content-length', function (content) {
      stream.emit('content-length', content);
    });
    readTarballStream.on('open', function () {
      // re-emitting open because it's handled in storage.js
      stream.emit('open');
      readTarballStream.pipe(stream);
    });
    return stream;
  }
  /**
   * Retrieve a package by name.
   * @param {*} name
   * @param {*} callback
   * @return {Function}
   */


  getPackageMetadata(name, callback = () => {}) {
    const storage = this._getLocalStorage(name);

    if (_lodash.default.isNil(storage)) {
      return callback(_utils.ErrorCode.getNotFound());
    }

    this._readPackage(name, storage, callback);
  }
  /**
   * Search a local package.
   * @param {*} startKey
   * @param {*} options
   * @return {Function}
   */


  search(startKey, options) {
    const stream = new _streams.ReadTarball({
      objectMode: true
    });

    this._searchEachPackage((item, cb) => {
      // @ts-ignore
      if (item.time > parseInt(startKey, 10)) {
        this.getPackageMetadata(item.name, (err, data) => {
          if (err) {
            return cb(err);
          } // @ts-ignore


          const time = new Date(item.time).toISOString();
          const result = (0, _storageUtils.prepareSearchPackage)(data, time);

          if (_lodash.default.isNil(result) === false) {
            stream.push(result);
          }

          cb(null);
        });
      } else {
        cb(null);
      }
    }, function onEnd(err) {
      if (err) {
        stream.emit('error', err);
        return;
      }

      stream.end();
    });

    return stream;
  }
  /**
   * Retrieve a wrapper that provide access to the package location.
   * @param {Object} pkgName package name.
   * @return {Object}
   */


  _getLocalStorage(pkgName) {
    return this.storagePlugin.getPackageStorage(pkgName);
  }
  /**
   * Read a json file from storage.
   * @param {Object} storage
   * @param {Function} callback
   */


  _readPackage(name, storage, callback) {
    storage.readPackage(name, (err, result) => {
      if (err) {
        if (err.code === _constants.STORAGE.NO_SUCH_FILE_ERROR || err.code === _constants.HTTP_STATUS.NOT_FOUND) {
          return callback(_utils.ErrorCode.getNotFound());
        }

        return callback(this._internalError(err, _constants.STORAGE.PACKAGE_FILE_NAME, 'error reading'));
      }

      callback(err, (0, _storageUtils.normalizePackage)(result));
    });
  }
  /**
   * Walks through each package and calls `on_package` on them.
   * @param {*} onPackage
   * @param {*} onEnd
   */


  _searchEachPackage(onPackage, onEnd) {
    // save wait whether plugin still do not support search functionality
    if (_lodash.default.isNil(this.storagePlugin.search)) {
      this.logger.warn('plugin search not implemented yet');
      onEnd();
    } else {
      this.storagePlugin.search(onPackage, onEnd, _utils.validateName);
    }
  }
  /**
   * Retrieve either a previous created local package or a boilerplate.
   * @param {*} pkgName
   * @param {*} callback
   * @return {Function}
   */


  _readCreatePackage(pkgName, callback) {
    const storage = this._getLocalStorage(pkgName);

    if (_lodash.default.isNil(storage)) {
      this._createNewPackage(pkgName, callback);

      return;
    }

    storage.readPackage(pkgName, (err, data) => {
      // TODO: race condition
      if (_lodash.default.isNil(err) === false) {
        if (err.code === _constants.STORAGE.NO_SUCH_FILE_ERROR || err.code === _constants.HTTP_STATUS.NOT_FOUND) {
          data = (0, _storageUtils.generatePackageTemplate)(pkgName);
        } else {
          return callback(this._internalError(err, _constants.STORAGE.PACKAGE_FILE_NAME, 'error reading'));
        }
      }

      callback(null, (0, _storageUtils.normalizePackage)(data));
    });
  }

  _createNewPackage(name, callback) {
    return callback(null, (0, _storageUtils.normalizePackage)((0, _storageUtils.generatePackageTemplate)(name)));
  }
  /**
   * Handle internal error
   * @param {*} err
   * @param {*} file
   * @param {*} message
   * @return {Object} Error instance
   */


  _internalError(err, file, message) {
    this.logger.error({
      err: err,
      file: file
    }, `${message}  @{file}: @{!err.message}`);
    return _utils.ErrorCode.getInternalError();
  }
  /**
   * @param {*} name package name
   * @param {*} updateHandler function(package, cb) - update function
   * @param {*} callback callback that gets invoked after it's all updated
   * @return {Function}
   */


  _updatePackage(name, updateHandler, callback) {
    const storage = this._getLocalStorage(name);

    if (!storage) {
      return callback(_utils.ErrorCode.getNotFound());
    }

    storage.updatePackage(name, updateHandler, this._writePackage.bind(this), _storageUtils.normalizePackage, callback);
  }
  /**
   * Update the revision (_rev) string for a package.
   * @param {*} name
   * @param {*} json
   * @param {*} callback
   * @return {Function}
   */


  _writePackage(name, json, callback) {
    const storage = this._getLocalStorage(name);

    if (_lodash.default.isNil(storage)) {
      return callback();
    }

    storage.savePackage(name, this._setDefaultRevision(json), callback);
  }

  _setDefaultRevision(json) {
    // calculate revision from couch db
    if (_lodash.default.isString(json._rev) === false) {
      json._rev = _constants.STORAGE.DEFAULT_REVISION;
    } // this is intended in debug mode we do not want modify the store revision


    if (_lodash.default.isNil(this.config._debug)) {
      json._rev = (0, _storageUtils.generateRevision)(json._rev);
    }

    return json;
  }

  _deleteAttachments(storage, attachments, callback) {
    debug('[storage/_deleteAttachments] delete attachments total: %o', attachments === null || attachments === void 0 ? void 0 : attachments.length);

    const unlinkNext = function (cb) {
      if (_lodash.default.isEmpty(attachments)) {
        return cb();
      }

      const attachment = attachments.shift();
      storage.deletePackage(attachment, function () {
        unlinkNext(cb);
      });
    };

    unlinkNext(function () {
      // try to unlink the directory, but ignore errors because it can fail
      storage.removePackage(function (err) {
        callback(err);
      });
    });
  }
  /**
   * Ensure the dist file remains as the same protocol
   * @param {Object} hash metadata
   * @param {String} upLinkKey registry key
   * @private
   */


  _updateUplinkToRemoteProtocol(hash, upLinkKey) {
    // if we got this information from a known registry,
    // use the same protocol for the tarball
    //
    // see https://github.com/rlidwka/sinopia/issues/166
    const tarballUrl = _url.default.parse(hash.url);

    const uplinkUrl = _url.default.parse(this.config.uplinks[upLinkKey].url);

    if (uplinkUrl.host === tarballUrl.host) {
      tarballUrl.protocol = uplinkUrl.protocol;
      hash.registry = upLinkKey;
      hash.url = _url.default.format(tarballUrl);
    }
  }

  async getSecret(config) {
    const secretKey = await this.storagePlugin.getSecret();
    return this.storagePlugin.setSecret(config.checkSecretKey(secretKey));
  }

  _loadStorage(config, logger) {
    const Storage = this._loadStorePlugin();

    if (_lodash.default.isNil(Storage)) {
      (0, _assert.default)(this.config.storage, 'CONFIG: storage path not defined');
      return new _localStorage.default(this.config, logger);
    }

    return Storage;
  }

  _loadStorePlugin() {
    const plugin_params = {
      config: this.config,
      logger: this.logger
    }; // eslint-disable-next-line max-len

    const plugins = (0, _pluginLoader.default)(this.config, this.config.store, plugin_params, plugin => {
      return plugin.getPackageStorage;
    });
    return _lodash.default.head(plugins);
  }

  saveToken(token) {
    if (_lodash.default.isFunction(this.storagePlugin.saveToken) === false) {
      return Promise.reject(_utils.ErrorCode.getCode(_constants.HTTP_STATUS.SERVICE_UNAVAILABLE, _constants.SUPPORT_ERRORS.PLUGIN_MISSING_INTERFACE));
    }

    return this.storagePlugin.saveToken(token);
  }

  deleteToken(user, tokenKey) {
    if (_lodash.default.isFunction(this.storagePlugin.deleteToken) === false) {
      return Promise.reject(_utils.ErrorCode.getCode(_constants.HTTP_STATUS.SERVICE_UNAVAILABLE, _constants.SUPPORT_ERRORS.PLUGIN_MISSING_INTERFACE));
    }

    return this.storagePlugin.deleteToken(user, tokenKey);
  }

  readTokens(filter) {
    if (_lodash.default.isFunction(this.storagePlugin.readTokens) === false) {
      return Promise.reject(_utils.ErrorCode.getCode(_constants.HTTP_STATUS.SERVICE_UNAVAILABLE, _constants.SUPPORT_ERRORS.PLUGIN_MISSING_INTERFACE));
    }

    return this.storagePlugin.readTokens(filter);
  }

}

var _default = LocalStorage;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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